/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import de.uni_paderborn.fujaba.metamodel.FDiagramItem;



/**
 * <h2>Associations</h2> <pre>
 *              0..1             N
 * UMLActivity -------------------- UMLTransition
 *              revEntry     entry
 *
 *              0..1             N
 * UMLActivity -------------------- UMLTransition
 *              revExit       exit
 * </pre>
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.72.2.3 $
 */
public class UMLTransition extends UMLConnection
{
   /**
    * @param coobraPersistent
    */
   public UMLTransition (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Constructor for class UMLTransition
    */
   public UMLTransition()
   {
      super();
   } // constructor


   /**
    * Constructor for class UMLTransition
    *
    * @param revExit   No description provided
    * @param revEntry  No description provided
    * @param guard     No description provided
    */
   public UMLTransition (UMLActivity revExit, UMLActivity revEntry,
                         UMLTransitionGuard guard)
   {
      super();
      setGuard (guard);
      setRevExit (revExit);
      setRevEntry (revEntry);
   } // constructor


   /**
    * <pre>
    *              0..1             N
    * UMLActivity -------------------- UMLTransition
    *              revEntry     entry
    * </pre>
    */
   private UMLActivity revEntry = null;


   /**
    * Sets the revEntry attribute of the UMLTransition object
    *
    * @param value  The new revEntry value
    * @return       No description provided
    */
   public boolean setRevEntry (UMLActivity value)
   {
      if ( (this.revEntry == null && value != null) ||
          (this.revEntry != null && !this.revEntry.equals (value)))
      {
         UMLActivity oldValue = this.revEntry;
         if (this.revEntry != null)
         {
            this.revEntry = null;
            oldValue.removeFromEntry (this);
         }
         this.revEntry = value;
         if (this.revEntry != null)
         {
            this.revEntry.addToEntry (this);
         }
         // side effects
         firePropertyChange ("revEntry", oldValue, value);

         return true;
      }

      return false;
   }


   /**
    * Get the revEntry attribute of the UMLTransition object
    *
    * @return   The revEntry value
    */
   public UMLActivity getRevEntry()
   {
      return this.revEntry;
   }


   /**
    * Sets the source attribute of the UMLTransition object
    *
    * @param source  The new source value
    * @return        No description provided
    */
   public boolean setSource (UMLActivity source)
   {
      return setRevExit (source);
   }


   /**
    * Get the source attribute of the UMLTransition object
    *
    * @return   The source value
    */
   public UMLActivity getSource()
   {
      return getRevExit();
   }


   /**
    * Sets the sourceConnector attribute of the UMLTransition object
    *
    * @param incr  The new sourceConnector value
    * @return      No description provided
    */
   public boolean setSourceConnector (FDiagramItem incr)
   {
      if (! (incr instanceof UMLActivity))
      {
         throw new IllegalArgumentException ("Argument is no UMLActivity");
      }
      return setRevEntry ((UMLActivity) incr);
   }


   /**
    * Get the sourceConnector attribute of the UMLTransition object
    *
    * @return   The sourceConnector value
    */
   public UMLDiagramItem getSourceConnector()
   {
      return getRevEntry();
   }


   /**
    * <pre>
    *              0..1             N
    * UMLActivity -------------------- UMLTransition
    *              revExit       exit
    * </pre>
    */
   private UMLActivity revExit = null;


   /**
    * Sets the revExit attribute of the UMLTransition object
    *
    * @param value  The new revExit value
    * @return       No description provided
    */
   public boolean setRevExit (UMLActivity value)
   {
      if ( (this.revExit == null && value != null) ||
          (this.revExit != null && !this.revExit.equals (value)))
      {
         UMLActivity oldValue = this.revExit;
         if (this.revExit != null)
         {
            this.revExit = null;
            oldValue.removeFromExit (this);
         }
         this.revExit = value;
         if (this.revExit != null)
         {
            this.revExit.addToExit (this);
         }

         // side effects
         firePropertyChange ("revExit", oldValue, value);

         return true;
      }

      return false;
   }


   /**
    * Get the revExit attribute of the UMLTransition object
    *
    * @return   The revExit value
    */
   public UMLActivity getRevExit()
   {
      return this.revExit;
   }


   /**
    * Sets the target attribute of the UMLTransition object
    *
    * @param target  The new target value
    * @return        No description provided
    */
   public boolean setTarget (UMLActivity target)
   {
      return setRevEntry (target);
   }


   /**
    * Get the target attribute of the UMLTransition object
    *
    * @return   The target value
    */
   public UMLActivity getTarget()
   {
      return getRevEntry();
   }


   /**
    * Sets the targetConnector attribute of the UMLTransition object
    *
    * @param incr  The new targetConnector value
    * @return      No description provided
    */
   public boolean setTargetConnector (FDiagramItem incr)
   {
      if (! (incr instanceof UMLActivity))
      {
         throw new IllegalArgumentException ("Argument is no UMLActivity");
      }
      return setRevExit ((UMLActivity) incr);
   }


   /**
    * Get the targetConnector attribute of the UMLTransition object
    *
    * @return   The targetConnector value
    */
   public UMLDiagramItem getTargetConnector()
   {
      return getRevExit();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String event = null;


   /**
    * Get the value of event.
    *
    * @return   Value of event.
    */
   public String getEvent()
   {
      return event;
   } // getEvent


   /**
    * Patch getName() to return a hopefully sensible name for the transition in AOAD contract
    * analysis
    *
    * @return   Content of event (whatever this is).
    */
   public String getName()
   {
      return event;
   }


   /**
    * Set the value of event.
    *
    * @param event  The new event value
    */
   public void setEvent (String event)
   {
      if ( (this.event == null && event != null) ||
          (this.event != null && !this.event.equals (event)))
      {
         String oldValue = this.event;
         this.event = event;
         firePropertyChange ("event", oldValue, this.event);
      }
   } // setEvent


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean hasEvent()
   {
      boolean result = false;

      if (event != null)
      {
         if (!event.equals (""))
         {
            result = true;
         }
      }

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String action = null;


   /**
    * Get the value of action.
    *
    * @return   Value of action.
    */
   public String getAction()
   {
      return action;
   } // getAction


   /**
    * Set the value of action.
    *
    * @param action  The new action value
    */
   public void setAction (String action)
   {
      if ( (this.action == null && action != null) ||
          (this.action != null && !this.action.equals (action)))
      {
         String oldValue = this.action;
         this.action = action;
         firePropertyChange ("action", oldValue, this.action);
      }
   } // setAction


   /**
    * Reverse UMLTransitionGuard revGuard.
    */
   private UMLTransitionGuard guard = null;


   /**
    * Get the guard attribute of the UMLTransition object
    *
    * @return   The guard value
    */
   public UMLTransitionGuard getGuard()
   {
      return guard;
   } // getGuard


   /**
    * Sets the guard attribute of the UMLTransition object
    *
    * @param guard  The new guard value
    */
   public void setGuard (UMLTransitionGuard guard)
   {

      if ( (this.guard == null && guard != null) ||
          (this.guard != null && !this.guard.equals (guard)))
      {
         // new partner
         UMLTransitionGuard oldGuard = this.guard;
         if (this.guard != null)
         {
            // revInform old partner
            this.guard = null;
            oldGuard.setRevGuard (null);
            oldGuard.removeYou();
         }
         this.guard = guard;
         if (this.guard != null)
         {
            // revInform new partner
            this.guard.setRevGuard (this);
         }
         firePropertyChange ("guard", oldGuard, this.guard);
      }
   } // setGuard


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient boolean isForwardEdge;


   /**
    * Sets the isForwardEdge attribute of the UMLTransition object
    *
    * @param isForwardEdge  The new isForwardEdge value
    */
   public void setIsForwardEdge (boolean isForwardEdge)
   {
      if (this.isForwardEdge != isForwardEdge)
      {
         boolean oldValue = this.isForwardEdge;
         this.isForwardEdge = isForwardEdge;
         firePropertyChange ("isForwardEdge", oldValue, isForwardEdge);
      }
   } // setIsForwardEdge


   /**
    * @return   TRUE - if Transition is a forward edge in the dfs tree FALSE - if Transition
    *      is a backward edge
    */
   public boolean getIsForwardEdge()
   {
      return isForwardEdge;
   } // getIsForwardEdge


   /**
    * Marker needed in Sequencer.explore()
    */
   private transient boolean isMarked = false;


   /**
    * Sets the isMarked attribute of the UMLTransition object
    *
    * @param isMarked  The new isMarked value
    */
   public void setIsMarked (boolean isMarked)
   {
      if (this.isMarked != isMarked)
      {
         this.isMarked = isMarked;
      }
   } //setIsMarked


   /**
    * Get the isMarked attribute of the UMLTransition object
    *
    * @return   The isMarked value
    */
   public boolean getIsMarked()
   {
      return isMarked;
   } // getIsMarked


   /**
    * Same functionality as equals() there are maybe problems with jgl sets work if this method
    * is called equals()
    *
    * @param curTransition  No description provided
    * @return               No description provided
    */
   public boolean looksLike (UMLTransition curTransition)
   {
      final UMLTransitionGuard myGuard = this.getGuard();
      final UMLTransitionGuard curGuard = curTransition.getGuard();
      return  (myGuard == curGuard) ||  (myGuard != null && myGuard.looksLike (curGuard));
   } // looksLike


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      UMLIncrement incr = getGuard();
      if (incr != null)
      {
         setGuard (null);
         incr.removeYou();
      }

      setRevEntry (null);
      setRevExit (null);

      super.removeYou();
   } // removeYou

}

/*
 * $Log: UMLTransition.java,v $
 * Revision 1.72.2.3  2006/06/07 09:13:32  creckord
 * - UMLTransitionGuard can be null instead of UMLTransitionGuard.NONE
 * - old "repair assoc" code removed (access methods are not kept, so repairing them is no longer needed)
 * - loop bends for assocs are removed when loop is removed
 *
 */
