/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml.actions;

import java.awt.event.ActionEvent;
import java.io.*;
import java.util.*;

import javax.swing.*;
import javax.swing.text.PlainDocument;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.basic.*;
import de.uni_paderborn.fujaba.codegen.CodeGenFactory;
import de.uni_paderborn.fujaba.parser.Parser;
import de.uni_paderborn.fujaba.parser.ParserManager;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.texteditor.TextEditor;
import de.uni_paderborn.fujaba.uml.*;


/**
 * Edit the java text of an increment.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.41.2.1 $
 */
public class EditTextAction extends AbstractAction implements TextEditor.BufferOwner, ProcessListener
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (EditTextAction.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLIncrement umlIncr;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String temporaryFileName = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    */
   public void actionPerformed (ActionEvent event)
   {
      temporaryFileName = null;

      Object source = event.getSource();
      if (source instanceof Iterator)
      {
         Iterator iter = (Iterator) source;
         if (iter.hasNext())
         {
            source = iter.next();
         }
      }
      else if (source instanceof String)
      {
         temporaryFileName = (String) source;
      }

      if (source instanceof UMLAttr)
      {
         source =  ((UMLAttr) source).getParent();
      }
      if (source instanceof UMLParam)
      {
         source =  ((UMLParam) source).getRevParam();
      }
      if (source instanceof UMLMethod)
      {
         source =  ((UMLMethod) source).getParent();
      }
      if (source instanceof UMLClass)
      {
         // ((UMLClass) source).checkFileAndPackageDependency();
         source =  ((UMLClass) source).getFile();
      }
      if (source instanceof UMLStoryPattern)
      {
         source =  ((UMLStoryPattern) source).getRevStoryPattern();
      }
      if (source instanceof UMLActivity)
      {
         source = (UMLDiagram)  ((UMLActivity) source).iteratorOfDiagrams().next();
      }
      if (source instanceof UMLActivityDiagram)
      {
         if ( ((UMLActivityDiagram) source).getStartActivity() != null)
         {
            UMLIncrement tmpUmlIncr =  ((UMLActivityDiagram) source).getStartActivity().getSpec();
            if (tmpUmlIncr == null)
            { // maybe its a start of statechart
               source =  ((UMLActivityDiagram) source).getStartActivity().getRevStartOfStateChart();
            }
            else
            {
               source = tmpUmlIncr;
            }
         }
      }

      UMLIncrement increment = null;
      if (source instanceof UMLIncrement)
      {
         increment = (UMLIncrement) source;
         umlIncr = increment;
      }

      if (increment != null || temporaryFileName != null)
      {
         try
         {
            if (GeneralPreferences.get().isUseExtEditor())
            {
               useExternalEditor (increment);
            }
            else
            {
               useTextEditor (increment);
            }
         }
         catch (Exception ex)
         {
            ex.printStackTrace();
            JOptionPane.showMessageDialog (FrameMain.get(), ex + "\n" +
               FujabaDebug.getStackTrace (ex, 0, 2),
               "Warning",
               JOptionPane.WARNING_MESSAGE);
         }
      }
   } // executeAction


   /**
    * This method uses the text editor to edit text.
    *
    * @param umlIncr  what to be edited (the source of this increment)
    * @return         No description provided
    */
   public TextEditor.Buffer useTextEditor (UMLIncrement umlIncr)
   {
      TextEditor editor = TextEditor.get();
      String text = null;

      StringTokenizer tokenizer;
      boolean lastWasDelimiter = true;

      if (umlIncr instanceof UMLCommentary)
      {
         text =  ((UMLCommentary) umlIncr).getText();
      }
      else if (umlIncr != null)
      {
         text = CodeGenFactory.get().generateFElement (umlIncr).toString();
      }
      else if (temporaryFileName != null)
      {
         text = openTextFile (temporaryFileName);
      }
      else
      {
         if (log.isInfoEnabled())
         {
            log.info ("nothing to do for this object???");
         }
      }

      if (text != null)
      {
         TextEditor.Buffer buffer;

         buffer = editor.createBuffer (this);

         //disable syntax highlight for UMLCommentary
         buffer.setSyntaxHighlightingEnabled (! (umlIncr instanceof UMLCommentary));

         tokenizer = new StringTokenizer (text, "\n", true);
         while (tokenizer.hasMoreElements())
         {
            String temp = (String) tokenizer.nextElement();
            if (temp.equals ("\n"))
            {
               if (lastWasDelimiter)
               {
                  buffer.addLine ("");
               }
               else
               {
                  lastWasDelimiter = true;
               }
            }
            else
            {
               buffer.addLine (temp);
               lastWasDelimiter = false;
            }
         }
         buffer.setDirty (false);

         buffer.setFilename ("Fujaba");

         return buffer;
      }
      else
      {
         return null;
      }
   }


   /**
    * This method uses an external editor to edit text.
    *
    * @param umlIncr  No description provided
    */
   private void useExternalEditor (UMLIncrement umlIncr)
   {
      ProcessOutputViewer process;

      GeneralPreferences options = GeneralPreferences.get();
      String temporaryPath = options.getTemporaryFolder();

      String text = null;

      if (umlIncr instanceof UMLCommentary)
      {
         text =  ((UMLCommentary) umlIncr).getText();
      }
      else if (umlIncr != null)
      {
         text = CodeGenFactory.get().generateFElement (umlIncr).toString();
      }
      else if (temporaryFileName != null)
      {
         text = openTextFile (temporaryFileName);
      }
      else
      {
         if (log.isInfoEnabled())
         {
            log.info ("nothing to do for this object???");
         }
      }

      File file;

      // temporaryFileName = temporaryPath + "fujaba_" + new Date().getTime() + ".java";

      // file = new File (temporaryFileName);
      file = new File (new File (temporaryPath), "fujaba_" + new Date().getTime() + ".java");

      temporaryFileName = file.getAbsolutePath();

      try
      {
         OutputStreamWriter out = new OutputStreamWriter (new FileOutputStream (file));
         out.write (text);
         out.close();

         process = new ProcessOutputViewer();
         // process.addProcessListener (this);
         // commented after frequent unwanted parsing of files. AZ
         process.executeCommand (options.getExternalEditor() + " " + temporaryFileName);

      }
      catch (IOException e)
      {
         if (log.isInfoEnabled())
         {
            log.info ("Could not write " + temporaryFileName);
         }
         if (log.isInfoEnabled())
         {
            log.info ("Error while executing external Editor");
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    */
   public void processFinished (ProcessEvent event)
   {
      try
      {
         BufferedReader in = new BufferedReader (new FileReader (temporaryFileName));
         PlainDocument doc = new PlainDocument();
         String temp;

         do
         {
            temp = in.readLine();
            if (temp != null)
            {
               doc.insertString (doc.getLength(), temp + "\n", null);
            }
         } while (temp != null);

         doParsing (umlIncr, doc.getText (0, doc.getLength()));

         new File (temporaryFileName).delete();
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param buffer  No description provided
    */
   public void bufferSaved (TextEditor.Buffer buffer)
   {
      doParsing (umlIncr, buffer.getText());
      UMLProject.get().refreshDisplay();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fileName  No description provided
    * @return          No description provided
    */
   public String openTextFile (String fileName)
   {
      String theFile = "";
      String currentLine;
      BufferedReader reader = null;
      try
      {
         reader = new BufferedReader (new FileReader (fileName));
      }
      catch (FileNotFoundException e)
      {
         if (log.isInfoEnabled())
         {
            log.info ("File " + temporaryFileName + " not found.");
         }
      }
      if (reader != null)
      {
         try
         {
            while ( (currentLine = reader.readLine()) != null)
            {
               theFile += currentLine + "\n";
            }
         }
         catch (IOException e)
         {
            e.printStackTrace();
         }
      }
      return theFile;
   } // openTextFile



   /**
    * parse code to put it into an increment
    *
    * @param incr  where to put the parsed information
    * @param text  to be parsed
    */
   public static void doParsing (UMLIncrement incr, String text)
   {
      if (incr instanceof UMLCommentary)
      {
          ((UMLCommentary) incr).setText (text);
      }
      else if (incr instanceof UMLFile)
      {
         Parser parser = ParserManager.get().getCurrentParser();

         UMLFile thisFile = (UMLFile) incr;

         if (parser != null)
         {
            parser.parseCompilationUnit (thisFile.getName(), new java.io.StringReader (text));
         }
         else
         {
            log.error ("EditTextAction has got no parser from the ParserManager!");
         }
      }
      else if (incr instanceof UMLMethod)
      {
         if (log.isInfoEnabled())
         {
            log.info ("please parse method by clicking \"Parse method\" in its context menu" + incr);
         }
      }
   }
}

/*
 * $Log: EditTextAction.java,v $
 * Revision 1.41.2.1  2005/09/30 18:51:11  mksoft
 * update temporaryFileName creation
 *
 */
