/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml.unparse;

import java.awt.Color;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JComponent;

import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.fsa.FSABendLine;
import de.uni_paderborn.fujaba.fsa.FSAEntryGrab;
import de.uni_paderborn.fujaba.fsa.FSAGrab;
import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.fsa.FSAPanel;
import de.uni_paderborn.fujaba.fsa.FSATextComponent;
import de.uni_paderborn.fujaba.fsa.listener.ArrowMoveListener;
import de.uni_paderborn.fujaba.fsa.listener.AscendDescendMouseHandler;
import de.uni_paderborn.fujaba.fsa.listener.BorderHighlighter;
import de.uni_paderborn.fujaba.fsa.listener.DoubleClickMouseListener;
import de.uni_paderborn.fujaba.fsa.listener.DragMouseListener;
import de.uni_paderborn.fujaba.fsa.listener.ForegroundHighlighter;
import de.uni_paderborn.fujaba.fsa.listener.GrabMouseListener;
import de.uni_paderborn.fujaba.fsa.listener.SelectionListenerHelper;
import de.uni_paderborn.fujaba.fsa.listener.SelectionMouseListener;
import de.uni_paderborn.fujaba.fsa.swing.DashedLineUI;
import de.uni_paderborn.fujaba.fsa.swing.JGrab;
import de.uni_paderborn.fujaba.fsa.swing.JLine;
import de.uni_paderborn.fujaba.fsa.swing.PanelGrabUI;
import de.uni_paderborn.fujaba.fsa.swing.border.CommentBorder;
import de.uni_paderborn.fujaba.fsa.unparse.AbstractUnparseModule;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.fsa.update.LogicFsaTranslator;
import de.uni_paderborn.fujaba.fsa.update.LogicToFsaUpdater;
import de.uni_paderborn.fujaba.fsa.update.Translator;
import de.uni_paderborn.fujaba.uml.UMLAttr;
import de.uni_paderborn.fujaba.uml.UMLCommentary;
import de.uni_paderborn.fujaba.uml.UMLMethod;
import de.upb.tools.pcs.CollectionChangeEvent;


/**
 * Unparse module for a UMLCommentary.
 *
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.10.2.2 $
 */
public class UMCommentary extends AbstractUnparseModule
{

   /**
    * Creates a FSAObject that consists of a FSAPanel that holds a FSATextComponent.
    * A FSABendLine that has two FSAGrabs connects the panel with the
    * UMLCommentary's parent.
    *
    *
    * @param parent  No description provided
    * @param incr    No description provided
    * @return        No description provided
    */
   public FSAObject create (FSAObject parent, LogicUnparseInterface incr)
   {
      UMLCommentary comment = (UMLCommentary) incr;

      // create a commentary-panel that has a line which is attached to its
      // target (revComment) which is of type UMLIncrement

      ////////////////////////////////////////////
      // filter those UML elements on which commentaries don't work at this
      // moment:

      // FIXME: allow commentaries to point at UMLMethod:
      // - line to a UMLMethod will point to 0/0
      // - updater on exitGrab doesn't seem to work for UMLMethod
      if (comment.getRevComment() instanceof UMLMethod)
      {
         return null;
      }

      // FIXME: allow commentaries to point at UMLAttr:
      // - line to a UMLAttr will create a new
      // non-moving-attribute-visualization that is placed at the current
      // attribute
      if (comment.getRevComment() instanceof UMLAttr)
      {
         return null;
      }
      ////////////////////////////////////////////

      FSAPanel commentPanel = new FSAPanel (incr, getMainFsaName(), parent.getJComponent(), false);
      //commentPanel.setTransientProperties(false);
      commentPanel.setBorder (new CommentBorder (Color.black));

      JComponent commentPanelJComponent = commentPanel.getJComponent();
      commentPanelJComponent.addKeyListener (ArrowMoveListener.get());
      //AscendDescendMouseHandler.addMouseInputListener
      //(commentPanelJComponent, ComponentCursorListener.get());
      AscendDescendMouseHandler.addMouseInputListener (commentPanelJComponent, SelectionMouseListener.get());
      AscendDescendMouseHandler.addMouseInputListener (commentPanelJComponent, DragMouseListener.get());
      AscendDescendMouseHandler.addMouseInputListener (commentPanelJComponent, DoubleClickMouseListener.get());
      SelectionListenerHelper.addSelectionListener (commentPanelJComponent, BorderHighlighter.get());
      SelectionListenerHelper.addSelectionListener (commentPanelJComponent, ArrowMoveListener.get());

      // the comment-text.
      FSATextComponent nameLabel = new FSATextComponent (incr, "text", commentPanelJComponent);
      nameLabel.addToUpdater (nameLabel.createDefaultUpdater());

      FSABendLine line = new FSABendLine (incr, getMainFsaName() + ".line", parent.getJComponent());
      JComponent jPanel = line.getJComponent();
       ((JLine) jPanel).setUI (DashedLineUI.createUI (jPanel));

      // add mouse listener for selecting
      AscendDescendMouseHandler.addMouseInputListener (jPanel, SelectionMouseListener.get());
      SelectionListenerHelper.addSelectionListener (jPanel, ForegroundHighlighter.get());

      FSAGrab startGrab = new FSAGrab (incr, "startGrab", parent.getJComponent());
      PanelGrabUI ui = (PanelGrabUI) PanelGrabUI.createUI (startGrab.getJComponent());
      startGrab.setUI (ui);
      startGrab.setTarget (commentPanel);
      startGrab.getJComponent().setOpaque (false);
      line.setStartBend (startGrab);

      FSAGrab exitGrab = new FSAEntryGrab (incr, "exitGrab", parent.getJComponent());
      LogicToFsaUpdater updater = new LogicToFsaUpdater();
      Translator translator = new LogicFsaTranslator (line, 1);
      updater.setTranslator (translator);
      updater.setLogicAttrName ("revComment");
      updater.setFsaAttrName ("target");
      updater.setLogicObject (comment);
      exitGrab.addToUpdater (updater);
      line.setEndBend (exitGrab);
      JGrab jExitGrab = (JGrab) exitGrab.getJComponent();
      SelectionListenerHelper.addSelectionListener (jExitGrab, ForegroundHighlighter.get());
      AscendDescendMouseHandler.addMouseInputListener (jExitGrab, GrabMouseListener.get());

      // add property change listener that handles removal of
      // the commentary-logic from the diagram
      new UMCommentaryPropertyChangeListener (comment, (ASGDiagram) FSAObject.getLogicFromJComponent (parent.getJComponent()), line);

      return commentPanel;
   }

   // ------------- inner classes --------------------------------------------
   /**
    * Class that reacts on changes of UMLCommentary. Its main purpose is to
    * remove line and grabs created by UMCommentary, if the UMLCommentary is
    * removed from a diagram.
    *
    *
    * @author    $Author: fklar $
    * @version   $Revision: 1.10.2.2 $
    */
   private class UMCommentaryPropertyChangeListener implements PropertyChangeListener
   {
      /**
       * The commentary-logic that has been added to a diagram.
       */
      private UMLCommentary commentary;

      /**
       * The diagram the commentary-logic has been added to.
       */
      private ASGDiagram diagram;

      /**
       * A line containing two grabs, that binds the commentary to an
       * UMLIncrement.
       */
      private FSABendLine fsaBendLine;


      /**
       * Constructor for class UMCommentaryPropertyChangeListener
       *
       *
       * @param commentary   Commentary logic.
       * @param diagram      Diagram the commentary is located.
       * @param fsaBendLine  Line that connects commentary with an UMLIncrement.
       */
      public UMCommentaryPropertyChangeListener (UMLCommentary commentary, ASGDiagram diagram, FSABendLine fsaBendLine)
      {
         this.commentary = commentary;
         this.diagram = diagram;
         this.fsaBendLine = fsaBendLine;

         diagram.addPropertyChangeListener (this);
      }


      /**
       * Reacts on changes of an ASGDiagram's "element"-collection (CollectionChangeEvent).
       * Removes the line, if UMLCommentary has been removed from the collection.
       *
       *
       * @param event  A property change event.
       */
      public void propertyChange (PropertyChangeEvent event)
      {
         String propertyName = event.getPropertyName();

         if (event instanceof CollectionChangeEvent)
         {
            CollectionChangeEvent collChangeEvent = (CollectionChangeEvent) event;

            if ("elements".equals (propertyName) && collChangeEvent.getType() == CollectionChangeEvent.REMOVED)
            {
               // delete both grabs and the line,
               // if the commentary has been removed from the diagram
               if (event.getOldValue() == this.commentary)
               {
                  diagram.removePropertyChangeListener (this);

                  fsaBendLine.removeYou();
               }
            }
         }
      }
   }
}

/*
 * $Log: UMCommentary.java,v $
 * Revision 1.10.2.2  2005/07/04 16:42:00  fklar
 * + visibility of commentary in a diagram may be changed independent from visibility in other diagrams
 * + allow commentary to be visualized for every UMLIncrement (not only UMLClass)
 * + TODO: commentaries can't be visualized for UMLMethod and UMLAttr at this moment
 *
 */
