/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views.cf;

import java.util.*;

import de.uni_paderborn.fujaba.uml.UMLDiagram;
import de.uni_paderborn.fujaba.uml.UMLDiagramItem;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashSet;


/**
 * Base class for all nodes in a CFDiagram
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.12 $
 */
public abstract class CFNode extends UMLDiagramItem implements Cloneable
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return                             No description provided
    * @throws CloneNotSupportedException  Exception description not provided
    */
   public Object clone() throws CloneNotSupportedException
   {
      return super.clone();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Set result = null;


   /**
    * Get the results attribute of the CFNode object
    *
    * @param diag  No description provided
    * @param item  No description provided
    * @return      The results value
    */
   public Set getResults (UMLDiagram diag, UMLDiagramItem item)
   {
      if (result != null)
      {
         return result;
      }

      FHashSet uncomplete = new FHashSet();
      return getResults (diag, item, uncomplete);
   }


   /**
    * Get the results attribute of the CFNode object
    *
    * @param diag        No description provided
    * @param item        No description provided
    * @param uncomplete  No description provided
    * @return            The results value
    */
   protected Set getResults (UMLDiagram diag, UMLDiagramItem item, Set uncomplete)
   {
      if (result == null)
      {
         uncomplete.add (this);

         Vector sourceResults = new Vector();

         Iterator entryIter = iteratorOfEntries();
         while (entryIter.hasNext())
         {
            CFTransition trans = (CFTransition) entryIter.next();
            CFNode node = trans.getSource();

            if (uncomplete.contains (node))
            {
               throw new RuntimeException ("Loop detected: " + this + " <-> " + node);
            }

            sourceResults.add (node.getResults (diag, item, uncomplete));
         }
         result = computeResult (diag, item, sourceResults);
         uncomplete.remove (this);
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void resetResults()
   {
      if (result != null)
      {
         result.clear();
         result = null;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void clearResults()
   {
      if (result != null)
      {
         resetResults();

         Iterator entryIter = iteratorOfEntries();
         while (entryIter.hasNext())
         {
            CFTransition trans = (CFTransition) entryIter.next();
            CFNode node = trans.getSource();

            node.clearResults();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag           No description provided
    * @param item           No description provided
    * @param sourceResults  No description provided
    * @return               No description provided
    */
   protected abstract Set computeResult (UMLDiagram diag, UMLDiagramItem item, Collection sourceResults);


   /**
    * <pre>
    *          0..1                            N
    * CFNode ----------------------------- CFTransition
    *          target     entries
    * </pre>
    */
   private FHashSet entries;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToEntries (CFTransition value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.entries == null)
         {
            this.entries = new FHashSet();
         }
         changed = this.entries.add (value);
         if (changed)
         {
            value.setTarget (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInEntries (CFTransition value)
   {
      return  ( (this.entries != null) &&
          (value != null) &&
         this.entries.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfEntries()
   {
      return  ( (this.entries == null)
         ? FEmptyIterator.get()
         : this.entries.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfEntries()
   {
      return  ( (this.entries == null)
         ? 0
         : this.entries.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromEntries (CFTransition value)
   {
      boolean changed = false;

      if ( (this.entries != null) &&  (value != null))
      {
         changed = this.entries.remove (value);
         if (changed)
         {
            value.setTarget (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromEntries()
   {
      CFTransition tmpValue;
      Iterator iter = this.iteratorOfEntries();

      while (iter.hasNext())
      {
         tmpValue = (CFTransition) iter.next();
         this.removeFromEntries (tmpValue);
      }
   }


   /**
    * <pre>
    *          0..1                            N
    * CFNode ----------------------------- CFTransition
    *          source     exits
    * </pre>
    */
   private FHashSet exits;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToExits (CFTransition value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.exits == null)
         {
            this.exits = new FHashSet();
         }
         changed = this.exits.add (value);
         if (changed)
         {
            value.setSource (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInExits (CFTransition value)
   {
      return  ( (this.exits != null) &&
          (value != null) &&
         this.exits.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfExits()
   {
      return  ( (this.exits == null)
         ? FEmptyIterator.get()
         : this.exits.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfExits()
   {
      return  ( (this.exits == null)
         ? 0
         : this.exits.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromExits (CFTransition value)
   {
      boolean changed = false;

      if ( (this.exits != null) &&  (value != null))
      {
         changed = this.exits.remove (value);
         if (changed)
         {
            value.setSource (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromExits()
   {
      CFTransition tmpValue;
      Iterator iter = this.iteratorOfExits();

      while (iter.hasNext())
      {
         tmpValue = (CFTransition) iter.next();
         this.removeFromExits (tmpValue);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      if (result != null)
      {
         result.clear();
         result = null;
      }
      Iterator transitionIter = iteratorOfEntries();
      while (transitionIter.hasNext())
      {
         CFTransition trans = (CFTransition) transitionIter.next();
         trans.removeYou();
      }
      transitionIter = iteratorOfExits();
      while (transitionIter.hasNext())
      {
         CFTransition trans = (CFTransition) transitionIter.next();
         trans.removeYou();
      }
      super.removeYou();
   }
}

/*
 * $Log: CFNode.java,v $
 * Revision 1.12  2004/10/20 17:50:32  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
