/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.lib.classloader;

import java.io.*;
import java.net.*;
import java.util.*;

import de.upb.tools.fca.FEmptyIterator;


/**
 * Load classes from other places than those specified in the java class path.
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.43 $
 */
public class UPBClassLoader extends ClassLoader
{
   /**
    * The default application class loader.
    */
   public final static String DEFAULT_CLASSLOADER = "fujaba.core";

   /**
    * No comment provided by developer, please add a comment to ensure improve documentation.
    */
   private String id;

   /**
    * stores all classes the class loader is responsible for.
    */
   private HashMap classtable;

   /**
    * stores all files of the classes the class loader is responsible for.
    */
   private LinkedList fileset;

   /**
    * stores additional resource paths where resources are searched after nothing was found
    * on the classpath (denoted by fileset).
    */
   private LinkedList resourcePaths;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient HashMap pathUrls;

   /**
    * The classLoaderTable stores all class loaders with its id's as key.
    */
   private static HashMap classLoaderTable;

   /**
    * The classLoaderChain list contains class loaders for a responsibility chain. If a plug-in
    * A depends on plug-in B and plug-in C then the class loader for plug-in A contains the
    * class loaders for B and C in this list.
    */
   private LinkedList classLoaderChain;

   /**
    * The application class loader is used as parent class loader for all UPBClassLoader. The
    * default value is the class loader that loaded the class UPBClassLoader.
    */
   private static ClassLoader applicationClassLoader = UPBClassLoader.class.getClassLoader();


   /**
    * This method sets the application class loader used as parent class loader for all UPBClassLoaders.
    * This method can only be called once before an instance of UPBClassLoader is created,
    * i.e. before (@link #get()) or (@link #get(String)) was called.
    *
    * @param classLoader             the application class loader
    * @throws IllegalStateException  when an instance of UPBClassLoader is already created.
    */
   public static void setApplicationClassLoader (ClassLoader classLoader)
   {
      if (classLoaderTable == null)
      {
         applicationClassLoader = classLoader;
      }
      else
      {
         throw new IllegalStateException ("Changing the application class loader when an instance\n"
            + "of UPBClassLoader was already created is not supported!");
      }
   }


   /**
    * The constructor is private, use (@link #get(String)) to get an instance.
    *
    * @see   #get()
    * @see   #get(String)
    */
   private UPBClassLoader()
   {
      super (applicationClassLoader);
      classtable = new HashMap();
      fileset = new LinkedList();
   }


   /**
    * @param loader  the parent class loader
    * @see           #get(String)
    * @deprecated    This public constructor will be removed soon. Use (@link #get(String))
    *      to get an instance.
    */
   public UPBClassLoader (ClassLoader loader)
   {
      super (loader);
      classtable = new HashMap();
      fileset = new LinkedList();
   }


   /**
    * This method returns the default application class loader. The method call UPBClassLoader.get
    * (DEFAULT_CLASSLOADER) returns the same class loader.
    *
    * @return   the default class loader
    * @see      #get(String)
    */
   public static UPBClassLoader get()
   {
      return get (DEFAULT_CLASSLOADER);
   }


   /**
    * Returns a class loader identified by the unique string id.
    *
    * @param id  to identify the class loader the default class loader has the id DEFAULT_CLASSLOADER.
    *      The class loader for plug-ins are identified by the pluginClass attribute in plugin.xml
    *      document.
    * @return    the insance of an class loader identified by the id
    */
   public static synchronized UPBClassLoader get (String id)
   {
      if (classLoaderTable == null)
      {
         classLoaderTable = new HashMap();
      }

      UPBClassLoader loader = (UPBClassLoader) classLoaderTable.get (id);
      if (loader == null)
      {
         loader = new UPBClassLoader();
         if (id != null)
         {
            classLoaderTable.put (id, loader);
            loader.id = id;
         }
         else
         {
            throw new IllegalArgumentException ("The id must not be null!");
         }
      }

      return loader;
   }


   /**
    * adds a class loader to a chain of responsible class loaders
    *
    * @param classLoader  class loader to be added
    */
   public void addToClassLoaderChain (UPBClassLoader classLoader)
   {
      if (classLoaderChain == null)
      {
         classLoaderChain = new LinkedList();
      }
      if ( (classLoader != null) &&  (classLoader != this))
      {
         classLoaderChain.add (classLoader);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfClassLoaderChain()
   {
      return classLoaderChain == null ? FEmptyIterator.get() : classLoaderChain.iterator();
   }


   /**
    * internal method to uncompress the byte code of a zip entry.
    *
    * @param url           No description provided
    * @return              the uncompressed byte code
    * @throws IOException  when an IO error occurs
    */
   private byte[] read (URL url) throws IOException
   {
      URLConnection connection = url.openConnection();
      connection.connect();

      int length = connection.getContentLength();
      byte[] uncompressed = new byte[length];

      InputStream input = connection.getInputStream();
      int begin = 0;

      while (begin < uncompressed.length)
      {
         begin = begin + input.read (uncompressed, begin, uncompressed.length - begin);
      }

      return uncompressed;
   }


   /**
    * internal method to find the byte code of a class specified by the name of the class.
    *
    * @param className  the specified class name
    * @return           the byte code encapsulated in an UPBByteCode instance
    */
   private UPBByteCode searchClass (String className)
   {
      UPBByteCode tmpClass = getFromClasses (className);

      if (tmpClass == null)
      {
         String tmpName = className;
         String sysProp = "/";

         int index = tmpName.indexOf (".");

         if (index >= 0)
         {
            StringBuffer buffer = new StringBuffer (tmpName);
            while (index >= 0)
            {
               buffer = buffer.replace (index, index + 1, sysProp);

               tmpName = buffer.toString();
               index = tmpName.indexOf (".");
            }
         } // end of if ()
         tmpName = tmpName.concat (".class");

         Iterator iter = fileset.iterator();

         while ( (tmpClass == null) &&  (iter.hasNext()))
         {
            String tmpPath;

            tmpPath = (String) iter.next();

            URL url = getURL (tmpPath, tmpName);

            if (url != null)
            {
               try
               {
                  byte result[] = read (url);
                  tmpClass = new UPBByteCode();
                  tmpClass.setClassName (className);
                  tmpClass.setCode (result);
               }
               catch (IOException io)
               {
                  tmpClass = null;
                  System.err.println ("IO-error occured during reading file...");
                  System.err.println ("\tPath:" + tmpPath);
                  System.err.println ("\tClass:" + tmpName);
                  io.printStackTrace();
               }
            }
         }
         if (tmpClass != null)
         {
            addToClasses (tmpClass);
         }
      }
      return tmpClass;
   }


   /**
    * adds a path to the path list which is controlled by this class loader.
    *
    * @param path  to be added to the path list
    * @see         #removeClassPath(String)
    */
   public synchronized void addClassPath (String path)
   {
      if (path != null && !path.equals (""))
      {
         String pathDelimiter = System.getProperty ("path.separator");
         StringTokenizer tokenizer = new StringTokenizer (path, pathDelimiter);

         while (tokenizer.hasMoreTokens())
         {
            String pathElem = tokenizer.nextToken();
            addClassLocation (pathElem);
         }
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param index  The object added.
    * @param path   The object added.
    */
   public synchronized void addClassLocation (int index, String path)
   {
      if (path != null && !path.equals ("") && !fileset.contains (path))
      {
         fileset.add (index, path);
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param path  The object added.
    */
   public synchronized void addClassLocation (String path)
   {
      if (path != null && !path.equals ("") && !fileset.contains (path))
      {
         fileset.add (path);
      }
   }


   /**
    * remove a path from the path list.
    *
    * @param path  the path
    * @see         #addClassPath
    */
   public synchronized void removeClassPath (String path)
   {
      if (path != null && fileset.contains (path))
      {
         fileset.remove (path);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param index  No description provided
    * @return       No description provided
    */
   public synchronized String removeClassPath (int index)
   {
      return (String) fileset.remove (index);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param path  No description provided
    * @return      No description provided
    */
   public int indexOfClassPath (String path)
   {
      return fileset.indexOf (path);
   }


   /**
    * adds a path to the path list which is controlled by this class loader.
    *
    * @return   The classPath value
    * @see      #removeClassPath(String)
    * @see      #addClassPath
    */
   public String getClassPath()
   {
      String classPath = "";
      String pathDelimiter = System.getProperty ("path.separator");
      Iterator iter = fileset.iterator();
      while (iter.hasNext())
      {
         String pathElem = (String) iter.next();
         classPath += pathElem;
         if (iter.hasNext())
         {
            classPath += pathDelimiter;
         }
      }
      iter = iteratorOfClassLoaderChain();
      while (iter.hasNext())
      {
         UPBClassLoader chainElem = (UPBClassLoader) iter.next();
         String parentClassPath = chainElem.getClassPath();
         if (parentClassPath.length() > 0)
         {
            if (classPath.length() > 0)
            {
               classPath += pathDelimiter;
            }
            classPath += parentClassPath;
         }
      }
      if ( (classLoaderChain == null) ||  (classLoaderChain.size() == 0))
      {
         // Systemclassloader
         String parentClassPath = System.getProperty ("java.class.path");
         if (parentClassPath.length() > 0)
         {
            if (classPath.length() > 0)
            {
               classPath += pathDelimiter;
            }
            classPath += parentClassPath;
         }
      }
      return classPath;
   }


   /**
    * adds a path to the resource path list which is controlled by this class loader.
    *
    * @param path  to be added to the path list
    * @see         #removeResourcePath
    * @see         #addClassPath
    * @see         #removeClassPath(String)
    */
   public synchronized void addResourcePath (String path)
   {
      if (path != null && !path.equals (""))
      {
         String pathDelimiter = System.getProperty ("path.separator");
         StringTokenizer tokenizer = new StringTokenizer (path, pathDelimiter);

         while (tokenizer.hasMoreTokens())
         {
            String pathElem = tokenizer.nextToken();
            addResourceLocation (pathElem);
         }
      }
   }


   /**
    * adds a path to the resource path list which is controlled by this class loader.
    *
    * @param path  to be added to the path list
    * @see         #removeResourcePath
    * @see         #addClassPath
    * @see         #removeClassPath(String)
    */
   public synchronized void addResourceLocation (String path)
   {
      if (resourcePaths == null)
      {
         resourcePaths = new LinkedList();
      }
      if (!resourcePaths.contains (path))
      {
         resourcePaths.add (path);
      }
   }


   /**
    * remove a path from the resource path list.
    *
    * @param path  the path
    * @see         #addResourcePath
    * @see         #addClassPath
    * @see         #removeClassPath(String)
    */
   public synchronized void removeResourcePath (String path)
   {
      if (resourcePaths != null && path != null && resourcePaths.contains (path))
      {
         resourcePaths.remove (path);
      }
   }


   /**
    * overrides loadClass of ClassLoader. At first the class loader tries to load the class
    * with the system class loader. If this fails the class loader tries to load the class.
    *
    * @param className                the full qualified class name of the class
    * @return                         the class instance when found in class path
    * @throws ClassNotFoundException  when class not found
    * @see                            ClassLoader#loadClass(String)
    */
   public synchronized Class loadClass (String className) throws ClassNotFoundException
   {
      return loadClass (className, false);
   }


   /**
    * overrides loadClass of ClassLoader.
    *
    * @param className                the full qualified class name of the class
    * @param resolveIt                resolve the class
    * @return                         the class instance when found in class path
    * @throws ClassNotFoundException  when class not found
    * @see                            ClassLoader#loadClass(String)
    */
   public synchronized Class loadClass (String className, boolean resolveIt)
       throws ClassNotFoundException
   {
      // Check our local cache of classes
      Class result = (Class) classtable.get (className);
      if (result != null)
      {
         return result;
      }

      // search in the class loader responsibility chain,
      // if the class can not be found then search for it
      // in the default class loader
      if (classLoaderChain != null)
      {
         Iterator iter = iteratorOfClassLoaderChain();
         while (iter.hasNext())
         {
            UPBClassLoader chainElem = (UPBClassLoader) iter.next();
            try
            {
               result = chainElem.loadClass (className, resolveIt);
               return result;
            }
            catch (ClassNotFoundException e)
            {
            }
         }
      }

      // The method super.loadClass tries to load the class by the
      // parent class loader first. If that fails it calls findClass().
      result = super.loadClass (className, resolveIt);

      // cache the class
      classtable.put (className, result);

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param className                No description provided
    * @return                         No description provided
    * @throws ClassNotFoundException  Exception description not provided
    */
   protected Class findClass (String className) throws ClassNotFoundException
   {
      // Check our local cache of classes
      UPBByteCode code = searchClass (className);
      if (code == null)
      {
         throw new ClassNotFoundException (className);
      }
      byte[] classData = code.getCode();

      Class result = defineClass (className, classData, 0, classData.length);

      return result;
   }


   /**
    * role of the classes assoziation between UPBClassLoader and UPBByteCode. <pre>
    *
    *
    *                  ------------- 0..1   classes   0..1
    *   UPBClassLoader | className |----------------------- UPBByteCode
    *                  ------------- loader        classes
    *
    *
    * </pre>
    *
    * @see   UPBClassLoader#hasInClasses
    * @see   UPBClassLoader#hasKeyInClasses
    * @see   UPBClassLoader#iteratorOfClasses
    * @see   UPBClassLoader#keysOfClasses
    * @see   UPBClassLoader#entriesOfClasses
    * @see   UPBClassLoader#sizeOfClasses
    * @see   UPBClassLoader#getFromClasses
    * @see   UPBClassLoader#addToClasses
    * @see   UPBClassLoader#removeFromClasses
    * @see   UPBClassLoader#removeKeyFromClasses
    * @see   UPBClassLoader#removeAllFromClasses
    */
   private HashMap classes;


   /**
    * access method to get the information if a specific class exists in the set of classes.
    *
    * @param value  the class as byte code
    * @return       true when the given class is found in the set of all classes.
    * @see          UPBClassLoader#hasKeyInClasses
    * @see          UPBClassLoader#iteratorOfClasses
    * @see          UPBClassLoader#keysOfClasses
    * @see          UPBClassLoader#entriesOfClasses
    * @see          UPBClassLoader#sizeOfClasses
    * @see          UPBClassLoader#getFromClasses
    * @see          UPBClassLoader#addToClasses
    * @see          UPBClassLoader#removeFromClasses
    * @see          UPBClassLoader#removeKeyFromClasses
    * @see          UPBClassLoader#removeAllFromClasses
    */
   public boolean hasInClasses (UPBByteCode value)
   {
      return  ( (this.classes != null) &&  (value != null) &&  (value.getClassName() != null) &&  (this.classes
         .get (value.getClassName()) == value));
   }


   /**
    * access method to get the information if a specific key exists in the set of classes.
    *
    * @param key  the class name searched for
    * @return     true when the given class name is found in the set of all classes.
    * @see        UPBClassLoader#hasInClasses
    * @see        UPBClassLoader#iteratorOfClasses
    * @see        UPBClassLoader#keysOfClasses
    * @see        UPBClassLoader#entriesOfClasses
    * @see        UPBClassLoader#sizeOfClasses
    * @see        UPBClassLoader#getFromClasses
    * @see        UPBClassLoader#addToClasses
    * @see        UPBClassLoader#removeFromClasses
    * @see        UPBClassLoader#removeKeyFromClasses
    * @see        UPBClassLoader#removeAllFromClasses
    */
   public boolean hasKeyInClasses (String key)
   {
      return  ( (this.classes != null) &&  (key != null) && this.classes.containsKey (key));
   }


   /**
    * access method to get an Iterator over all classes stored in the set of classes.
    *
    * @return   an iterator over all elements of the set of all classes, may be null!
    * @see      UPBClassLoader#hasInClasses
    * @see      UPBClassLoader#hasKeyInClasses
    * @see      UPBClassLoader#keysOfClasses
    * @see      UPBClassLoader#entriesOfClasses
    * @see      UPBClassLoader#sizeOfClasses
    * @see      UPBClassLoader#getFromClasses
    * @see      UPBClassLoader#addToClasses
    * @see      UPBClassLoader#removeFromClasses
    * @see      UPBClassLoader#removeKeyFromClasses
    * @see      UPBClassLoader#removeAllFromClasses
    */
   protected Iterator iteratorOfClasses()
   {
      return  ( (this.classes == null) ? null : this.classes.values().iterator());
   }


   /**
    * access method to get an Iterator over all keys stored in the set of classes.
    *
    * @return   an iterator over all keys of the set of all classes, may be null!
    * @see      UPBClassLoader#hasInClasses
    * @see      UPBClassLoader#hasKeyInClasses
    * @see      UPBClassLoader#iteratorOfClasses
    * @see      UPBClassLoader#entriesOfClasses
    * @see      UPBClassLoader#sizeOfClasses
    * @see      UPBClassLoader#getFromClasses
    * @see      UPBClassLoader#addToClasses
    * @see      UPBClassLoader#removeFromClasses
    * @see      UPBClassLoader#removeKeyFromClasses
    * @see      UPBClassLoader#removeAllFromClasses
    */
   protected Iterator keysOfClasses()
   {
      return  ( (this.classes == null) ? null : this.classes.keySet().iterator());
   }


   /**
    * access method to get an Iterator over all classes stored in the set of classes.
    *
    * @return   an iterator over all elements of the set of all classes, may be null!
    * @see      UPBClassLoader#hasInClasses
    * @see      UPBClassLoader#hasKeyInClasses
    * @see      UPBClassLoader#iteratorOfClasses
    * @see      UPBClassLoader#keysOfClasses
    * @see      UPBClassLoader#sizeOfClasses
    * @see      UPBClassLoader#getFromClasses
    * @see      UPBClassLoader#addToClasses
    * @see      UPBClassLoader#removeFromClasses
    * @see      UPBClassLoader#removeKeyFromClasses
    * @see      UPBClassLoader#removeAllFromClasses
    */
   protected Iterator entriesOfClasses()
   {
      return  ( (this.classes == null) ? null : this.classes.entrySet().iterator());
   }


   /**
    * access method to get the count of classes stored in the set of classes.
    *
    * @return   the size of the set of classes.
    * @see      UPBClassLoader#hasInClasses
    * @see      UPBClassLoader#hasKeyInClasses
    * @see      UPBClassLoader#iteratorOfClasses
    * @see      UPBClassLoader#keysOfClasses
    * @see      UPBClassLoader#entriesOfClasses
    * @see      UPBClassLoader#getFromClasses
    * @see      UPBClassLoader#addToClasses
    * @see      UPBClassLoader#removeFromClasses
    * @see      UPBClassLoader#removeKeyFromClasses
    * @see      UPBClassLoader#removeAllFromClasses
    */
   public int sizeOfClasses()
   {
      return  ( (this.classes == null) ? 0 : this.classes.size());
   }


   /**
    * access method to get the byte code of a key specified class.
    *
    * @param key  the full qualified name of the class.
    * @return     the class specified by key.
    * @see        UPBClassLoader#hasInClasses
    * @see        UPBClassLoader#hasKeyInClasses
    * @see        UPBClassLoader#iteratorOfClasses
    * @see        UPBClassLoader#keysOfClasses
    * @see        UPBClassLoader#entriesOfClasses
    * @see        UPBClassLoader#sizeOfClasses
    * @see        UPBClassLoader#addToClasses
    * @see        UPBClassLoader#removeFromClasses
    * @see        UPBClassLoader#removeKeyFromClasses
    * @see        UPBClassLoader#removeAllFromClasses
    */
   public UPBByteCode getFromClasses (String key)
   {
      return  ( ( (this.classes == null) ||  (key == null)) ? null : (UPBByteCode) this.classes.get (key));
   }


   /**
    * access method to add a class to the set of classes.
    *
    * @param value  class as byte code.
    * @return       true when the class was added to the set
    * @see          UPBClassLoader#hasInClasses
    * @see          UPBClassLoader#hasKeyInClasses
    * @see          UPBClassLoader#iteratorOfClasses
    * @see          UPBClassLoader#keysOfClasses
    * @see          UPBClassLoader#entriesOfClasses
    * @see          UPBClassLoader#sizeOfClasses
    * @see          UPBClassLoader#getFromClasses
    * @see          UPBClassLoader#removeFromClasses
    * @see          UPBClassLoader#removeKeyFromClasses
    * @see          UPBClassLoader#removeAllFromClasses
    */
   public boolean addToClasses (UPBByteCode value)
   {
      boolean changed = false;
      if ( (value != null) &&  (value.getClassName() != null))
      {
         if (this.classes == null)
         {
            this.classes = new HashMap();
         }

         UPBByteCode oldValue = (UPBByteCode) this.classes.put (value.getClassName(), value);

         if (oldValue != value)
         {
            if (oldValue != null)
            {
               oldValue.setLoader (null);
            }
            value.setLoader (this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * access method to remove a class from the set of classes.
    *
    * @param value  class as byte code.
    * @return       true when the class was removed from the set
    * @see          UPBClassLoader#hasInClasses
    * @see          UPBClassLoader#hasKeyInClasses
    * @see          UPBClassLoader#iteratorOfClasses
    * @see          UPBClassLoader#keysOfClasses
    * @see          UPBClassLoader#entriesOfClasses
    * @see          UPBClassLoader#sizeOfClasses
    * @see          UPBClassLoader#getFromClasses
    * @see          UPBClassLoader#addToClasses
    * @see          UPBClassLoader#removeKeyFromClasses
    * @see          UPBClassLoader#removeAllFromClasses
    */
   public boolean removeFromClasses (UPBByteCode value)
   {
      boolean changed = false;
      if ( (this.classes != null) &&  (value != null) &&  (value.getClassName() != null))
      {
         UPBByteCode oldValue = (UPBByteCode) this.classes.get (value.getClassName());

         if (oldValue == value)
         {
            this.classes.remove (value.getClassName());
            value.setLoader (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * access method to remove the key specified class from the set of classes.
    *
    * @param key  the class name of the class.
    * @return     true when the class was removed from the set
    * @see        UPBClassLoader#hasInClasses
    * @see        UPBClassLoader#hasKeyInClasses
    * @see        UPBClassLoader#iteratorOfClasses
    * @see        UPBClassLoader#keysOfClasses
    * @see        UPBClassLoader#entriesOfClasses
    * @see        UPBClassLoader#sizeOfClasses
    * @see        UPBClassLoader#getFromClasses
    * @see        UPBClassLoader#addToClasses
    * @see        UPBClassLoader#removeFromClasses
    * @see        UPBClassLoader#removeAllFromClasses
    */
   public boolean removeKeyFromClasses (String key)
   {
      boolean changed = false;
      if ( (this.classes != null) &&  (key != null))
      {
         UPBByteCode tmpValue = (UPBByteCode) this.classes.get (key);
         if (tmpValue != null)
         {
            this.classes.remove (key);
            tmpValue.setLoader (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * access method to remove all classes from the set of classes.
    *
    * @see   UPBClassLoader#hasInClasses
    * @see   UPBClassLoader#hasKeyInClasses
    * @see   UPBClassLoader#iteratorOfClasses
    * @see   UPBClassLoader#keysOfClasses
    * @see   UPBClassLoader#entriesOfClasses
    * @see   UPBClassLoader#sizeOfClasses
    * @see   UPBClassLoader#getFromClasses
    * @see   UPBClassLoader#addToClasses
    * @see   UPBClassLoader#removeFromClasses
    * @see   UPBClassLoader#removeKeyFromClasses
    */
   public void removeAllFromClasses()
   {
      UPBByteCode tmpValue;
      Iterator iter = this.iteratorOfClasses();
      while (iter != null && iter.hasNext())
      {
         tmpValue = (UPBByteCode) iter.next();
         this.removeFromClasses (tmpValue);
      }
   }


   /**
    * searches the classpath and the jars for a given resource. Needed to load images etc.
    * from the plugins. <p>
    *
    * If no matching resource was found on the classpath the resource path is searched.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   protected URL findResource (String name)
   {
      URL url = super.findResource (name);
      if (url == null)
      {
         Iterator iter = fileset.iterator();

         while ( (url == null) &&  (iter.hasNext()))
         {
            String tmpPath = (String) iter.next();
            url = getURL (tmpPath, name);
         }

         if (url == null && resourcePaths != null)
         {
            iter = resourcePaths.iterator();
            while ( (url == null) &&  (iter.hasNext()))
            {
               String tmpPath = (String) iter.next();
               url = getURL (tmpPath, name);
            }
         }
      }
      return url;
   }


   /**
    * searches the classpath and the jars for a given resource. Needed to load images etc.
    * from the plugins. <p>
    *
    * If no matching resource was found on the classpath the resource path is searched.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   protected Enumeration findResources (String name)
   {
      Vector resources = new Vector();

      try
      {
         Enumeration enumeration = super.findResources (name);
         while (enumeration.hasMoreElements())
         {
            resources.add (enumeration.nextElement());
         } // while
      }
      catch (IOException ioEx)
      {
      } // catch

      URL url;
      Iterator iter;

      iter = fileset.iterator();
      while (iter.hasNext())
      {
         String tmpPath = (String) iter.next();
         url = getURL (tmpPath, name);
         if (url != null)
         {
            resources.add (url);
         }
      }

      if (resourcePaths != null)
      {
         iter = resourcePaths.iterator();
         while (iter.hasNext())
         {
            String tmpPath = (String) iter.next();
            url = getURL (tmpPath, name);
            if (url != null)
            {
               resources.add (url);
            }
         }
      }

      return resources.elements();
   }


   /**
    * Get the uRL attribute of the UPBClassLoader object
    *
    * @param path  No description provided
    * @param name  No description provided
    * @return      The uRL value
    */
   private URL getURL (String path, String name)
   {
      URL url = getURL (path);
      if (url != null)
      {
         try
         {
            url = new URL (url, name);
            url.openStream().close();
         }
         catch (Exception ex)
         {
            url = null;
         }
      }
      return url;
   }


   /**
    * Get the uRL attribute of the UPBClassLoader class
    *
    * @param path  No description provided
    * @return      The uRL value
    */
   private static synchronized URL getURL (String path)
   {
      if (pathUrls == null)
      {
         pathUrls = new HashMap();
      }
      URL url = (URL) pathUrls.get (path);

      if (url == null && !pathUrls.containsKey (path))
      {
         File file = new File (path);
         if (file.exists())
         {
            file = file.getAbsoluteFile();
            URI fileUri = file.toURI();
            try
            {
               if (file.isDirectory())
               {
                  url = fileUri.toURL();
               }
               else
               {
                  String lowerPath = path.toLowerCase();
                  if (lowerPath.endsWith (".jar"))
                  {
                     url = new URL ("jar:" + fileUri.toASCIIString() + "!/");
                  }
                  else if (lowerPath.endsWith (".zip"))
                  {
                     url = new URL ("zip:" + fileUri.toASCIIString() + "!/");
                  }
                  else
                  {
                     url = fileUri.toURL();
                  }
               }
            }
            catch (MalformedURLException mue)
            {
               url = null;
            }
         }
         else
         {
            /*
             *  might already be an URL. However, checking that is dangerous, because
             *  a non-existant URLStreamHandler for the URL's protocol might lead to
             *  an infinite loop in the ClassLoader. So pre-assign the path with a
             *  null URL.
             */
            pathUrls.put (path, null);
            try
            {
               url = new URL (path);
            }
            catch (MalformedURLException e)
            {
               url = null;
            }
         }
         pathUrls.put (path, url);
      }
      return url;
   }


   /**
    * access method to remove all references so that the garbage collector can free this instance.
    */
   public void removeYou()
   {
      removeAllFromClasses();
   }


   /**
    * @return   the identifier
    */
   public String getId()
   {
      return id;
   }

}

/*
 * $Log: UPBClassLoader.java,v $
 * Revision 1.43  2005/04/05 08:43:56  cschneid
 * Make Fujaba Webstart-compatible
 *
 */
