/*
 * The FUJABA [Just Draw It!] project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2002 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: fujaba@upb.de
 *
 */
package de.uni_paderborn.lib.javax.swing.progress;

import javax.swing.SwingUtilities;


/**
 * This is the 3rd version of SwingWorker (also known as SwingWorker 3), an abstract class
 * that you subclass to perform GUI-related work in a dedicated thread. For instructions on
 * using this class, see: http://java.sun.com/docs/books/tutorial/uiswing/misc/threads.html
 * Note that the API changed slightly in the 3rd version: You must now invoke start() on the
 * SwingWorker after creating it.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.5 $
 */
public abstract class SwingWorker
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Object value; // see getValue(), setValue()


   /**
    * Class to maintain reference to current worker thread under separate synchronization control.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.5 $
    */
   private static class ThreadVar
   {
      /**
       * the internal controlled thread.
       */
      private Thread thread;


      /**
       * Constructor for class ThreadVar
       *
       * @param t  No description provided
       */
      ThreadVar (Thread t)
      {
         thread = t;
      }


      /**
       * returns the controlled thread of this worker.
       *
       * @return   the controlled thread.
       */
      synchronized Thread get()
      {
         return thread;
      }


      /**
       * deletes the controlled thread.
       */
      synchronized void clear()
      {
         thread = null;
      }
   }


   /**
    * internal used attribute to store an instance of the inner class ThreadVar.
    */
   ThreadVar threadVar;


   /**
    * get the value of the worker.
    *
    * @return   The value value
    */
   protected synchronized Object getValue()
   {
      return value;
   }


   /**
    * Set the value produced by worker thread
    *
    * @param x  The new value value
    */
   synchronized void setValue (Object x)
   {
      value = x;
   }


   /**
    * Compute the value to be returned by the <code>get</code> method.
    *
    * @return   No description provided
    */
   public abstract Object construct();


   /**
    * Called on the event dispatching thread (not on the worker thread) after the <code>construct</code>
    * method has returned.
    */
   public void finished()
   {
   }


   /**
    * A new method that interrupts the worker thread. Call this method to force the worker
    * to stop what it's doing.
    */
   public void interrupt()
   {
      Thread t = threadVar.get();
      if (t != null)
      {
         t.interrupt();
      }
      threadVar.clear();
   }


   /**
    * Return the value created by the <code>construct</code> method. Returns null if either
    * the constructing thread or the current thread was interrupted before a value was produced.
    *
    * @return   the value created by the <code>construct</code> method
    */
   public Object get()
   {
      while (true)
      {
         Thread t = threadVar.get();
         if (t == null)
         {
            return getValue();
         }
         try
         {
            t.join();
         }
         catch (InterruptedException e)
         {
            Thread.currentThread().interrupt(); // propagate
            return null;
         }
      }
   }


   /**
    * Start a thread that will call the <code>construct</code> method and then exit.
    */
   public SwingWorker()
   {
      final Runnable doFinished =
         new Runnable()
         {
            public void run()
            {
               finished();
            }
         };

      Runnable doConstruct =
         new Runnable()
         {
            public void run()
            {
               try
               {
                  setValue (construct());
               }
               finally
               {
                  threadVar.clear();
               }

               SwingUtilities.invokeLater (doFinished);
            }
         };

      Thread t = new Thread (doConstruct);
      threadVar = new ThreadVar (t);
   }


   /**
    * Start the worker thread.
    */
   public void start()
   {
      Thread t = threadVar.get();
      if (t != null)
      {
         t.start();
      }
   }
}

/*
 * $Log: SwingWorker.java,v $
 * Revision 1.5  2004/08/02 07:34:13  lowende
 * Getter method for class loader chain of UPBClassloader added. Javadoc corrected.
 *
 */
