/*
 * The FUJABA [Just Draw It!] project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2002 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: fujaba@upb.de
 *
 */
package de.upb.lib.html;

import java.awt.*;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Vector;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.Document;
import javax.swing.text.html.*;

import de.upb.lib.plugins.DownloadProgress;


/**
 * @author    $Author: lowende $
 * @version   $Revision: 1.2 $
 */
public class HtmlPanel extends JPanel implements HyperlinkListener
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JEditorPane html;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   NavigationToolBar navigationBar = new NavigationToolBar (this);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   Vector history = new Vector (50);


   /**
    * Constructor for class HtmlPanel
    *
    * @param url  No description provided
    */
   public HtmlPanel (URL url)
   {
      setBorder (new EmptyBorder (10, 10, 10, 10));
      setLayout (new BorderLayout());
      getAccessibleContext().setAccessibleName ("HTML panel");
      getAccessibleContext().setAccessibleDescription ("A panel for viewing HTML documents, and following their links");

      try
      {
         if (url != null)
         {

            html = new JEditorPane (url);
            history.add (url);
            html.setEditable (false);
            html.addHyperlinkListener (this);
            JScrollPane scroller = new JScrollPane();
            scroller.setPreferredSize (new Dimension (400, 300));
            scroller.setBorder (new SoftBevelBorder (BevelBorder.LOWERED));
            JViewport vp = scroller.getViewport();
            vp.add (html);
            vp.setScrollMode (JViewport.BACKINGSTORE_SCROLL_MODE);

            add (navigationBar, BorderLayout.NORTH);

            add (scroller, BorderLayout.CENTER);

         }
      }
      catch (MalformedURLException e)
      {
         add (new JLabel ("Malformed URL: " + e));
      }
      catch (IOException e)
      {
         add (new JLabel ("IOException: " + e));
      }
   }


   /**
    * Sets the navigationBarVisible attribute of the HtmlPanel object
    *
    * @param visible  The new navigationBarVisible value
    */
   public void setNavigationBarVisible (boolean visible)
   {
      navigationBar.setVisible (visible);
   }


   /**
    * Turns buttons in navigation bar on or off
    *
    * @param backEnabled     turn back button on or off
    * @param forwardEnabled  turn forward button on or off
    * @param homeEnabled     turn home button on or off
    */
   public void setNavigationBarButtonEnablity (boolean backEnabled, boolean forwardEnabled, boolean homeEnabled)
   {
      navigationBar.setButtonEnablity (backEnabled, forwardEnabled, homeEnabled);
   }


   /**
    * Constructor for class HtmlPanel
    *
    * @param urlString  No description provided
    */
   public HtmlPanel (String urlString)
   {
      setBorder (new EmptyBorder (10, 10, 10, 10));
      setLayout (new BorderLayout());
      getAccessibleContext().setAccessibleName ("HTML panel");
      getAccessibleContext().setAccessibleDescription ("A panel for viewing HTML documents, and following their links");

      try
      {
         URL url = null;

         String prefix = "file:///" +
            System.getProperty ("user.dir") +
            System.getProperty ("file.separator");

         try
         {
            url = new URL (prefix + urlString);
         }
         catch (java.net.MalformedURLException exc)
         {
            add (new JLabel ("Attempted to open a bad URL: " + url));
            url = null;
         }

         if (url != null)
         {
            html = new JEditorPane();
            html.setEditorKitForContentType ("text/html", new HTMLEditorKit());
            html.setContentType ("text/html");

            // check for fujab template files (*.ft.html)
            String fileName = url.getFile();
            int pos = fileName.indexOf ('?');
            if (pos >= 0)
            {
               fileName = fileName.substring (0, pos);
            }

            html.setPage (url);
            history.add (url);
            html.setEditable (false);

            html.addHyperlinkListener (this);

            JScrollPane scroller = new JScrollPane();
            scroller.setPreferredSize (new Dimension (400, 300));
            scroller.setBorder (new SoftBevelBorder (BevelBorder.LOWERED));
            JViewport vp = scroller.getViewport();
            vp.add (html);
            vp.setScrollMode (JViewport.BACKINGSTORE_SCROLL_MODE);
            add (scroller, BorderLayout.CENTER);
         }
      }
      catch (MalformedURLException e)
      {
         add (new JLabel ("Malformed URL: " + e));
      }
      catch (IOException e)
      {
         add (new JLabel ("IOException: " + e));
      }
   }


   /**
    * Notification of a change relative to a hyperlink.
    *
    * @param e  No description provided
    */
   public void hyperlinkUpdate (HyperlinkEvent e)
   {
      if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED)
      {
         String fileName = e.getURL().getFile();
         //fileName = fileName.substring (fileName.lastIndexOf (System.getProperty ("file.separator")) + 1, fileName.length());
         // might be a plugin for download
         if (fileName.endsWith (".zip") || fileName.endsWith (".ZIP"))
         {
            try
            {
               DownloadProgress dp = new DownloadProgress (e.getURL(), "Downloading...");
               dp.show();
               dp.download();
               return;
            }
            catch (Exception ex)
            {
               JOptionPane.showMessageDialog (getRootPane(), "<html>An error is occured while downloading of plug-in:<br>" +
                  "\"" + fileName + "\" <br>" +
                  "Error: " + ex + "</html>", "Error", JOptionPane.ERROR_MESSAGE);

               ex.printStackTrace();
            }
         }
         else
         { // html doc handling
            linkActivated (e);
         }

      }
   }


   /**
    * Follows the reference in an link. The given url is the requested reference. By default
    * this calls <a href="#setPage">setPage</a> , and if an exception is thrown the original
    * previous document is restored and a beep sounded. If an attempt was made to follow a
    * link, but it represented a malformed url, this method will be called with a null argument.
    *
    * @param e  No description provided
    */
   protected void linkActivated (HyperlinkEvent e)
   {
      Cursor c = html.getCursor();
      Cursor waitCursor = Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR);
      html.setCursor (waitCursor);
      SwingUtilities.invokeLater (new PageLoader (e, c));
   }


   /**
    * temporary class that loads synchronously (although later than the request so that a cursor
    * change can be done).
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.2 $
    */
   class PageLoader implements Runnable
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      HyperlinkEvent event;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      URL url;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      Cursor cursor;


      /**
       * Constructor for class PageLoader
       *
       * @param e  No description provided
       * @param c  No description provided
       */
      PageLoader (HyperlinkEvent e, Cursor c)
      {
         event = e;
         url = e.getURL();
         navigationBar.incPageIndex();
         history.add (navigationBar.getPageIndex(), url);
         navigationBar.getBackButton().setEnabled (true);
         cursor = c;
      }


      /**
       * Main processing method for the PageLoader object
       */
      public void run()
      {
         if (url == null)
         {
            // restore the original cursor
            html.setCursor (cursor);

            // PENDING(prinz) remove this hack when
            // automatic validation is activated.
            Container parent = html.getParent();
            parent.repaint();
         }
         else
         {
            Document doc = html.getDocument();
            try
            {
               // might be an frame event
               if (event instanceof HTMLFrameHyperlinkEvent)
               {
                  HTMLFrameHyperlinkEvent evt = (HTMLFrameHyperlinkEvent) event;

                  html = (JEditorPane) event.getSource();
                  doc = (HTMLDocument) html.getDocument();
                   ((HTMLDocument) doc).processHTMLFrameHyperlinkEvent (evt);
               }
               else
               {

                  // handle standard href
                  html.setPage (url);
               }

            }
            catch (Exception e)
            {
               html.setDocument (doc);
               e.printStackTrace();
               getToolkit().beep();
            }
            finally
            {
               // schedule the cursor to revert after
               // the paint has happended.
               url = null;
               SwingUtilities.invokeLater (this);
            }
         }
      }
   }
}

/*
 * $Log: HtmlPanel.java,v $
 * Revision 1.2  2004/08/02 07:34:14  lowende
 * Getter method for class loader chain of UPBClassloader added. Javadoc corrected.
 *
 */
