/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.upb.lib.userinterface;

import java.net.URL;
import java.util.Stack;

import javax.swing.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import de.uni_paderborn.lib.basic.ImageResourceManager;
import de.uni_paderborn.lib.classloader.UPBClassLoader;
import de.upb.lib.userinterface.action.CompositeAction;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *                           0..1                  0..1
 *  UserInterfaceSaxHandler ---------------------------- UserInterfaceManager
 *                           handler            manager
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.28 $
 */
class UserInterfaceSaxHandler extends DefaultHandler
{
   /**
    * URL used in UserInterface definitions (XML-files) to identify the DTD. DTD given as "SYSTEM"
    * (non-public) definition.
    */
   private final static String USERINTERFACE_SYSTEM_ID = "http://www.upb.de/cs/fujaba/DTDs/UserInterface.dtd";

   /**
    * Alternative URL used in UserInterface definitions (XML-files) to identify the DTD. DTD
    * given as "SYSTEM" (non-public) definition.
    */
   private final static String USERINTERFACE_SYSTEM_ID2 = "http://www.fujaba.de/DTDs/UserInterface.dtd";

   /**
    * Name used in UserInterface definitions (XML-files) to identify the DTD. DTD given as
    * "PUBLIC" definition.
    */
   private final static String USERINTERFACE_PUBLIC_ID = "-//Fujaba//Fujaba User Interface Definition//EN//1.0";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static String USERINTERFACE_DTD_RESOURCE = "DTDs/UserInterface.dtd";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient URL userInterfaceDtdUrl = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int NONE = 0;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int NAME = 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int SHORTCUT = 2;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int MNEMONIC = 3;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int TOOLTIP = 4;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int ICON = 5;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int RESPONSIBLE = 6;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int state;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private AbstractAction currentAction;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private CompositeAction currentCompositeAction;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Stack sectionStack;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Stack containerStack;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String key;


   /**
    * Get the key attribute of the UserInterfaceSaxHandler object
    *
    * @return   The key value
    */
   public String getKey()
   {
      return key;
   }


   /**
    * Sets the key attribute of the UserInterfaceSaxHandler object
    *
    * @param key  The new key value
    */
   public void setKey (String key)
   {
      if (key != null)
      {
         this.key = key;
      }
   }


   /**
    * <pre>
    *                           0..1                  0..1
    *  UserInterfaceSaxHandler ---------------------------- UserInterfaceManager
    *                           handler            manager
    * </pre>
    */
   private UserInterfaceManager manager = UserInterfaceManager.get();


   /**
    * Sets the manager attribute of the UserInterfaceSaxHandler object
    *
    * @param value  The new manager value
    * @return       No description provided
    */
   public boolean setManager (UserInterfaceManager value)
   {
      if (this.manager != value)
      {
         if (this.manager != null)
         {
            //UserInterfaceManager oldValue = this.manager;
            this.manager = null;
            //oldValue.setHandler (null);
         }
         this.manager = value;
         if (value != null)
         {
            //this.manager.setHandler (this);
         }

         return true;
      }

      return false;
   }


   /**
    * Get the manager attribute of the UserInterfaceSaxHandler object
    *
    * @return   The manager value
    */
   public UserInterfaceManager getManager()
   {
      return this.manager;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param publicId       No description provided
    * @param systemId       No description provided
    * @return               No description provided
    * @throws SAXException  Exception description not provided
    */
   public InputSource resolveEntity (String publicId, String systemId) throws SAXException
   {
      if (USERINTERFACE_PUBLIC_ID.equals (publicId)
         || USERINTERFACE_SYSTEM_ID.equals (systemId)
         || USERINTERFACE_SYSTEM_ID2.equals (systemId))
      {
         if (userInterfaceDtdUrl == null)
         {
            userInterfaceDtdUrl = UPBClassLoader.get (key).getResource (USERINTERFACE_DTD_RESOURCE);
         }
         if (userInterfaceDtdUrl != null)
         {
            try
            {
               return new InputSource (userInterfaceDtdUrl.openStream());
            }
            catch (Exception e)
            {
               System.out.println ("Could not solve SYSTEM or PUBLIC reference for DTD.");
               throw new SAXException (e);
            }
         }
      }

      return super.resolveEntity (publicId, systemId);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void startDocument()
   {
      state = NONE;
      containerStack = new Stack();
      sectionStack = new Stack();
   } // startDocument


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void endDocument()
   {
      state = NONE;
      containerStack = null;
      sectionStack = null;
      currentAction = null;
      currentCompositeAction = null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param uri    No description provided
    * @param name   No description provided
    * @param qName  No description provided
    * @param atts   No description provided
    */
   public void startElement (String uri, String name, String qName, Attributes atts)
   {
      try
      {
         // normalize the tag identifier
         name = name.toLowerCase();

         if (name.equals ("action"))
         {
            currentAction = createAction (atts);
         }
         else
            if (name.equals ("initializer"))
         {
            createInitializer (atts);
         }
         else
            if (name.equals ("menubar"))
         {
            containerStack.push (createMenuBar (atts));
         }
         else
            if (name.equals ("menu"))
         {
            containerStack.push (createMenu (atts));
         }
         else
            if (name.equals ("popupmenu"))
         {
            containerStack.push (createPopupMenu (atts));
         }
         else
            if (name.equals ("toolbar"))
         {
            containerStack.push (createToolBar (atts));
         }
         else
            if (name.endsWith ("section"))
         {
            sectionStack.push (createSection (atts));
         }
         else
            if (name.equals ("menuitem"))
         {
            ActionItem actionItem = createMenuItem (atts);
            if (actionItem != null)
            {
                ((Section) sectionStack.peek()).addToItems (actionItem);
            }
         }
         else
            if (name.equals ("button"))
         {
            ActionItem actionItem = createButton (atts);
            if (actionItem != null)
            {
                ((Section) sectionStack.peek()).addToItems (actionItem);
            }
         }
         else
            if (name.equals ("composite"))
         {
            currentCompositeAction = getComposite (atts);
         }
         else
            if (name.equals ("responsible"))
         {
            state = RESPONSIBLE;
         }
         else
            if (name.equals ("name"))
         {
            state = NAME;
         }
         else
            if (name.equals ("shortcut"))
         {
            state = SHORTCUT;
         }
         else
            if (name.equals ("mnemonic"))
         {
            state = MNEMONIC;
         }
         else
            if (name.equals ("tooltip"))
         {
            state = TOOLTIP;
         }
         else
            if (name.equals ("icon"))
         {
            state = ICON;
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
   } // startElement


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param ch      No description provided
    * @param start   No description provided
    * @param length  No description provided
    */
   public void characters (char[] ch, int start, int length)
   {
      try
      {
         StringBuffer buffer = new StringBuffer (length);

         buffer.append (ch, start, length);

         switch (state)
         {
            case NAME:
               if (currentAction != null)
               {
                  currentAction.putValue (Action.NAME, buffer.toString());
               }
               else
                  if (!containerStack.empty())
               {
                  SectionContainer container = (SectionContainer) containerStack.peek();
                  container.setName (buffer.toString());
               }
               break;
            case SHORTCUT:
            {
               KeyStroke keyStroke = KeyStroke.getKeyStroke (buffer.toString());
               if (buffer.toString().trim().length() > 0 && keyStroke == null)
               {
                  System.err.println (
                     "Failed to parse keystroke '"
                     + buffer.toString()
                     + "' "
                     + "see javax.swing.KeyStroke.getKeyStroke(String) for correct syntax.");
               }
               currentAction.putValue (Action.ACCELERATOR_KEY, keyStroke);
               break;
            }
            case MNEMONIC:
            {
               String character = new Character (buffer.charAt (0)).toString().toUpperCase();
               KeyStroke keyStroke = KeyStroke.getKeyStroke ("pressed " + character);
               if (keyStroke != null)
               {
                  Integer mnemonic = new Integer (keyStroke.getKeyCode());

                  if (currentAction != null)
                  {
                     currentAction.putValue (Action.MNEMONIC_KEY, mnemonic);
                  }
                  else
                     if (!containerStack.empty())
                  {
                     Object container = containerStack.peek();
                     if (container instanceof MenuContainer)
                     {
                         ((MenuContainer) container).setMnemonic (mnemonic.intValue());
                     }
                  }
               }
               else
               {
                  System.err.println ("Failed to find key code for '" + buffer.charAt (0) + "'");
               }
               break;
            }
            case TOOLTIP:
               if (currentAction != null)
               {
                  currentAction.putValue (Action.SHORT_DESCRIPTION, buffer.toString());
               }
               break;
            case ICON:
               String iconPath = buffer.toString();
               if (iconPath.length() != 0)
               {
                  ImageIcon icon = ImageResourceManager.get().getImageIcon (key, iconPath);
                  if (icon != null)
                  {
                     if (currentAction != null)
                     {
                        currentAction.putValue (Action.SMALL_ICON, icon);
                     }
                     else
                        if (!containerStack.empty())
                     {
                        Object container = containerStack.peek();
                        if (container instanceof MenuContainer)
                        {
                            ((MenuContainer) container).setIcon (icon);
                        }
                     }
                  }
                  else
                  {
                     System.err.println ("Icon " + iconPath + " could not be loaded!");
                  }
               }
               break;
            case RESPONSIBLE:
               String currentActionName = currentAction.getValue (Action.NAME).toString();
               if (currentCompositeAction == null)
               {
                  System.err.println ("No composite action previously selected in action named '" + currentActionName + "'.");
                  return;
               }
               String responsibleClassName = buffer.toString().trim();
               currentCompositeAction.registerAction (responsibleClassName, currentAction);
               break;
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param uri    No description provided
    * @param name   No description provided
    * @param qName  No description provided
    */
   public void endElement (String uri, String name, String qName)
   {
      // normalize the tag identifier
      name = name.toLowerCase();

      if (name.equals ("action"))
      {
         currentAction = null;
      }

      else
         if (name.equals ("menubar"))
      {
         containerStack.clear();
      }
      else
         if (name.equals ("menu") || name.equals ("popupmenu") || name.equals ("toolbar"))
      {
         SectionContainer container = (SectionContainer) containerStack.pop();

         if (containerStack.empty())
         {
            getManager().addToContainers (container);
         }
      }

      else
         if (name.endsWith ("section"))
      {
         sectionStack.pop();
      }
      else
         if (name.equals ("composite"))
      {
         currentCompositeAction = null;
      }
      else
         if (name.equals ("menuitem")
         || name.equals ("button")
         || name.equals ("shortcut")
         || name.equals ("mnemonic")
         || name.equals ("tooltip")
         || name.equals ("responsible")
         || name.equals ("icon"))
      {
         state = NONE;
      }
   } // endElement


   /**
    * Get the composite attribute of the UserInterfaceSaxHandler object
    *
    * @param attrs  No description provided
    * @return       The composite value
    */
   private CompositeAction getComposite (Attributes attrs)
   {
      CompositeAction compositeAction = null;
      String id = attrs.getValue ("id");

      AbstractAction action = getManager().getFromActions (id);
      if (action == null)
      {
         System.err.println ("No action with '" + id + "' created before!");
      }
      else
         if (! (action instanceof CompositeAction))
      {
         System.err.println ("The action with '" + id + "' is not a composite action!");
      }
      else
      {
         compositeAction = (CompositeAction) action;
      }
      return compositeAction;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private AbstractAction createAction (Attributes atts)
   {
      AbstractAction abstractAction = null;
      UPBClassLoader loader = UPBClassLoader.get (key);

      String className = atts.getValue ("class");
      String id = atts.getValue ("id");

      if (getManager().getFromActions (id) == null)
      {
         try
         {
            Class actionClass = Class.forName (className, true, loader);

            try
            {
               abstractAction = (AbstractAction) actionClass.newInstance();
               abstractAction.setEnabled (Boolean.valueOf (atts.getValue ("enabled")).booleanValue());
               abstractAction.putValue ("id", id);

               getManager().addToActions (id, abstractAction);
            }
            catch (Exception e)
            {
               System.err.println ("Action '" + className + "' could not be instantiated!");
               e.printStackTrace();
            }
         }
         catch (ClassNotFoundException e)
         {
            System.err.println ("Action '" + className + "' could not be loaded due to: " + e);
         }
         catch (NoClassDefFoundError e)
         {
            System.err.println ("Action '" + className + "' could not be loaded due to: " + e);
         }
      }
      else
      {
         System.err.println ("An action with id '" + id + "' has already been loaded!");
      }

      return abstractAction;
   } // createAction


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private ElementInitializer createInitializer (Attributes atts)
   {
      ElementInitializer initializer = null;
      UPBClassLoader loader = UPBClassLoader.get (key);

      String className = atts.getValue ("class");
      String id = atts.getValue ("id");

      if (getManager().getFromInitializers (id) == null)
      {
         try
         {
            Class initializerClass = Class.forName (className, true, loader);

            try
            {
               initializer = (ElementInitializer) initializerClass.newInstance();
               getManager().addToInitializers (id, initializer);
            }
            catch (Exception e)
            {
               System.err.println ("Initializer '" + className + "' could not be instantiated!");
               e.printStackTrace();
            }
         }
         catch (ClassNotFoundException e)
         {
            System.err.println ("Initializer '" + className + "' could not be loaded!");
         }
      }
      else
      {
         System.err.println ("An Initializer with id '" + id + "' has already been loaded!");
      }

      return initializer;
   } // createInitializer


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private MenubarContainer createMenuBar (Attributes atts)
   {
      String id = atts.getValue ("id");
      SectionContainer container = getManager().getFromContainers (id);
      MenubarContainer menubarContainer = null;

      if (container == null)
      {
         container = new MenubarContainer (id);
         getManager().addToContainers (container);
         menubarContainer = (MenubarContainer) container;
      }
      else
         if (! (container instanceof MenubarContainer))
      {
         System.err.println (
            "UserInterfaceSaxHandler: Trying to insert menu bar '"
            + id
            + "' but menu or popup menu with same name already exists!");
      }
      else
      {
         menubarContainer = (MenubarContainer) container;
      }

      return menubarContainer;
   } // createMenuBar


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private MenuContainer createMenu (Attributes atts)
   {
      String id = atts.getValue ("id");
      Section currentSection = (Section) sectionStack.peek();
      SectionItem sectionItem = currentSection.getFromItems (id);
      MenuContainer menuContainer = null;

      if (sectionItem == null)
      {
         sectionItem = new MenuContainer (id);
         currentSection.addToItems (sectionItem);
         menuContainer = (MenuContainer) sectionItem;
      }
      else
      {
         menuContainer = (MenuContainer) sectionItem;
      }

      String visibleText = atts.getValue ("visible");
      if (visibleText != null)
      {
         menuContainer.setVisible (Boolean.valueOf (visibleText).booleanValue());
      }

      return menuContainer;
   } // createMenu


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private PopupMenuContainer createPopupMenu (Attributes atts)
   {
      String className = atts.getValue ("class");
      SectionContainer container = getManager().getFromContainers (className);
      PopupMenuContainer popupMenuContainer = null;

      if (container == null)
      {
         container = new PopupMenuContainer (className);
         getManager().addToContainers (container);
         popupMenuContainer = (PopupMenuContainer) container;
      }
      else
         if (! (container instanceof PopupMenuContainer))
      {
         System.err.println (
            "UserInterfaceSaxHandler: Trying to insert popup menu '"
            + className
            + "' but menu or toolbar with same name already exists!");
      }
      else
      {
         popupMenuContainer = (PopupMenuContainer) container;
      }

      return popupMenuContainer;
   } // createPopupMenu


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private ToolbarContainer createToolBar (Attributes atts)
   {
      String id = atts.getValue ("id");
      SectionContainer container = getManager().getFromContainers (id);
      ToolbarContainer toolbarContainer = null;

      if (container == null)
      {
         container = new ToolbarContainer (id);
         getManager().addToContainers (container);
         toolbarContainer = (ToolbarContainer) container;
      }
      else
         if (! (container instanceof ToolbarContainer))
      {
         System.err.println (
            "UserInterfaceSaxHandler: Trying to insert toolbar '"
            + id
            + "' but menu or popup menu with same name already exists!");
      }
      else
      {
         toolbarContainer = (ToolbarContainer) container;
      }

      String rolloverButtonsText = atts.getValue ("rolloverButtons");
      if (rolloverButtonsText != null)
      {
         toolbarContainer.setRolloverButtons (Boolean.valueOf (rolloverButtonsText).booleanValue());
      }

      String visibleText = atts.getValue ("visible");
      if (visibleText != null)
      {
         toolbarContainer.setVisible (Boolean.valueOf (visibleText).booleanValue());
      }

      return toolbarContainer;
   } // createToolBar


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private Section createSection (Attributes atts)
   {
      String id = atts.getValue ("id");
      Section section =  ((SectionContainer) containerStack.peek()).getFromSections (id);

      if (section == null)
      {
         section = new Section (id);
          ((SectionContainer) containerStack.peek()).addToSections (section);
      }

      return section;
   } // createSection


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private ActionItem createMenuItem (Attributes atts)
   {
      ActionItem actionItem = null;
      String actionId = atts.getValue ("actionId");
      AbstractAction action = getManager().getFromActions (actionId);

      if (action != null)
      {
         ElementInitializer initializer = null;
         String initText = atts.getValue ("initializer");
         if (initText != null)
         {
            initializer = getManager().getFromInitializers (initText);
            if (initializer == null)
            {
               System.err.println ("The Initializer `" + initText + "` doesn't exist.");
            }
         }

         String typeText = atts.getValue ("type");

         int type = ActionItem.MENU_ITEM;

         if (typeText != null)
         {
            if (typeText.equals ("menuItem"))
            {
               type = ActionItem.MENU_ITEM;
            }
            else
               if (typeText.equals ("checkBox"))
            {
               type = ActionItem.CHECKBOX;
            }
            else
               if (typeText.equals ("radioButton"))
            {
               type = ActionItem.RADIOBUTTON;
            }
         }

         actionItem = new ActionItem (action, type, initializer);

         String selectedText = atts.getValue ("selected");
         if (selectedText != null)
         {
            actionItem.setSelected (Boolean.valueOf (selectedText).booleanValue());
         }
      }
      else
      {
         System.err.println ("The actionId `" + actionId + "` doesn't exist.");
      }

      return actionItem;
   } // createMenuItem


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param atts  No description provided
    * @return      No description provided
    */
   private ActionItem createButton (Attributes atts)
   {
      ActionItem actionItem = null;
      String actionId = atts.getValue ("actionId");
      AbstractAction action = getManager().getFromActions (actionId);

      if (action != null)
      {
         if ("toggle".equals (atts.getValue ("type")))
         {
            actionItem = new ActionItem (action, ActionItem.TOGGLE_BUTTON);
         }
         else
         {
            actionItem = new ActionItem (action, ActionItem.BUTTON);
         }

         String value = atts.getValue ("showText");
         if (value != null)
         {
            actionItem.setShowText (Boolean.valueOf (value).booleanValue());
         }
      }
      else
      {
         System.err.println ("The actionId `" + actionId + "` doesn't exist.");
      }

      return actionItem;
   } // createButton

}

/*
 * $Log: UserInterfaceSaxHandler.java,v $
 * Revision 1.28  2005/08/23 07:59:03  lowende
 * Removed compile warnings.
 *
 */
