/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.app;

import java.awt.*;
import java.lang.reflect.InvocationTargetException;
import java.util.Enumeration;
import java.util.Iterator;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.tree.*;

import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.upb.tools.fca.*;


/**
 * <h2>Associations</h2> <pre>
 *            0..1   tabbedPaneProxy    0..1
 * FrameMain -------------------------------- TabbedPaneProxy
 *            frameMain      tabbedPaneProxy
 *
 *                 --------- 0..1      tabProxy       0..1
 * TabbedPaneProxy | title |------------------------------- TabProxy
 *                 --------- tabbedPaneProxy      tabProxy
 * </pre>
 *
 * @author    $Author: trinet $
 * @version   $Revision: 1.37 $
 */
public class TabbedPaneProxy
{
   // --------------------------------------------------------------------------
   // CONSTRUCTORS
   // --------------------------------------------------------------------------

   /**
    * Constructor for class TabbedPaneProxy
    */
   public TabbedPaneProxy()
   {
      this.tabbedPane = new JTabbedPane (SwingConstants.BOTTOM);
      this.tabbedPane.addChangeListener (new TabbedPaneProxyListener());
   }


   // --------------------------------------------------------------------------
   // ASSOCS
   // --------------------------------------------------------------------------

   /**
    * <pre>
    *            0..1   tabbedPaneProxy    0..1
    * FrameMain -------------------------------- TabbedPaneProxy
    *            frameMain      tabbedPaneProxy
    * </pre>
    */
   private FrameMain frameMain;


   /**
    * Sets the frameMain attribute of the TabbedPaneProxy object
    *
    * @param value  The new frameMain value
    * @return       No description provided
    */
   public boolean setFrameMain (FrameMain value)
   {
      boolean changed = false;
      if (this.frameMain != value)
      {
         if (this.frameMain != null)
         {
            FrameMain oldValue = this.frameMain;
            this.frameMain = null;
            oldValue.setTabbedPaneProxy (null);
         }
         this.frameMain = value;
         if (value != null)
         {
            value.setTabbedPaneProxy (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * Get the frameMain attribute of the TabbedPaneProxy object
    *
    * @return   The frameMain value
    */
   public FrameMain getFrameMain()
   {
      return this.frameMain;
   }


   /**
    * <pre>
    *                 --------- 0..1      tabProxy       0..1
    * TabbedPaneProxy | title |------------------------------- TabProxy
    *                 --------- tabbedPaneProxy      tabProxy
    * </pre>
    */
   private FTreeMap tabProxy;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInTabProxy (TabProxy value)
   {
      return  ( (this.tabProxy != null) &&
          (value != null) &&  (value.getTitle() != null) &&
          (this.tabProxy.get (value.getTitle()) == value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInTabProxy (String key)
   {
      return  ( (this.tabProxy != null) &&
          (key != null) &&
         this.tabProxy.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfTabProxy()
   {
      return  ( (this.tabProxy == null)
         ? FEmptyIterator.get()
         : this.tabProxy.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfTabProxy()
   {
      return  ( (this.tabProxy == null)
         ? FEmptyIterator.get()
         : this.tabProxy.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfTabProxy()
   {
      return  ( (this.tabProxy == null)
         ? FEmptyIterator.get()
         : this.tabProxy.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfTabProxy()
   {
      return  ( (this.tabProxy == null)
         ? 0
         : this.tabProxy.size());
   }


   /**
    * Get the fromTabProxy attribute of the TabbedPaneProxy object
    *
    * @param key  No description provided
    * @return     The fromTabProxy value
    */
   public TabProxy getFromTabProxy (String key)
   {
      return  ( ( (this.tabProxy == null) ||  (key == null))
         ? null
         : (TabProxy) this.tabProxy.get (key));
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToTabProxy (TabProxy value)
   {
      boolean changed = false;
      if ( (value != null) &&  (value.getTitle() != null))
      {
         if (this.tabProxy == null)
         {
            this.tabProxy = new FTreeMap();
         }
         TabProxy oldValue = (TabProxy) this.tabProxy.put (value.getTitle(), value);
         if (oldValue != value)
         {
            if (oldValue != null)
            {
               oldValue.setTabbedPaneProxy (null);
            }
            value.setTabbedPaneProxy (this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromTabProxy (TabProxy value)
   {
      boolean changed = false;
      if ( (this.tabProxy != null) &&  (value != null) &&  (value.getTitle() != null))
      {
         TabProxy oldValue = (TabProxy) this.tabProxy.get (value.getTitle());
         if (oldValue == value)
         {
            this.tabProxy.remove (value.getTitle());
            value.setTabbedPaneProxy (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromTabProxy (String key)
   {
      boolean changed = false;
      if ( (this.tabProxy != null) &&  (key != null))
      {
         TabProxy tmpValue = (TabProxy) this.tabProxy.get (key);
         if (tmpValue != null)
         {
            this.tabProxy.remove (key);
            tmpValue.setTabbedPaneProxy (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromTabProxy()
   {
      TabProxy tmpValue;
      Iterator iter = this.iteratorOfTabProxy();
      while (iter.hasNext())
      {
         tmpValue = (TabProxy) iter.next();
         this.removeFromTabProxy (tmpValue);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      FrameMain tmpFrameMain = getFrameMain();
      if (tmpFrameMain != null)
      {
         setFrameMain (null);
      }

      removeAllFromTabProxy();
   }


   // --------------------------------------------------------------------------
   // MANIPULATORS
   // --------------------------------------------------------------------------

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JTabbedPane tabbedPane;


   /**
    * Sets the tabbedPane attribute of the TabbedPaneProxy object
    *
    * @param tabbedPane  The new tabbedPane value
    */
   public void setTabbedPane (JTabbedPane tabbedPane)
   {
      if (this.tabbedPane != tabbedPane)
      {
         this.tabbedPane = tabbedPane;
      }
   } // setTabbedPane


   /**
    * Get the tabbedPane attribute of the TabbedPaneProxy object
    *
    * @return   The tabbedPane value
    */
   public JTabbedPane getTabbedPane()
   {
      return this.tabbedPane;
   } // getTabbedPane


   /**
    * Updates all tabbed panes with new tree items.
    */
   public void update()
   {
      // Save user selection in tree.
      ASGDiagram selectedDiagram = null;
      Object selectedObject = null;
      DefaultMutableTreeNode selectedNode = getFrameMain().getSelectedTreeNode();

      if (selectedNode != null)
      {
         selectedObject = selectedNode.getUserObject();
         if (selectedObject != null && selectedObject instanceof ASGDiagram)
         {
            selectedDiagram = (ASGDiagram) selectedObject;
         }
      }

      Iterator tabIter = iteratorOfTabProxy();
      while (tabIter.hasNext())
      {
         TabProxy tmpTab = (TabProxy) tabIter.next();

         // Remove children.
         tmpTab.getRoot().removeAllChildren();
      }

      // Retrieve all tree nodes.
      Iterator decoratorIter = getFrameMain().iteratorOfDecorators();
      while (decoratorIter.hasNext())
      {
         FrameDecorator decorator = (FrameDecorator) decoratorIter.next();

         Iterator diagsIter = UMLProject.get().iteratorOfDiags();
         DefaultMutableTreeNode node = decorator.getTreeNode (new EnumerationForAnIterator (diagsIter));
         if ( (node != null) &&  (node.getChildCount() != 0))
         {
            if (!node.isLeaf())
            {
               TabProxy tmpTab = instanceOfTabProxy (decorator.getDestTabTitle());
               tmpTab.addTreeNode (node);
               tmpTab.setTreeCellRenderer (decorator.getTreeCellRenderer());
            }
         }
         else
         {
            String tabTitle = decorator.getDestTabTitle();
            if ( (tabTitle != null) && instanceOfTabProxy (decorator.getDestTabTitle()).isTabEmpty() && !tabTitle.equals (TabbedPaneProxy.TAB_PRJ))
            {
               removeKeyFromTabProxy (tabTitle);
            }
         }
      }

      // Add all tabs in sorted order
      Iterator iter = iteratorOfTabProxy();
      final int tabCount = sizeOfTabProxy();
      final TabProxy[] proxies = new TabProxy[tabCount];
      for (int i = 0; i < tabCount && iter.hasNext(); i++)
      {
         TabProxy tmpTab = (TabProxy) iter.next();
         tmpTab.init();
         tmpTab.getTree().setModel (new DefaultTreeModel (tmpTab.getRoot()));
         proxies[i] = tmpTab;
      }

      final JTabbedPane pane = getTabbedPane();

      Runnable tabUpdate =
         new Runnable()
         {
            public void run()
            {
               // Remove all tabs from tabbed pane
               pane.removeAll();
               TabProxy tmpTab;
               // Add all tabs in sorted order
               for (int i = 0; i < tabCount; i++)
               {
                  tmpTab = proxies[i];
                  pane.addTab (tmpTab.getTitle(), tmpTab.getIcon(), tmpTab.getComponent(), tmpTab.getTip());
               }
            }
         };

      if (SwingUtilities.isEventDispatchThread())
      {
         tabUpdate.run();
      }
      else
      {
         try
         {
            SwingUtilities.invokeAndWait (tabUpdate);
         }
         catch (InterruptedException ex)
         {
         }
         catch (InvocationTargetException ex)
         {
            throw new RuntimeException (ex);
         }
      }

      // Set old selection.
      if (selectedObject != null)
      {
         getFrameMain().selectTreeItem (selectedObject);
      }

      // Update frame decorators
      getFrameMain().doEverythingOnEntry (selectedDiagram);
   } // update


   // --------------------------------------------------------------------------
   // COMFORT
   // --------------------------------------------------------------------------

   // Shortcuts for tab titles
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_PRJ = "UML";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_SDL = "SDL";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_CMS = "CMS";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_SQL = "SQL";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_FP = "Pattern";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_PATH = "Paths";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_XML = "XML";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_FUI = "FUI";

   // Tooltips for tabs
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_PRJ_TIP = "UML Diagram View";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_SDL_TIP = "SDL Diagram View";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_CMS_TIP = "Consistency Management System";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_SQL_TIP = "SQL Diagram View";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_FP_TIP = "Pattern Specification";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_PATH_TIP = "Path Specification";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_XML_TIP = "XML Diagram View";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TAB_FUI_TIP = "Fujaba User Interface";


   /**
    * Return the tab with the specified title. If no such a tab exists, create a new one.
    *
    * @param title  No description provided
    * @return       No description provided
    */
   public TabProxy instanceOfTabProxy (String title)
   {
      TabProxy tmpTabProxy = null;

      if (title != null)
      {
         if (hasKeyInTabProxy (title))
         {
            tmpTabProxy = getFromTabProxy (title);
         }
         else
         {
            if (title.equals (TAB_PRJ))
            {
               tmpTabProxy = new TabProxy (TAB_PRJ, TAB_PRJ, TAB_PRJ_TIP, null);
               addToTabProxy (tmpTabProxy);
            }
            else if (title.equals (TAB_SDL))
            {
               tmpTabProxy = new TabProxy (TAB_SDL, TAB_SDL, TAB_SDL_TIP, null);
               addToTabProxy (tmpTabProxy);
            }
            else if (title.equals (TAB_CMS))
            {
               tmpTabProxy = new TabProxy (TAB_CMS, TAB_CMS, TAB_CMS_TIP, null);
               addToTabProxy (tmpTabProxy);
            }
            else if (title.equals (TAB_FP))
            {
               tmpTabProxy = new TabProxy (TAB_FP, TAB_FP, TAB_FP_TIP, null);
               addToTabProxy (tmpTabProxy);
            }
            else if (title.equals (TAB_SQL))
            {
               tmpTabProxy = new TabProxy (TAB_SQL, TAB_SQL, TAB_SQL_TIP, null);
               addToTabProxy (tmpTabProxy);
            }
            else if (title.equals (TAB_PATH))
            {
               tmpTabProxy = new TabProxy (TAB_PATH, TAB_PATH, TAB_PATH_TIP, null);
               addToTabProxy (tmpTabProxy);
            }
            else if (title.equals (TAB_FUI))
            {
               tmpTabProxy = new TabProxy (TAB_FUI, TAB_FUI, TAB_FUI_TIP, null);
               addToTabProxy (tmpTabProxy);
            }
            else if (title.equals (TAB_XML))
            {
               tmpTabProxy = new TabProxy (TAB_XML, TAB_XML, TAB_XML_TIP, null);
               addToTabProxy (tmpTabProxy);
            }
            else
            {
               tmpTabProxy = new TabProxy (title, title, title, null);
               addToTabProxy (tmpTabProxy);
            }

         }
      }

      return tmpTabProxy;
   } // getTabProxy


   /**
    * Return first tab which contains the specified object.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public TabProxy findTabProxy (Object obj)
   {
      TabProxy tab = null;

      if (obj != null)
      {
         Iterator iter = iteratorOfTabProxy();
         while ( (tab == null) && iter.hasNext())
         {
            TabProxy tmpTabProxy = (TabProxy) iter.next();

            JTree tree = tmpTabProxy.getTree();
            TreeModel model = tree.getModel();

            Enumeration enumeration =  ((DefaultMutableTreeNode) model.getRoot()).breadthFirstEnumeration();
            while (enumeration.hasMoreElements())
            {
               DefaultMutableTreeNode tmpNode = (DefaultMutableTreeNode) enumeration.nextElement();

               if (obj.equals (tmpNode.getUserObject()))
               {
                  tab = tmpTabProxy;
               }
            }
         }
      }

      return tab;
   } // findTabProxy


   /**
    * Reloads all tree models.
    */
   public void reloadTreeModels()
   {
      Iterator iter = iteratorOfTabProxy();
      while (iter.hasNext())
      {
         TabProxy tmpTab = (TabProxy) iter.next();
         JTree tree = tmpTab.getTree();

         if (tree != null)
         {
             ((DefaultTreeModel) tree.getModel()).reload();
         }
      }
   } // reloadTreeModels


   // --------------------------------------------------------------------------
   // INNER CLASSES
   // --------------------------------------------------------------------------

   /**
    * ChangeListener for tabbed pane.
    *
    * @author    $Author: trinet $
    * @version   $Revision: 1.37 $
    */
   private class TabbedPaneProxyListener implements ChangeListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void stateChanged (ChangeEvent e)
      {
         DefaultMutableTreeNode node = getFrameMain().getSelectedTreeNode();
         JTree tree = null;

         // get JTree on selected tab
         Component component = getTabbedPane().getSelectedComponent();
         if (component != null)
         {
            if (component instanceof JScrollPane)
            {
               component =  ((JScrollPane) component).getViewport().getView();
            }

            if (component instanceof JTree)
            {
               tree = (JTree) component;
            }
         }

         // clear and reset selection
         if (tree != null)
         {
            tree.clearSelection();
         }

         if (node != null)
         {
            Object obj = node.getUserObject();

            if (obj != null)
            {
               getFrameMain().selectTreeItem (obj);
            }
         }
         else
         {
            // select first node if nothing was selected yet
            if (tree != null)
            {
               tree.setSelectionRow (0);
            }
         }
      }
   } // TabbedPaneProxyListener

}

/*
 * $Log: TabbedPaneProxy.java,v $
 * Revision 1.37  2004/11/08 19:16:10  trinet
 * changed saving of FSAProperties, added String support for Maps in BasicIncrement, some cleanup
 *
 */
