/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.asg;

import java.awt.Point;
import java.beans.*;
import java.util.*;

import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

import de.tu_bs.coobra.*;
import de.tu_bs.xmlreflect.XMLReflect;
import de.uni_kassel.prop.InspectionAware;
import de.uni_kassel.prop.ObjectInspector;
import de.uni_paderborn.fujaba.basic.*;
import de.uni_paderborn.fujaba.codegen.OOGenToken;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.fsa.*;
import de.uni_paderborn.fujaba.fsa.swing.JCollapsable;
import de.uni_paderborn.fujaba.fsa.unparse.*;
import de.uni_paderborn.fujaba.messages.Message;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.upb.tools.fca.*;
import de.upb.tools.pcs.*;


/**
 * This is the base class for all model elements contained in the abstract syntax graph.
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *            0..n    hasElements    0..n
 * ASGElement --------------------------- ASGDiagram
 *            elements           diagrams
 *
 *            -------------- 0..1     hasElementReferences    0..1
 * ASGElement | getClass() |--------------------------------------- ASGElementRef
 *            -------------- element             elementReferences
 *
 *               ------- 0..n   annotations   0..n
 * ASGAnnotation | key |--------------------------- ASGElement
 *               ------- annotations      elements
 *
 *             0..1                        0..1
 * ASGElement ---------------------------------- OOGenToken
 *             asgElement       firstOOGenToken
 *
 *             0..1                           0..1
 * ASGElement ------------------------------------- OOGenToken
 *             lastUmlIncrement     lastOOGenToken
 * </pre>
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.112.2.7 $
 */
public abstract class ASGElement extends BasicIncrement implements LogicUnparseInterface, PropertyChangeClient, ObjectChangeAware, InspectionAware, FElement
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (ASGElement.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final transient boolean coobraPersistent;


   /**
    * Comparator for sorting ASGElements by name
    *
    * @author    $Author: fklar $
    * @version   $Revision: 1.112.2.7 $ $Date: 2005/11/30 12:05:53 $
    */
   public static class SortByNameComparator implements Comparator
   {
      /**
       * Compares its two ASGElements for ordering by name.
       *
       * @param o1                   the first object to be compared.
       * @param o2                   the second object to be compared.
       * @return                     a negative integer, zero, or a positive integer as the
       *         first argument is less than, equal to, or greater than the second.
       * @throws ClassCastException  if the arguments' types prevent them from being compared
       *                            by this Comparator.
       */
      public int compare (Object o1, Object o2)
      {
         ASGElement element1 = (ASGElement) o1;
         ASGElement element2 = (ASGElement) o2;
         if (element1.getName() != null)
         {
            return element1.getName().compareTo (element2.getName());
         }
         else if (element2.getName() != null)
         {
            return -element2.getName().compareTo (element1.getName());
         }
         else
         {
            return 0;
         }
      }
   }


   /**
    * If this attribute is set, every created ASGElement is created transient.
    */
   private static boolean inTransientMode = false;


   /**
    * Constructor. This constructor inspects the list of registered additional listeners and
    * adds them to the property change support. It also allows CoObRa to recognize creation.
    */
   public ASGElement()
   {
      this (!inTransientMode);
   }


   /**
    * Constructor for class ASGElement
    *
    * @param coobraPersistent  No description provided
    */
   protected ASGElement (boolean coobraPersistent)
   {
      this.coobraPersistent = coobraPersistent;

      if (!coobraPersistent)
      {
         //ignore object
         addToTransientElements();
      }
      else if (FujabaChangeManager.isInUndoRedo())
      {
         //ignore object
      }
      else
      {
         initPersistency();
      }

      // make sure the global element listener reacts on
      // property changes of this element
      // (the global element listener will be added to
      // 'additionalListeners' by this call)
      initElementListener();

      if (additionalListeners != null)
      {
         getPropertyChangeSupport();
      }
   }


   /**
    * Searches the ASG tree for a given id
    *
    * @param id  The id to search for.
    * @return    The Element with the given id, null if not found.
    */
   public ASGElement searchID (String id)
   {
      if (id.equals (getID()))
      {
         return this;
      }
      Iterator iter = iteratorOfElementReferences();
      ASGElement elem = null;
      while ( (elem == null) &&  (iter.hasNext()))
      {
         elem =  ((ASGElementRef) iter.next()).searchID (id);
      }
      return elem;
   }


   /**
    * Get the coobraPersistent attribute of the ASGElement object
    *
    * @return   The coobraPersistent value
    */
   public boolean isCoobraPersistent()
   {
      return this.coobraPersistent;
   }


   /**
    * @return       always false
    * @deprecated   for backward compatibility only
    */
   public boolean isReadOnly()
   {
      return false;
   }


   /**
    * <pre>
    *            0..n    hasElements    0..n
    * ASGElement --------------------------- ASGDiagram
    *            elements           diagrams
    * </pre>
    */
   private FLinkedList diagrams = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfDiagrams()
   {
      return  ( (diagrams == null)
         ? 0
         : diagrams.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diagram  No description provided
    * @return         No description provided
    */
   public boolean hasInDiagrams (FDiagram diagram)
   {
      return  (this.diagrams == null ? false : this.diagrams.contains (diagram));
   }
   // hasInDiagrams


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfDiagrams()
   {
      return  (this.diagrams == null
         ? FEmptyIterator.get()
         : this.diagrams.iterator());
   }


   /**
    * Access method for an one to n association.
    *
    * @param diagram  The object added.
    */
   public void addToDiagrams (FDiagram diagram)
   {
      if (diagram != null && !hasInDiagrams (diagram))
      {
         if (this.diagrams == null)
         {
            this.diagrams = new FPropLinkedList (this, DIAGRAMS_PROPERTY);
         }
         diagrams.add (diagram);
         diagram.addToElements (this);
      }
   }
   // addToDiagrams


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diagram  No description provided
    */
   public void removeFromDiagrams (FDiagram diagram)
   {
      if (this.hasInDiagrams (diagram))
      {
         this.diagrams.remove (diagram);
         diagram.removeFromElements (this);
         ASGDiagram asgDiagram = (ASGDiagram) diagram;
         ASGUnparseInformation unparseInformation = getFromUnparseInformations (asgDiagram);
         if (unparseInformation != null)
         {
            unparseInformation.removeYou();
         }
      }
   }
   // removeFromDiagrams


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromDiagrams()
   {
      ASGDiagram item;
      Iterator iter = iteratorOfDiagrams();

      while (iter.hasNext())
      {
         item = (ASGDiagram) iter.next();
         removeFromDiagrams (item);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return       No description provided
    * @deprecated   Use iteratorOfDiagrams instead
    */
   public Enumeration elementsOfDiagrams()
   {
      return new EnumerationForAnIterator (iteratorOfDiagrams());
   }


   /**
    * <pre>
    *            -------------- 0..1      hasReferences       0..1
    * ASGElement | getClass() |------------------------------------ ASGElementRef
    *            -------------- element          elementReferences
    * </pre>
    */
   private FHashMap elementReferences;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInElementReferences (FElementRef value)
   {
      return  ( (this.elementReferences != null) &&  (value != null) && this.elementReferences.containsValue (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInElementReferences (String key, FElementRef value)
   {
      return  ( (this.elementReferences != null) &&
          (value != null) &&
          (key != null) &&
          (this.elementReferences.get (key) == value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInElementReferences (String key)
   {
      return  ( (this.elementReferences != null) &&  (key != null) && this.elementReferences.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfElementReferences()
   {
      return  ( (this.elementReferences == null) ? FEmptyIterator.get() : this.elementReferences.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfElementReferences()
   {
      return  ( (this.elementReferences == null) ? FEmptyIterator.get() : this.elementReferences.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfElementReferences()
   {
      return  ( (this.elementReferences == null) ? FEmptyIterator.get() : this.elementReferences.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfElementReferences()
   {
      return  ( (this.elementReferences == null) ? 0 : this.elementReferences.size());
   }


   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromReferences attribute of the ASGElement object
    *
    * @param key  No description provided
    * @return     The fromReferences value
    */
   public ASGElementRef getFromElementReferences (String key)
   {
      return  ( ( (this.elementReferences == null) ||  (key == null)) ? null : (ASGElementRef) this.elementReferences.get (key));
   }


   /**
    * Get the fromFElementReferences attribute of the ASGElement object
    *
    * @param key  No description provided
    * @return     The fromFElementReferences value
    */
   public FElementRef getFromFElementReferences (String key)
   {
      return getFromElementReferences (key);
   }
   // TODO-END

   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToElementReferences (String key, FElementRef value)
   {
      boolean changed = false;
      if ( (value != null) &&  (key != null))
      {
         if (this.elementReferences == null)
         {
            this.elementReferences = new FPropHashMap (this, ELEMENT_REFERENCES_PROPERTY);
         }
         ASGElementRef oldValue = (ASGElementRef) this.elementReferences.put (key, value);
         if (oldValue != value)
         {
            if (oldValue != null)
            {
               oldValue.setElement (null, null);
            }
             ((ASGElementRef) value).setElement (key, this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * Access method for an one to n association.
    *
    * @param entry  The object added.
    * @return       No description provided
    */
   public boolean addToElementReferences (java.util.Map.Entry entry)
   {
      return addToElementReferences ((String) entry.getKey(), (ASGElementRef) entry.getValue());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromElementReferences (FElementRef value)
   {
      boolean changed = false;
      if ( (this.elementReferences != null) &&  (value != null))
      {
         Iterator iter = this.entriesOfElementReferences();
         Map.Entry entry;
         while (iter.hasNext())
         {
            entry = (Map.Entry) iter.next();
            if (entry.getValue() == value)
            {
               changed = changed || this.removeFromElementReferences ((String) entry.getKey(), value);
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromElementReferences (String key, FElementRef value)
   {
      boolean changed = false;
      if ( (this.elementReferences != null) &&  (value != null) &&  (key != null))
      {
         ASGElementRef oldValue = (ASGElementRef) this.elementReferences.get (key);
         if (oldValue == value)
         {
            this.elementReferences.remove (key);
            value.removeYou();
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromElementReferences (String key)
   {
      boolean changed = false;
      if ( (this.elementReferences != null) &&  (key != null))
      {
         ASGElementRef tmpValue = (ASGElementRef) this.elementReferences.get (key);
         if (tmpValue != null)
         {
            this.elementReferences.remove (key);
            tmpValue.setElement (null, null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromElementReferences()
   {
      Iterator iter = entriesOfElementReferences();
      Map.Entry entry;
      while (iter.hasNext())
      {
         entry = (Map.Entry) iter.next();
         removeFromElementReferences ((String) entry.getKey(), (ASGElementRef) entry.getValue());
      }
   }


   /**
    * <pre>
    *               ------- 0..n   Annotations   0..n
    * ASGAnnotation | key |--------------------------- ASGElement
    *               ------- annotations      elements
    * </pre>
    */
   private FHashSet annotations;


   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToAnnotations (String key, FAnnotation value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.annotations == null)
         {
            this.annotations = new FHashSet();
         }
         changed = this.annotations.add (value);
         if (changed)
         {
            value.addToElements (key, this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAnnotations()
   {
      return  ( (this.annotations == null)
         ? FEmptyIterator.get()
         : this.annotations.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInAnnotations (FAnnotation value)
   {
      return  ( (this.annotations != null) &&
          (value != null) &&
         this.annotations.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfAnnotations()
   {
      return  ( (this.annotations == null)
         ? 0
         : this.annotations.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromAnnotations (String key, FAnnotation value)
   {
      boolean changed = false;
      if ( (this.annotations != null) &&  (value != null))
      {
         changed = this.annotations.remove (value);
         if (changed)
         {
            value.removeFromElements (key, this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromAnnotations()
   {
      ASGAnnotation tmpValue;
      Iterator iter = this.iteratorOfAnnotations();
      while (iter.hasNext())
      {
         tmpValue = (ASGAnnotation) iter.next();
         tmpValue.removeFromElements (this);
      }
   }


   /**
    * <pre>
    *             0..1                        0..1
    * ASGElement ---------------------------------- OOGenToken
    *             asgElement       firstOOGenToken
    * </pre> token for codegen directory
    */
   private transient OOGenToken firstOOGenToken;


   /**
    * Sets the firstOOGenToken attribute of the ASGElement object
    *
    * @param value  The new firstOOGenToken value
    * @return       No description provided
    */
   public boolean setFirstOOGenToken (OOGenToken value)
   {
      if (this.firstOOGenToken != value)
      {
         if (this.firstOOGenToken != null)
         {
            OOGenToken oldValue = this.firstOOGenToken;
            this.firstOOGenToken = null;
            oldValue.setFirstAsgElement (null);
         }
         this.firstOOGenToken = value;
         if (value != null)
         {
            this.firstOOGenToken.setFirstAsgElement (this);
         }

         return true;
      }

      return false;
   }


   /**
    * Get the firstOOGenToken attribute of the ASGElement object
    *
    * @return   The firstOOGenToken value
    */
   public OOGenToken getFirstOOGenToken()
   {
      return this.firstOOGenToken;
   }


   /**
    * <pre>
    *             0..1                           0..1
    * ASGElement ------------------------------------- OOGenToken
    *             lastUmlIncrement     lastOOGenToken
    * </pre>
    */
   private transient OOGenToken lastOOGenToken;


   /**
    * Sets the lastOOGenToken attribute of the ASGElement object
    *
    * @param value  The new lastOOGenToken value
    * @return       No description provided
    */
   public boolean setLastOOGenToken (OOGenToken value)
   {
      if (this.lastOOGenToken != value)
      {
         if (this.lastOOGenToken != null)
         {
            OOGenToken oldValue = this.lastOOGenToken;
            this.lastOOGenToken = null;
            oldValue.setLastAsgElement (null);
         }
         this.lastOOGenToken = value;
         if (value != null)
         {
            this.lastOOGenToken.setLastAsgElement (this);
         }

         return true;
      }

      return false;
   }


   /**
    * Get the lastOOGenToken attribute of the ASGElement object
    *
    * @return   The lastOOGenToken value
    */
   public OOGenToken getLastOOGenToken()
   {
      return this.lastOOGenToken;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void deleteTokens()
   {
      OOGenToken current = getFirstOOGenToken();
      OOGenToken lastToken = getLastOOGenToken();
      OOGenToken oldToken;
      while (current != null && current != lastToken)
      {
         oldToken = current;
         current = current.getNext();
         oldToken.removeYouFromList();
      }
      if (current != null)
      {
         current.removeYouFromList();
      }
      setFirstOOGenToken (null);
      setLastOOGenToken (null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient FSAInterface fsaInterface = null;


   /**
    * Get the fSAInterface attribute of the ASGElement object
    *
    * @return   The fSAInterface value
    */
   public synchronized FSAInterface getFSAInterface()
   {
      if (this.fsaInterface == null)
      {
         this.fsaInterface = new FSAInterface (this);
      }
      return this.fsaInterface;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient String unparseModuleName = null;


   /**
    * Get the unparseModuleNameImpl attribute of the ASGElement object
    *
    * @return   The unparseModuleNameImpl value
    */
   protected final String getUnparseModuleNameImpl()
   {
      if (this.unparseModuleName != null)
      {
         return this.unparseModuleName;
      }
      return getClass().getName();
   }


   /**
    * Sets the unparseModuleName attribute of the ASGElement object
    *
    * @param name  The new unparseModuleName value
    */
   protected void setUnparseModuleName (String name)
   {
      this.unparseModuleName = name;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected String createUnparseModuleName()
   {
      return UnparseManager.get().getUnparseModuleName (this);
   }


   /**
    * Get the unparseModuleName attribute of the ASGElement object
    *
    * @return   The unparseModuleName value
    */
   public final String getUnparseModuleName()
   {
      if (this.unparseModuleName == null)
      {
         this.unparseModuleName = createUnparseModuleName();
      }
      return this.unparseModuleName;
   }


   /**
    * <pre>
    *            +---------------------+ 0..1   swingAdapter   0..1
    * ASGElement | getQualifiedName () |---------------------------- FSAObject
    *            +---------------------+ umlIncr         fsaObjects
    * </pre> Where qualifiedName is DiagramName.PropertyName
    *
    * @param elem  The object added.
    */
   public void addToFsaObjects (FSAObject elem)
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         fsaIface.addToFsaObjects (elem);
      }
   }


   /**
    * Get the fromFsaObjects attribute of the ASGElement object
    *
    * @param qualifiedName  No description provided
    * @return               The fromFsaObjects value
    */
   public FSAObject getFromFsaObjects (String qualifiedName)
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         return fsaIface.getFromFsaObjects (qualifiedName);
      }
      return null;
   }


   /**
    * Get the firstFromFSAObjects attribute of the ASGElement object
    *
    * @return   The firstFromFSAObjects value
    */
   public FSAObject getFirstFromFSAObjects()
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         Iterator iter = fsaIface.iteratorOfFsaObjects();
         if (!iter.hasNext())
         {
            return null;
         }
         return (FSAObject) iter.next();
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfFsaObjects()
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         return fsaIface.sizeOfFsaObjects();
      }
      return 0;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInFsaObjects (FSAObject elem)
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         return fsaIface.hasInFsaObjects (elem);
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param qualifiedName  No description provided
    * @return               No description provided
    */
   public boolean hasKeyInFsaObjects (String qualifiedName)
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         return fsaIface.hasKeyInFsaObjects (qualifiedName);
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfFsaObjects()
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         return fsaIface.iteratorOfFsaObjects();
      }
      return FEmptyIterator.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfFsaObjects()
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         return fsaIface.keysOfFsaObjects();
      }
      return FEmptyIterator.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfFsaObjects()
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         return fsaIface.entriesOfFsaObjects();
      }
      return FEmptyIterator.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromFsaObjects (FSAObject elem)
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         fsaIface.removeFromFsaObjects (elem);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param qualifiedName  No description provided
    */
   public void removeKeyFromFsaObjects (String qualifiedName)
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         fsaIface.removeKeyFromFsaObjects (qualifiedName);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromFsaObjects()
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         fsaIface.removeAllFromFsaObjects();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param oldKey  No description provided
    * @param object  No description provided
    */
   public void updateKeyInFsaObjects (String oldKey, FSAObject object)
   {
      FSAInterface fsaIface = getFSAInterface();
      if (fsaIface != null)
      {
         fsaIface.updateKeyInFsaObjects (oldKey, object);
      }
   }


   /**
    * remains here for backward compatibility (load old fprs)
    */
   private transient Map locations = new FHashMap();

   /**
    * map from ASGElement (parent key) to ASGUnparseInformation
    */
   private FHashMap unparseInformations;


   /**
    * @param key    to be removed
    * @param value  information
    * @return       true when something was changed
    * @see          #addToUnparseInformations(ASGElement, ASGUnparseInformation)
    */
   public boolean removeFromUnparseInformations (ASGElement key, ASGUnparseInformation value)
   {
      boolean changed = false;
      if ( (this.unparseInformations != null) &&  (value != null) &&  (key != null))
      {

         ASGUnparseInformation oldValue = (ASGUnparseInformation) this.unparseInformations.remove (key);
         if (oldValue != null)
         {
            value.setASGElement (null, null);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * @param key  key to be removed
    * @return     true when something was changed
    * @see        #addToUnparseInformations(ASGElement, ASGUnparseInformation)
    */
   public boolean removeFromUnparseInformations (ASGElement key)
   {
      boolean changed = false;
      if ( (this.unparseInformations != null) &&  (key != null))
      {
         ASGUnparseInformation oldValue = (ASGUnparseInformation) this.unparseInformations.remove (key);
         if (oldValue != null)
         {
            oldValue.setASGElement (null, null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * @param value  information to be removed
    * @return       true when something was changed
    * @see          #addToUnparseInformations(ASGElement, ASGUnparseInformation)
    */
   public boolean removeFromUnparseInformations (ASGUnparseInformation value)
   {
      boolean changed = false;
      if ( (this.unparseInformations != null) &&  (value != null))
      {
         Iterator iter = this.entriesOfUnparseInformations();
         Map.Entry entry;
         while (iter.hasNext())
         {
            entry = (Map.Entry) iter.next();
            if (entry.getValue() == value)
            {
               changed = changed || this.removeFromUnparseInformations ((ASGElement) entry.getKey(), value);
            }
         }
      }
      return changed;
   }


   /**
    * remove all
    *
    * @see   #addToUnparseInformations(ASGElement, ASGUnparseInformation)
    */
   public void removeAllFromUnparseInformations()
   {
      Iterator iter = entriesOfUnparseInformations();
      Map.Entry entry;
      while (iter.hasNext())
      {
         entry = (Map.Entry) iter.next();
         ASGUnparseInformation unparseInfo = (ASGUnparseInformation) entry.getValue();
         removeFromUnparseInformations ((ASGElement) entry.getKey(), unparseInfo);
         unparseInfo.removeYou();
      }
   }


   /**
    * @return   No description provided
    * @see      #addToUnparseInformations(ASGElement, ASGUnparseInformation)
    */
   public Iterator keysOfUnparseInformations()
   {
      return  ( (this.unparseInformations == null)
         ? FEmptyIterator.get()
         : this.unparseInformations.keySet().iterator());
   }


   /**
    * @return   No description provided
    * @see      #addToUnparseInformations(ASGElement, ASGUnparseInformation)
    */
   public Iterator entriesOfUnparseInformations()
   {
      return  ( (this.unparseInformations == null)
         ? FEmptyIterator.get()
         : this.unparseInformations.entrySet().iterator());
   }


   /**
    * add an information about unparsing of this ASGElement
    *
    * @param key    parent of the ASGElement regarding this unparse information
    * @param value  information
    * @return       true when information was added
    */
   public boolean addToUnparseInformations (ASGElement key, ASGUnparseInformation value)
   {
      boolean changed = false;
      if ( (value != null) &&  (key != null))
      {
         if (this.unparseInformations == null)
         {
            this.unparseInformations = new FPropHashMap (this, "unparseInformations");
         }
         ASGUnparseInformation oldValue = (ASGUnparseInformation) this.unparseInformations.get (key);
         if (oldValue != value)
         {
            this.unparseInformations.put (key, value);

            if (oldValue != null)
            {
               oldValue.setASGElement (null, null);
               //nobody will miss this
               oldValue.removeYou();
            }
            value.setASGElement (key, this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * @param entry  what to add
    * @return       true when entry was added
    * @see          #addToUnparseInformations(ASGElement, ASGUnparseInformation)
    */
   public boolean addToUnparseInformations (java.util.Map.Entry entry)
   {
      return addToUnparseInformations ((ASGElement) entry.getKey(), (ASGUnparseInformation) entry.getValue());
   }


   /**
    * @return   iterator through all keys (parents) in uparseInformations
    */
   public Iterator iteratorOfKeyFromUnparseInformations()
   {
      if (unparseInformations != null)
      {
         return unparseInformations.keySet().iterator();
      }
      else
      {
         return FEmptyIterator.get();
      }
   }


   /**
    * @return   iterator through all entries in uparseInformations
    */
   public Iterator iteratorOfUnparseInformations()
   {
      if (unparseInformations != null)
      {
         return unparseInformations.entrySet().iterator();
      }
      else
      {
         return FEmptyIterator.get();
      }
   }


   /**
    * @param key  parent
    * @return     unparse information
    */
   public ASGUnparseInformation getFromUnparseInformations (ASGElement key)
   {
      if (unparseInformations != null)
      {
         return (ASGUnparseInformation) unparseInformations.get (key);
      }
      else
      {
         return null;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void removeObsoleteUnparseInformation()
   {
      UMLProject project = UMLProject.get();

      Iterator iter = iteratorOfKeyFromUnparseInformations();
      while (iter.hasNext())
      {
         // remove unparse information for diagrams that are not in the project
         ASGElement asgElement = (ASGElement) iter.next();
         if (asgElement instanceof ASGDiagram)
         {
            ASGDiagram asgDiagram = (ASGDiagram) asgElement;
            if (!project.hasInDiags (asgDiagram))
            {
               ASGUnparseInformation unparseInformation = getFromUnparseInformations (asgDiagram);
               if (unparseInformation != null)
               {
                  unparseInformation.removeYou();
               }
               continue;
            }
         }

         //remove empty ASGUnparseInformation (for old projects)
         ASGUnparseInformation unparseInformation = getFromUnparseInformations (asgElement);
         if (unparseInformation.sizeOfASGInformation() == 0)
         {
            unparseInformation.removeYou();
         }
      }
   }


   /**
    * save locations of FSAs
    *
    * @param removeOldLocations  unused
    */
   public void saveLocations (boolean removeOldLocations)
   {
      Iterator iter = iteratorOfFsaObjects();
      while (iter.hasNext())
      {
         FSAObject fsa = (FSAObject) iter.next();
         fsa.saveLocation();
      }
   }


   /**
    * save properties of FSAs
    */
   public void saveFSAProperties()
   {
      Iterator iter = iteratorOfFsaObjects();
      while (iter.hasNext())
      {
         FSAObject fsa = (FSAObject) iter.next();
         fsa.saveFSAProperties();
      }
   }


   /**
    * moves all FSAObject with the specified propertyName and logical parent to the location
    *
    * @param parent        logical parent of the moved fsa
    * @param propertyName  which objects to move
    * @param location      where to move
    */
   public void updateFSALocation (ASGElement parent, String propertyName, Point location)
   {
      if (getFSAInterface() != null && propertyName != null)
      {
         Iterator it = getFSAInterface().iteratorOfFsaObjects();
         while (it.hasNext())
         {
            FSAObject obj = (FSAObject) it.next();
            if (obj.getFSAQualifier() == parent && propertyName.equals (obj.getPropertyName()))
            {
               obj.getJComponent().setLocation (location);
            }
         }
      }
   }


   /**
    * un/collapse all FSAObject with the specified propertyName and logical parent
    *
    * @param parent        logical parent of the moved fsa
    * @param propertyName  which objects to move
    * @param collapsed     to collapse or not to collapse
    */
   public void updateFSACollapsed (ASGElement parent, String propertyName, boolean collapsed)
   {
      if (getFSAInterface() != null && propertyName != null)
      {
         Iterator it = getFSAInterface().iteratorOfFsaObjects();
         while (it.hasNext())
         {
            FSAObject obj = (FSAObject) it.next();
            if (obj.getFSAQualifier() == parent && propertyName.equals (obj.getPropertyName()))
            {
                ((JCollapsable) obj.getJComponent()).setCollapsed (collapsed);
            }
         }
      }
   }


   /**
    * This method is overridden to save the location of all associated FSAObjects in a hashtable.
    *
    * @param data             No description provided
    * @param setOfNeighbours  No description provided
    */
   public void writeAttributes (StringBuffer data, FTreeSet setOfNeighbours)
   {
      removeObsoleteUnparseInformation();
      saveFSAProperties();
      super.writeAttributes (data, setOfNeighbours);
   }


   /**
    * keep for loading old fprs
    *
    * @param entry  The object added.
    * @deprecated   use {@link #addToUnparseInformations(ASGElement, ASGUnparseInformation)}
    */
   public void addToLocations (Map.Entry entry)
   {
      String qualifier = (String) entry.getKey();
      PointIncrement incr = (PointIncrement) entry.getValue();

      addToLocations (qualifier, incr);
   }


   /**
    * add a point information of a specific parent property
    *
    * @param parent    key for the unparseInfoamtion
    * @param property  property affected property (key)
    * @param point     new value
    */
   public void addPointToUnparseInformation (ASGElement parent, String property,
                                             Point point)
   {
      if (point != null && parent != null)
      {
         ASGUnparseInformation unparseInformation = getFromUnparseInformations (parent);
         if (unparseInformation == null)
         {
            unparseInformation = new ASGUnparseInformation();
            addToUnparseInformations (parent, unparseInformation);
         }

         ASGInformation info = unparseInformation.getFromASGInformation (property);
         if (info == null)
         {
            info = new ASGInformation();
            unparseInformation.addToASGInformation (property, info);
         }

         info.addToInformation (FSAObject.LOCATION + "_X", Integer.toString (point.x));
         info.addToInformation (FSAObject.LOCATION + "_Y", Integer.toString (point.y));
      }
   }


   /**
    * Convenience method for getUnparseInformation.
    *
    * @param parent    regarding parent
    * @param property  affected property (key)
    * @return          the position of this ASGElement property
    */
   public Point getPointFromUnparseInformation (ASGElement parent, String property)
   {
      ASGUnparseInformation unparseInformation = getFromUnparseInformations (parent);
      if (unparseInformation != null)
      {
         ASGInformation info = unparseInformation.getFromASGInformation (property);
         if (info != null)
         {
            String xLoc = info.getFromInformation (FSAObject.LOCATION + "_X");
            String yLoc = info.getFromInformation (FSAObject.LOCATION + "_Y");

            if (xLoc != null && yLoc != null)
            {
               try
               {
                  return new Point (Integer.parseInt (xLoc), Integer.parseInt (yLoc));
               }
               catch (NumberFormatException e)
               {
                  e.printStackTrace();
                  return null;
               }
            }
         }
      }

      return null;
   }


   /**
    * remains for loading old fprs
    *
    * @param qualifier  The object added.
    * @param incr       The object added.
    * @deprecated       use {@link #addToUnparseInformations(ASGElement, ASGUnparseInformation)}
    */
   public void addToLocations (String qualifier, PointIncrement incr)
   {
      if (qualifier != null)
      {
         String id = null;
         String property = qualifier;
         if (qualifier.indexOf ('.') >= 0)
         {
            property = qualifier.substring (qualifier.indexOf ('.') + 1);
            id = qualifier.substring (0, qualifier.indexOf ('.'));
         }
         ASGElement parent;
         if (UMLProject.isLoading())
         {
            parent = (ASGElement) UMLProject.getFromObjectHashTable (id);
         }
         else
         {
            parent = UMLProject.get().searchID (id);
         }
         if (parent != null)
         {
            incr.notifyUponChange (this, parent, property);
            addPointToUnparseInformation (parent, property, incr.getPoint());
         }
         else
         {
            log.error ("could not find ASGElement(" + id + ") for adding ASGUnparseInforamtion: " + qualifier);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient FujabaPropertyChangeSupport propertyChangeSupport = null;


   /**
    * Get the propertyChangeSupport attribute of the ASGElement object
    *
    * @return   The propertyChangeSupport value
    */
   public PropertyChangeSupport getPropertyChangeSupport()
   {
      if (propertyChangeSupport == null)
      {
         propertyChangeSupport = new FujabaPropertyChangeSupport (this);

         if (additionalListeners != null)
         {

            Iterator iter = additionalListeners.iterator();
            while (iter.hasNext())
            {
               PropertyChangeListener listener = (PropertyChangeListener) iter.next();
               addPropertyChangeListener (listener);
            }

         }
      }
      return propertyChangeSupport;
   }


   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addToPropertyChangeListeners (PropertyChangeListener listener)
   {
      addPropertyChangeListener (listener);
   }


   /**
    * Access method for an one to n association.
    *
    * @param propertyName  The object added.
    * @param listener      The object added.
    */
   public void addToPropertyChangeListeners (String propertyName, PropertyChangeListener listener)
   {
      addPropertyChangeListener (propertyName, listener);
   }


   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addPropertyChangeListener (PropertyChangeListener listener)
   {
      getPropertyChangeSupport().addPropertyChangeListener (listener);
   }


   /**
    * Access method for an one to n association.
    *
    * @param propertyName  The object added.
    * @param listener      The object added.
    */
   public void addPropertyChangeListener (String propertyName, PropertyChangeListener listener)
   {
      getPropertyChangeSupport().addPropertyChangeListener (propertyName, listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    */
   public void removeFromPropertyChangeListeners (PropertyChangeListener listener)
   {
      removePropertyChangeListener (listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param propertyName  No description provided
    * @param listener      No description provided
    */
   public void removeFromPropertyChangeListeners (String propertyName, PropertyChangeListener listener)
   {
      removePropertyChangeListener (propertyName, listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    */
   public void removePropertyChangeListener (PropertyChangeListener listener)
   {
      if (propertyChangeSupport != null)
      {
         propertyChangeSupport.removePropertyChangeListener (listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param propertyName  No description provided
    * @param listener      No description provided
    */
   public void removePropertyChangeListener (String propertyName, PropertyChangeListener listener)
   {
      if (propertyChangeSupport != null)
      {
         propertyChangeSupport.removePropertyChangeListener (propertyName, listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    */
   public void fireEvent (PropertyChangeEvent event)
   {
      firePropertyChange (event);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static Logger logger = Logger.getLogger (ASGElement.class);


   /**
    * fires property change events via the property change support <br>
    * (warns when to CoObRA persistency listener is subscribed)
    *
    * @param e  what to be fired
    */
   protected void firePropertyChange (PropertyChangeEvent e)
   {
      if (logger.isEnabledFor (Priority.WARN))
      {
         if (!getClass().getName().startsWith ("de.uni_paderborn.dobs") && FujabaChangeManager.getVMRepository() != null)
         {
            if (persistencyListener != null)
            {
               if ( (propertyChangeSupport == null
                  || !propertyChangeSupport.hasInAllListeners (persistencyListener))
                  && !FujabaChangeManager.isInUndoRedo())
               {
                  logger.warn ("firing " + this.getClass().getName() + "." + e.getPropertyName()
                     + " but the persistency listener is not subscribed");
               }
            }
            else
            {
               logger.warn ("firing " + e + " but object has no persitency listener");
            }
         }
      }
      if (propertyChangeSupport != null)
      {
         propertyChangeSupport.firePropertyChange (e);
      }
   }


   /**
    * CoObRA: stores repository of this object
    */
   private LocalRepository repository;


   /**
    * CoObRA: Obtain the repository of this object
    *
    * @return   the repository this object resides in
    */
   public final LocalRepository getRepository()
   {
      return repository;
   }


   /**
    * CoObRA: Changes the repository this object resides in
    *
    * @param value  The new repository value
    * @return       true when repository was changed
    */
   public boolean setRepository (LocalRepository value)
   {
      if (XMLReflect.getFromSingletons (getClass()) != null)
      {
         //ignore singleton instances for subclasses of ASGElement
         return false;
      }
      boolean changed = false;
      final LocalRepository oldRepository = this.repository;
      if (oldRepository != null && value == null)
      {
         ObjectChange change = ObjectChangeAwareHelper.fireChange (this, null, ObjectChange.CHANGE_TYPE_REMOVE_OBJECT, this, null, null);
         acknowledgeChange (change);
         LocalRepository oldValue = oldRepository;
         //this.repository = null;
         oldValue.removeFromKnownObjects (this);
         oldValue.removeFromKnownIds (this.getCoObRAId());
      }
      if ( (oldRepository == null && value != null) || oldRepository == value)
      {
         this.repository = value;
         if (value != null)
         {
            repository.addToKnownObjects (this);
            if (this.getCoObRAId() != null)
            {
               value.addToKnownIds (value.getIdForObject (this), this);
            }
            ObjectChangeAwareHelper.fireChange (this, null, ObjectChange.CHANGE_TYPE_ADD_OBJECT, null, this.getClass(), null);
         }
      }
      else if (value != null)
      {
         throw new RuntimeException ("repository may not be changed!");
      }
      changed = true;
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, Object oldValue, Object newValue)
   {
      if (oldValue == newValue)
      {
         return;
      }

      firePropertyChange (new PropertyChangeEvent (this, name, oldValue, newValue));
   }


   /**
    * CoObRA: This method is called to notify the object, that changes to its fields have been
    * made.
    *
    * @param change  the change that has occured, may be null to indicate an unknown change
    *               or to summarize multiple changes
    */
   public void acknowledgeChange (ObjectChange change)
   {
   }


   /**
    * persistency id for coobra
    */
   private transient String coobraId;


   /**
    * CoObRA: Attention - Getter for CoObRA-ID!
    *
    * @return   the id of the object.
    */
   public String getCoObRAId()
   {
      return coobraId;
   }


   /**
    * CoObRA: Setter for CoObRA-ID. May only be called by LocalRepository!
    *
    * @param coobraId  New value of id.
    */
   public void setCoObRAId (String coobraId)
   {
      this.coobraId = coobraId;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, boolean oldValue, boolean newValue)
   {
      if (oldValue == newValue || propertyChangeSupport == null)
      {
         return;
      }

      firePropertyChange (name, Boolean.valueOf (oldValue), Boolean.valueOf (newValue));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, int oldValue, int newValue)
   {
      if (oldValue == newValue)
      {
         return;
      }
      firePropertyChange (name, new Integer (oldValue), new Integer (newValue));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, double oldValue, double newValue)
   {
      if (oldValue == newValue || propertyChangeSupport == null)
      {
         return;
      }
      firePropertyChange (name, new Double (oldValue), new Double (newValue));
   }

   // --------------------------------------------------------------------
   // Additional property change support for plugins.
   // --------------------------------------------------------------------

   /**
    * Set of additional listeners.
    */
   private static volatile HashSet additionalListeners;


   /**
    * Adds an additional property change listener. If, e.g. a plugin wants to be notified about
    * changes in the ASG, it has to register a listener at startup time. Each time a new <code>ASGElement</code>
    * is created, all registered listeners are added to the propertyChangeSupport of this element.
    *
    * @param listener
    */
   public static void addAdditionalListener (PropertyChangeListener listener)
   {
      if (listener != null)
      {

         if (additionalListeners == null)
         {
            additionalListeners = new HashSet();
         }

         // just add once!
         if (!additionalListeners.contains (listener))
         {
            additionalListeners.add (listener);
         }
      }
   }


   /**
    * Get the persistencyChange attribute of the ASGElement object
    *
    * @param e  No description provided
    * @return   The persistencyChange value
    */
   protected boolean isPersistencyChange (PropertyChangeEvent e)
   {
      if ("userMessages".equals (e.getPropertyName()))
      {
         return false; // ignore userMessages (transient)
      }
      if (!this.isCoobraPersistent())
      {
         return false;
      }
      return true;
   }


   /**
    * Instance that listens on all changes of ASGElement objects.
    * Only one listener will be created for the environment.
    */
   private static volatile ASGElementListener elementListener;


   /**
    * Lets the element listener react on property changes
    * of the calling ASGElement.
    */
   protected void initElementListener()
   {
      if (elementListener == null)
      {
         elementListener = new ASGElementListener();
      }
      addAdditionalListener (elementListener);
   }


   /**
    * Listener that evaluates property changes of ASGElement objects.
    * If a property changes, the corresponding project must be saved.
    *
    * @author    $Author: fklar $
    * @version   $Revision: 1.112.2.7 $ $Date: 2005/11/30 12:05:53 $
    */
   private static class ASGElementListener implements PropertyChangeListener
   {

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param evt  No description provided
       */
      public void propertyChange (PropertyChangeEvent evt)
      {
         Object source = evt.getSource();
         if (! (source instanceof ASGElement))
         {
            return;
         }

         // at this moment we only check if the property realy changes
         // and if so, we inform the project in which the element is placed
         // to set its state to 'unchanged'.
         Object oldValue = evt.getOldValue();
         Object newValue = evt.getNewValue();

         if (oldValue == newValue)
         {
            return;
         }

         UMLProject project = UMLProject.get();

         // if the project is currently loading, we do not inform the project
         if (UMLProject.isLoading())
         {
            return;
         }

         String propertyName = evt.getPropertyName();

         if (project == null)
         {
            return;
         }

         if (project == source && "saved".equals (propertyName))
         {
            return;
         }

         // TODO: check some more cases, in which the project does not need
         // to be safed.
         // i.e., "well known" transient properties or properties
         // that don't make sense to be safed

         // now we can be sure that the project has been changed
         // in such a way, that it needs to be safed
         project.setSaved (false);
      }
   }


   /**
    * CoObRA: Listener to recognize all changes to ASGElements (including collection based)
    *
    * @author    $Author: fklar $
    * @version   $Revision: 1.112.2.7 $
    */
   private static class PersistencyListener implements PropertyChangeListener, PopupSourceListener
   {
      /**
       * Called upon all changes to ASGElements
       *
       * @param e  the change event
       */
      public void propertyChange (PropertyChangeEvent e)
      {
         Object source = e.getSource();
         if (source instanceof ASGElement)
         {
            if (! ((ASGElement) source).isPersistencyChange (e))
            {
               return;
            }
         }

         Object oldValue = e.getOldValue();
         if (oldValue instanceof ASGElement)
         {
            ASGElement oldAsg = (ASGElement) oldValue;
            if (!oldAsg.isCoobraPersistent())
            {
               oldValue = null;
            }
         }

         Object newValue = e.getNewValue();
         if (newValue instanceof ASGElement)
         {
            ASGElement newAsg = (ASGElement) newValue;
            if (!newAsg.isCoobraPersistent())
            {
               newValue = null;
            }
         }

         if (oldValue != newValue)
         {
            int changeType = ObjectChange.CHANGE_TYPE_ALTER;
            Object key = null;

            if (!UMLProject.isLoading() && UMLProject.get() != null)
            {
               FujabaChangeManager.updateUndoRedoActions();
            }

            if ("removeYou".equals (e.getPropertyName()))
            {
               changeType = ObjectChange.CHANGE_TYPE_REMOVE_OBJECT;
            }
            else if (e instanceof CollectionChangeEvent)
            {
               CollectionChangeEvent ce = (CollectionChangeEvent) e;
               if (ce.getType() == CollectionChangeEvent.ADDED ||
                  ce.getType() == CollectionChangeEvent.ADDED_BEFORE ||
                  ce.getType() == CollectionChangeEvent.ADDED_AFTER)
               {
                  changeType = ObjectChange.CHANGE_TYPE_ADD;
               }
               else if (ce.getType() == CollectionChangeEvent.REMOVED ||
                  ce.getType() == CollectionChangeEvent.REMOVED_AFTER)
               {
                  changeType = ObjectChange.CHANGE_TYPE_REMOVE;
               }
               else if (ce.getType() == CollectionChangeEvent.CHANGED)
               {
                  changeType = ObjectChange.CHANGE_TYPE_ALTER;
               }
               else
               {
                  new RuntimeException ("Unknown CollectionChangeEvent type: " + ce.getType()).printStackTrace();
               }

               if (e instanceof MapChangeEvent)
               {
                  key =  ((MapChangeEvent) e).getKey();
               }
               else if (e instanceof ListChangeEvent)
               {
                  key =  ((ListChangeEvent) e).getPos();
               }
            }

            if (changeType != ObjectChange.CHANGE_TYPE_REMOVE_OBJECT)
            {
               ObjectChange change = ObjectChangeAwareHelper.preSetter ((ASGElement) e.getSource(), e.getPropertyName()
                  , changeType, oldValue, newValue, key);
               ObjectChangeAwareHelper.postSetter (change);
            }
         }
      }


      /**
       * store the locations whenever the popupSource changes
       *
       * @param newSource
       */
      public void popupSourceChanged (LogicUnparseInterface newSource)
      {
         // doesn't seem to be required any more,
         // because saving of location is done in mouse-listeners

//         ObjectChangeCause saveLocationsCause = new ObjectChangeStringCause ("move");
//         ObjectChange.pushCause (saveLocationsCause);
//         try
//         {
//            if (newSource != null)
//            {
//               LogicUnparseInterface unparse = newSource.getFSAInterface().getLogic();
//               if (unparse instanceof ASGElement)
//               {
//                  ASGElement element = (ASGElement) unparse;
//                  element.saveFSAProperties();
//               }
//            }
//         }
//         finally
//         {
//            ObjectChange.popCause (saveLocationsCause);
//         }
      }
   }


   /**
    * for listening to all changes and new selection
    */
   private static volatile PersistencyListener persistencyListener;


   /**
    * enables the persistencyListener when class is loaded
    */
   protected void initPersistency()
   {
      if (persistencyListener == null)
      {
         // the following code will only be called, if
         // the user has activated the repository in the
         // preference settings
         if (FujabaChangeManager.getVMRepository() != null)
         {
            persistencyListener = new PersistencyListener();
            LocalRepository.IGNORE_CHANGES_TO_REMOVED_OBJECTS = true;
            SelectionManager.get().addToPopupSourceListeners (persistencyListener);

            //override CoObRAs valueEqual() method to ignore changed ids
            ObjectChangeUtils.set (
               new ObjectChangeUtils()
               {
                  public boolean valuesEqual (Object value1, Object value2)
                  {
                     if (value1 instanceof String || value2 instanceof String)
                     {
                        if (value2 == null || value1 == null)
                        {
                           String string;
                           if (value2 == null)
                           {
                              string = (String) value1;
                           }
                           else
                           {
                              string = (String) value2;
                           }

                           //ignore changed name from null to id
                           if (string.startsWith ("_@id"))
                           {
                              return true;
                           }

                           //ignore changed statement from null to "//statement"
                           if (string.trim().equals ("//Statement"))
                           {
                              return true;
                           }

                           //ignore changed name from null to ""
                           if (string.equals (""))
                           {
                              return true;
                           }
                        }
                        else if (value1 instanceof String && value2 instanceof String)
                        {
                           String string1 = (String) value1;
                           String string2 = (String) value2;

                           //ignore changed name from id to other id
                           if (string1.startsWith ("_@id") && string2.startsWith ("_@id"))
                           {
                              return true;
                           }
                        }
                     }
                     else if (value1 instanceof FPackage && value2 instanceof FPackage)
                     {
                        //todo: RootPackage equals RootPackage here - second RootPackage should be avoided totally!
                        if (UMLProject.get().getRootPackage().getName().equals ( ((FPackage) value1).getName()) &&
                           UMLProject.get().getRootPackage().getName().equals ( ((FPackage) value2).getName())
                           )
                        {
                           return true;
                        }
                     }
                     return super.valuesEqual (value1, value2);
                  }
               });
         }
      }
      addAdditionalListener (persistencyListener);

      //exclude DOBS Objects from changes recognition
      if (!getClass().getName().startsWith ("de.uni_paderborn.dobs") && persistencyListener != null)
      {
         setRepository (FujabaChangeManager.getVMRepository());
         FujabaChangeManager.addActionCauseListener();
      }
   }


   /**
    * Removes an additional property change listener.
    *
    * @param listener
    */
   public static void removeAdditionalListener (PropertyChangeListener listener)
   {
      if (listener != null)
      {
         if (additionalListeners != null)
         {
            additionalListeners.remove (listener);
         }
      }
   }


   /**
    * Get the name attribute of the ASGElement object
    *
    * @return   The name value
    */
   public String getName()
   {
      return "no Name";
   }


   /**
    * Sets the name attribute of the ASGElement object
    *
    * @param newName  The new name value
    */
   public void setName (String newName)
   {
      throw new RuntimeException ("You have called 'setName (" + newName +
         ")' of class 'ASGElement'.\n" +
         "Please overwrite it in your derived class '" + this.getClass() + "'");
   }


   /**
    * Get the text attribute of the ASGElement object
    *
    * @return   The text value
    */
   public String getText()
   {
      return "";
   }


   //***************************************************************************************

   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      this.deleteTokens();
      this.setFirstOOGenToken (null);
      this.setLastOOGenToken (null);

      if (propertyChangeSupport != null)
      {
         firePropertyChange ("removeYou", this, null);
         propertyChangeSupport.removeYou();
         propertyChangeSupport = null;
      } // end of if ()

      if (fsaInterface != null)
      {
         fsaInterface.removeYou();
         fsaInterface = null;
      }
      removeAllFromElementReferences();
      removeAllFromDiagrams();
      removeAllFromAnnotations();
      removeAllFromUnparseInformations();

      //CoObRA:
      setRepository (null);
      if (!coobraPersistent)
      {
         removeFromTransientElements();
      }

      removeAllFromUserMessages();

      super.removeYou();
   }


   /**
    * Hangs the current ASGElement into the ASG-tree. Needed for cut'n'paste.
    *
    * @param parent
    */
   public void setCutCopyPasteParent (FElement parent)
   {
      removeAllFromDiagrams();
      if (parent instanceof ASGDiagram)
      {
          ((ASGDiagram) parent).addToElements (this);
      }
      else if (parent != null)
      {
         Iterator it = parent.iteratorOfDiagrams();
         if (it.hasNext())
         {
             ((ASGDiagram) it.next()).addToElements (this);
         }
      }
   }


   /**
    * Sets the generated attribute of the BasicIncrement object
    *
    * @param value  The new generated value
    */
   public void setGenerated (boolean value)
   {
      if (value != isGenerated())
      {
         firePropertyChange (FElement.GENERATED_PROPERTY, isGenerated(), value);
      }
      super.setGenerated (value);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static volatile Set uneditableProperties;


   /**
    * Should return true when an inspection of the named field is senseful. (for property editor)
    *
    * @param fieldName  the name of the field
    * @return           false when the field should not be inspected.
    */
   public boolean isInspectableField (String fieldName)
   {
      if (uneditableProperties == null)
      {
         uneditableProperties = new FTreeSet();
         uneditableProperties.add ("id");
         uneditableProperties.add ("iD");
         uneditableProperties.add ("generated");
      }

      //allow only writable fields that are primitives or for which values are proposed
      if (ObjectInspector.get().hasSetter (this.getClass(), fieldName))
      {
         Class cls = ObjectInspector.get().getFieldClass (this.getClass(), fieldName);
         if (cls.isPrimitive() || cls.equals (String.class))
         {
            return !uneditableProperties.contains (fieldName);
         }
         else
         {
            return proposeFieldValues (fieldName, cls) != null;
         }
      }
      else
      {
         return "lastModified".equals (fieldName);
      }
   }


   /**
    * Returns proposals for values that could be inserted into the field (for property editor)
    *
    * @param fieldName   name of the field that should get the values
    * @param fieldClass  class of the field for which to propose values
    * @return            an Iterator through value proposals, may return null when no proposals
    *         are available
    */
   public Enumeration proposeFieldValues (String fieldName, Class fieldClass)
   {
      Class[] interfaces = fieldClass.getInterfaces();
      for (int i = 0; i < interfaces.length; i++)
      {
         if (interfaces[i].equals (FClass.class))
         {
            return UMLProject.get().elementsOfClasses();
         }
      }
      return null;
   }


   /**
    * Get the lastModified attribute of the ASGElement object
    *
    * @return   The lastModified value
    */
   public String getLastModified()
   {
      if (getRepository() != null)
      {
         ObjectChange change = getRepository().getLastChange (this, null);
         if (change != null)
         {
            ObjectChangeCause cause = change.getCause();
            String causeString = cause != null ? cause.toString() : "";
            String timestamp = change.getTimestamp() != 0 ? "(" + new Date (change.getTimestamp()).toString() + ")" : "";
            return causeString + timestamp;
         }
         else
         {
            return "not modified";
         }
      }
      else
      {
         return "?";
      }
   }


   /**
    * @return   Returns the inTransientMode.
    */
   public static boolean isInTransientMode()
   {
      return inTransientMode;
   }


   /**
    * @param inTransientMode  The inTransientMode to set.
    * @deprecated             ARG!!! Use _at_least_ something thread dependent!
    */
   public static void setInTransientMode (boolean inTransientMode)
   {
      ASGElement.inTransientMode = inTransientMode;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient Map transientElements = new WeakHashMap();


   /**
    * Access method for a To N-association.
    */
   private void addToTransientElements()
   {
      transientElements.put (this, null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void removeFromTransientElements()
   {
      transientElements.remove (this);
   }


   /**
    * Get the transientElements attribute of the ASGElement class
    *
    * @return   The transientElements value
    */
   public static Set getTransientElements()
   {
      return Collections.unmodifiableSet (transientElements.keySet());
   }


   /**
    * set of messages regarding this element
    */
   private transient Map userMessages; //FIX ME: this is a map as java brings no WeakHashSet, change to Set if utils can be used (Java 1.5 lib)


   /**
    * Stick a user message onto this element. This element will become part of the context of the message.
    *
    * @param value  new message
    * @return       true if something was changed
    */
   public boolean addToUserMessages (Message value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.userMessages == null)
         {
            this.userMessages = new WeakHashMap();
         }
         changed = this.userMessages.put (value, "") != null;
         firePropertyChange ("userMessages", null, value);
         if (changed)
         {
            value.addToContext (this);
         }
      }
      return changed;
   }


   /**
    * Iterate through all user messages this element if context of.
    *
    * @return   iterator through Messages
    */
   public Iterator iteratorOfUserMessages()
   {
      if (this.userMessages == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return this.userMessages.keySet().iterator();
      }
   }


   /**
    * remove this element from all message contexts.
    */
   public void removeAllFromUserMessages()
   {
      Message tmpValue;
      Iterator iter = this.iteratorOfUserMessages();
      while (iter.hasNext())
      {
         tmpValue = (Message) iter.next();
         this.removeFromUserMessages (tmpValue);
      }
   }


   /**
    * Remove this element from the context of the message.
    *
    * @param value  which message
    * @return       true if something was changed
    */
   public boolean removeFromUserMessages (Message value)
   {
      boolean changed = false;
      if ( (this.userMessages != null) &&  (value != null))
      {
         changed = this.userMessages.remove (value) != null;
         firePropertyChange ("userMessages", value, null);
         if (changed)
         {
            value.removeFromContext (this);
         }
      }
      return changed;
   }


   /**
    * @return   number of messages this element is context of.
    */
   public int sizeOfUserMessages()
   {
      return  ( (this.userMessages == null)
         ? 0
         : this.userMessages.size());
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    * This method allows to navigate in direction of the model root (project) from any element within a project.
    *
    * @return   the logical parent of this element, may not return null unless this is the top level node (project)
    *         or is not contained in any parent yet
    */
   public FElement getParentElement()
   {
      throw new UnsupportedOperationException (getClass() + " has no implementation for getParentElement()");
   }
}

/*
 * $Log: ASGElement.java,v $
 * Revision 1.112.2.7  2005/11/30 12:05:53  fklar
 * bugfix: extracted setting project to 'unsafed' from PersistencyListener and introduced a new PropertyChangeListener for ASGElements, because otherwise the project is never set to 'unsafed', if 'versioning' is deactivated in general preferences (which means e.g., that listeners for a project won't be informed if the project has been safed)
 *
 */
