/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.asg;

import java.util.Iterator;
import java.util.Map;
import de.uni_paderborn.fujaba.metamodel.FElement;

import de.upb.tools.fca.*;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *                -------           information      0..1
 * ASGInformation | key |---------------------------------> String
 *                -------                     information
 *
 *                       -------- 0..1   aSGInformation   0..1
 * ASGUnparseInformation | name |------------------------------ ASGInformation
 *                       -------- parent        aSGinformation
 * </pre>
 *
 * @author    trioptimum
 * @version   $Revision: 1.7 $ $Date: 2005/02/14 14:28:03 $
 */
public class ASGInformation extends ASGElement
{
   /**
    * default Constructor.
    */
   public ASGInformation() { }


   /**
    * copy Constructor.
    *
    * @param toCopy  No description provided
    */
   public ASGInformation (ASGInformation toCopy)
   {
      for (Iterator it = toCopy.entriesOfInformation(); it.hasNext(); )
      {
         Map.Entry entry = (Map.Entry) it.next();
         addToInformation (entry);
      }
   }


   /**
    * <pre>
    *                -------           information      0..1
    * ASGInformation | key |---------------------------------> String
    *                -------                     information
    * </pre>
    */
   private FHashMap information;


   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToInformation (String key, String value)
   {
      boolean changed = false;
      if ( (value != null) &&  (key != null))
      {
         if (this.information == null)
         {
            this.information = new FPropHashMap (this, "information");
         }
         String oldValue = (String) this.information.put (key, value);
         if (oldValue != value)
         {
            changed = true;
         }
      }
      return changed;
   }


   /**
    * Access method for an one to n association.
    *
    * @param entry  The object added.
    * @return       No description provided
    */
   public boolean addToInformation (Map.Entry entry)
   {
      return addToInformation ((String) entry.getKey(), (String) entry.getValue());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfInformation()
   {
      return  ( (this.information == null) ? FEmptyIterator.get()
         : this.information.entrySet().iterator());
   }


   /**
    * Get the fromInformation attribute of the ASGInformation object
    *
    * @param key  No description provided
    * @return     The fromInformation value
    */
   public String getFromInformation (String key)
   {
      return  ( ( (this.information == null) ||  (key == null)) ? null
         : (String) this.information.get (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInInformation (String key, String value)
   {
      return  ( (this.information != null) &&  (value != null) &&  (key != null) &&  (this.information
         .get (key) == value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInInformation (String value)
   {
      return  ( (this.information != null) &&  (value != null) && this.information
         .containsValue (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInInformation (String key)
   {
      return  ( (this.information != null) &&  (key != null) && this.information
         .containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfInformation()
   {
      return  ( (this.information == null) ? FEmptyIterator.get()
         : this.information.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfInformation()
   {
      return  ( (this.information == null) ? FEmptyIterator.get()
         : this.information.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromInformation()
   {
      Iterator iter = entriesOfInformation();
      Map.Entry entry;
      while (iter.hasNext())
      {
         entry = (Map.Entry) iter.next();
         removeFromInformation ((String) entry.getKey(), (String) entry.getValue());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromInformation (String key, String value)
   {
      boolean changed = false;
      if ( (this.information != null) &&  (value != null) &&  (key != null))
      {
         String oldValue = (String) this.information.get (key);
         if (oldValue == value)
         {
            this.information.remove (key);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromInformation (String value)
   {
      boolean changed = false;
      if ( (this.information != null) &&  (value != null))
      {
         Iterator iter = this.entriesOfInformation();
         Map.Entry entry;
         while (iter.hasNext())
         {
            entry = (Map.Entry) iter.next();
            if (entry.getValue() == value)
            {
               changed = changed
                  || this.removeFromInformation ((String) entry.getKey(), value);
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromInformation (String key)
   {
      boolean changed = false;
      if ( (this.information != null) &&  (key != null))
      {
         String tmpValue = (String) this.information.get (key);
         if (tmpValue != null)
         {
            this.information.remove (key);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfInformation()
   {
      return  ( (this.information == null) ? 0 : this.information.size());
   }


   /**
    * <pre>
    *                       -------- 0..1   ASGInformation   0..1
    * ASGUnparseInformation | name |------------------------------ ASGInformation
    *                       -------- parent        aSGinformation
    * </pre>
    */
   private transient ASGUnparseInformation parent;


   /**
    * Get the parent attribute of the ASGInformation object
    *
    * @return   The parent value
    */
   public ASGUnparseInformation getParent()
   {
      return this.parent;
   }


   /**
    * Sets the parent attribute of the ASGInformation object
    *
    * @param key    The new parent value
    * @param value  The new parent value
    * @return       No description provided
    */
   public boolean setParent (String key, ASGUnparseInformation value)
   {
      boolean changed = false;
      if (this.parent != value)
      {
         if (this.parent != null)
         {
            ASGUnparseInformation oldValue = this.parent;
            this.parent = null;

            //key is null (removeYou)
            if (key == null && !oldValue.removeFromASGInformation (key, this))
            {
               oldValue.removeFromASGInformation (this);
            }
         }
         this.parent = value;
         if (value != null)
         {
            value.addToASGInformation (key, this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      removeAllFromInformation();

      ASGUnparseInformation tmpParent = getParent();
      if (tmpParent != null)
      {
         setParent (null, null);
      }
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return getParent();
   }
}

/*
 * $Log: ASGInformation.java,v $
 * Revision 1.7  2005/02/14 14:28:03  cschneid
 * introduced message view, getParentElement() for FElement, code generation and compiler messages in message view
 *
 */
