/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.asg;

import java.awt.*;
import java.util.*;

import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.upb.tools.fca.*;


/**
 * Stores information about an UnparseInterface of an ASGElement
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *                       -------- 0..1   ASGInformation   0..1
 * ASGUnparseInformation | name |------------------------------ ASGInformation
 *                       -------- parent        aSGInformation
 * </pre>
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.15 $
 */
public class ASGUnparseInformation extends ASGElement
{
   /**
    * flag for enabling conversion of old location info
    */
   private static boolean pointsCalled;


   /**
    * default ctor
    */
   public ASGUnparseInformation() { }


   /**
    * copy ctor
    *
    * @param toCopy  what to copy
    */
   public ASGUnparseInformation (ASGUnparseInformation toCopy)
   {
      for (Iterator it = toCopy.entriesOfASGInformation(); it.hasNext(); )
      {
         Map.Entry entry = (Map.Entry) it.next();
         addToASGInformation ((String) entry.getKey(), new ASGInformation ((ASGInformation) entry.getValue()));
      }
   }


   /**
    * @see   #getASGElement
    */
   private transient ASGElement aSGElement;


   /**
    * assign an ASGElement to this information
    *
    * @param key    used to assign this information to the ASGElement
    * @param value  ASGElement
    * @return       true when value was changed
    */
   public boolean setASGElement (ASGElement key, ASGElement value)
   {
      boolean changed = false;
      if (this.aSGElement != value)
      {
         ASGElement oldValue = this.aSGElement;
         if (this.aSGElement != null)
         {
            this.aSGElement = null;
            oldValue.removeFromUnparseInformations (key, this);
         }

         this.aSGElement = value;
         this.aSGElementKey = key;
         firePropertyChange ("aSGElement", oldValue, value);
         if (value != null)
         {
            value.addToUnparseInformations (key, this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * used for loading/redo (do not use directly)
    *
    * @param value  new assigned ASGElement
    * @return       true when something was changed
    */
   public boolean setASGElement (ASGElement value)
   {
      if (aSGElementKey != null)
      {
         return setASGElement (aSGElementKey, value);
      }
      else
      {
         return false;
      }
   }


   /**
    * @return   assigned ASGElement
    */
   public ASGElement getASGElement()
   {
      return this.aSGElement;
   }


   /**
    * @see   #getASGElementKey
    */
   private transient ASGElement aSGElementKey;


   /**
    * @return   the key which is used to assign this information to the
    *         ASGElement
    */
   public ASGElement getASGElementKey()
   {
      return aSGElementKey;
   }


   /**
    * <pre>
    *                       -------- 0..1   ASGInformation   0..1
    * ASGUnparseInformation | name |------------------------------ ASGInformation
    *                       -------- parent        aSGInformation
    * </pre>
    */
   private FHashMap aSGInformation;


   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToASGInformation (String key, ASGInformation value)
   {
      boolean changed = false;
      if ( (value != null) &&  (key != null))
      {
         if (this.aSGInformation == null)
         {
            this.aSGInformation = new FPropHashMap (this, "aSGInformation"); // or FTreeMap
         }
         ASGInformation oldValue = (ASGInformation) this.aSGInformation.put (key, value);
         if (oldValue != value)
         {
            if (oldValue != null)
            {
               oldValue.setParent (null, null);
            }
            value.setParent (key, this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * Access method for an one to n association.
    *
    * @param entry  The object added.
    * @return       No description provided
    */
   public boolean addToASGInformation (Map.Entry entry)
   {
      return addToASGInformation ((String) entry.getKey(),
         (ASGInformation) entry.getValue());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfASGInformation()
   {
      return  ( (this.aSGInformation == null) ? FEmptyIterator.get()
         : this.aSGInformation.entrySet().iterator());
   }


   /**
    * Get the fromASGInformation attribute of the ASGUnparseInformation object
    *
    * @param key  No description provided
    * @return     The fromASGInformation value
    */
   public ASGInformation getFromASGInformation (String key)
   {
      return  ( ( (this.aSGInformation == null) ||  (key == null)) ? null
         : (ASGInformation) this.aSGInformation.get (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInASGInformation (ASGInformation value)
   {
      return  ( (this.aSGInformation != null) &&  (value != null) && this.aSGInformation
         .containsValue (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInASGInformation (String key, ASGInformation value)
   {
      return  ( (this.aSGInformation != null) &&  (value != null)
         &&  (key != null) &&  (this.aSGInformation.get (key) == value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInASGInformation (String key)
   {
      return  ( (this.aSGInformation != null) &&  (key != null) && this.aSGInformation
         .containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfASGInformation()
   {
      return  ( (this.aSGInformation == null) ? FEmptyIterator.get()
         : this.aSGInformation.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfASGInformation()
   {
      return  ( (this.aSGInformation == null) ? FEmptyIterator.get()
         : this.aSGInformation.keySet().iterator());
   }


   /**
    * Remove information from the map without calling removeYou() on the objects.
    */
   public void removeAllFromASGInformation()
   {
      Iterator iter = entriesOfASGInformation();
      Map.Entry entry;
      while (iter.hasNext())
      {
         entry = (Map.Entry) iter.next();
         removeFromASGInformation ((String) entry.getKey(),
            (ASGInformation) entry.getValue());
      }
   }


   /**
    * Remove information from the map and call removeYou() on each object.
    */
   public void removeAllFromASGInformationAndRemoveThem()
   {
      Iterator iter = entriesOfASGInformation();
      Map.Entry entry;
      while (iter.hasNext())
      {
         entry = (Map.Entry) iter.next();
         ASGInformation information = (ASGInformation) entry.getValue();
         removeFromASGInformation ((String) entry.getKey(), information);
         information.removeYou();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromASGInformation (ASGInformation value)
   {
      boolean changed = false;
      if ( (this.aSGInformation != null) &&  (value != null))
      {
         Iterator iter = this.entriesOfASGInformation();
         Map.Entry entry;
         while (iter.hasNext())
         {
            entry = (Map.Entry) iter.next();
            if (entry.getValue() == value)
            {
               changed = changed
                  || this.removeFromASGInformation ((String) entry.getKey(), value);
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromASGInformation (String key, ASGInformation value)
   {
      boolean changed = false;
      if ( (this.aSGInformation != null) &&  (value != null) &&  (key != null))
      {
         ASGInformation oldValue = (ASGInformation) this.aSGInformation
            .get (key);
         if (oldValue == value)
         {
            this.aSGInformation.remove (key);
            value.setParent (null, null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromASGInformation (String key)
   {
      boolean changed = false;
      if ( (this.aSGInformation != null) &&  (key != null))
      {
         ASGInformation tmpValue = (ASGInformation) this.aSGInformation
            .get (key);
         if (tmpValue != null)
         {
            this.aSGInformation.remove (key);
            tmpValue.setParent (null, null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfASGInformation()
   {
      return  ( (this.aSGInformation == null) ? 0 : this.aSGInformation.size());
   }


   //----- Methods for compatibility with old fpr files-----
   //TODO Remove these methods and ASGCollapsableUnparseInformation,
   //warning: this will prevent old fprs from being loaded

   /**
    * @deprecated   use {@link #addToASGInformation(String, ASGInformation)}
    */
   private Map points;


   /**
    * reset flag for enabling conversion of old location info
    */
   public static void resetPointsCalled()
   {
      pointsCalled = false;
   }


   /**
    * flag for enabling conversion of old location info
    *
    * @return   true if {@link #addToPoints(String, java.awt.Point)} was called
    */
   public static boolean isPointsCalled()
   {
      return pointsCalled;
   }


   /**
    * Adds a Point (location) for a specific FSAObject (propertyName).
    * Convenience method.
    *
    * @param key    propertyName
    * @param value  point (location)
    * @return       No description provided
    */
   public boolean addToPoints (String key, Point value)
   {
      pointsCalled = true;
      ASGInformation info = getFromASGInformation (key);
      if (info == null)
      {
         info = new ASGInformation();
         addToASGInformation (key, info);
      }

      return  (info.addToInformation (FSAObject.LOCATION + "_X", Integer.toString (value.x)) | info.addToInformation (FSAObject.LOCATION + "_Y", Integer.toString (value.y)));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    */
   public void removeKeyFromPoints (String key)
   {
      ASGInformation info = getFromASGInformation (key);
      if (info != null)
      {
         info.removeFromInformation (FSAObject.LOCATION + "_X");
         info.removeFromInformation (FSAObject.LOCATION + "_Y");
         if (info.sizeOfInformation() == 0)
         {
            removeKeyFromASGInformation (key);
         }
      }
   }


   /**
    * Adds a Point (location) for a specific FSAObject (propertyName), kept for
    * loading old fprs
    *
    * @param entry  Map.Entry that hold the pair
    * @return       No description provided
    * @deprecated   use {@link #addToASGInformation(String, ASGInformation)}
    */
   protected boolean addToPoints (Map.Entry entry)
   {
      return addToPoints ((String) entry.getKey(), (Point) entry.getValue());
   }


   /**
    * remove all links to other objects
    */
   public void removeYou()
   {
      ASGElement tmpASGElement = getASGElement();
      if (tmpASGElement != null)
      {
         setASGElement (null, null);
      } // if

      removeAllFromASGInformationAndRemoveThem();
      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return getASGElement();
   }
}

/*
 * $Log: ASGUnparseInformation.java,v $
 * Revision 1.15  2005/02/14 14:28:03  cschneid
 * introduced message view, getParentElement() for FElement, code generation and compiler messages in message view
 *
 */
