/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.RuntimeExceptionWithContext;

import de.uni_paderborn.fujaba.metamodel.FElement;


/**
 * Class OOActivityCodeGenHandler
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.12 $ <h2> Associations </h2> <pre>
 *                       0..1          0..1
 * OOGenStrategyHandler -------------------- OOGenStrategyClient
 *                       handler     client
 *
 *                       0..1                 0..1
 * OOGenStrategyHandler --------------------------- OOGenStrategyHandler
 *                       successor     prevHandler
 * </pre>
 */
public abstract class CodeGenStrategyHandler
{
   /**
    * Default Constructor
    */
   public CodeGenStrategyHandler()
   {
      this (false);
   }


   /**
    * Constructor for class OOGenStrategyHandler
    *
    * @param continueChain  No description provided
    */
   public CodeGenStrategyHandler (boolean continueChain)
   {
      setContinueChain (continueChain);
   }


   /**
    * UMLAttribute : 'continueChain : Boolean '
    */
   private boolean continueChain;


   /**
    * Get the value of continueChain.
    *
    * @return   Value of continueChain.
    */
   public boolean isContinueChain()
   {
      return this.continueChain;
   }


   /**
    * Set the value of continueChain.
    *
    * @param continueChain  Value to assign to continueChain.
    */
   public void setContinueChain (boolean continueChain)
   {
      if (this.continueChain != continueChain)
      {
         this.continueChain = continueChain;
      }
   }


   /**
    * Get the value of continue.
    *
    * @param prevToken   No description provided
    * @param param       No description provided
    * @param asgElement  No description provided
    * @param isNewStyle  No description provided
    * @return            Value of continue.
    */

   public final OOGenToken generate (FElement asgElement, OOGenToken prevToken,
                                     Object param[], boolean isNewStyle)
   {
      if (isResponsible (asgElement))
      {
         try
         {
            OOGenToken result = generateSourceCode (asgElement,
               prevToken,
               param);
            if (!isNewStyle && isContinueChain() && getSuccessor() != null)
            {
               result = getSuccessor().generate (asgElement, result, param);
            }

            return result;
         }
         catch (RuntimeException e)
         {
            if (e instanceof RuntimeExceptionWithContext || e instanceof CodeGenException)
            {
               throw e;
            }
            else
            {
               throw new RuntimeExceptionWithContext (e, asgElement);
            }
         }
      }
      else if (!isNewStyle && getSuccessor() != null)
      {
         return getSuccessor().generate (asgElement, prevToken, param);
      }
      else
      {
         throw new IllegalStateException ("incr=" + asgElement + ",prevToken="
            + prevToken + ",param=" + param);
      }

   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param asgElement  No description provided
    * @param prevToken   No description provided
    * @param param       No description provided
    * @return            No description provided
    */
   public final OOGenToken generate (FElement asgElement, OOGenToken prevToken, Object param[])
   {
      return this.generate (asgElement, prevToken, param, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param asgElement  No description provided
    * @return            No description provided
    */
   public final OOGenToken generate (FElement asgElement)
   {
      return generate (asgElement, null, null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param prevToken   No description provided
    * @param asgElement  No description provided
    * @return            No description provided
    */
   public final OOGenToken generate (FElement asgElement, OOGenToken prevToken)
   {
      return generate (asgElement, prevToken, null);
   }


   /**
    * check if given FElement incr needs a <code>OOGenToken</code> != <code>null</code> as
    * parameter for generate method
    *
    * @param asgElement  No description provided
    * @return            true, if <code>incr</code> needs a <code>OOGenToken</code> != <code>null</code>
    *      , else false
    * @see               #generate(FElement,OOGenToken,Object[])
    */
   public final boolean incrNeedsToken (FElement asgElement)
   {
      if (isResponsible (asgElement))
      {
         return needToken();
      }
      else
      {
         throw new IllegalStateException ("incr=" + asgElement);
      }
   }


   /**
    * Returns true if this handler is responsible for generating code for the lement. Subclasses
    * must override this method. For backward compatibility this method is not abstract but
    * invokes the old implementation.
    *
    * @param element  No description provided
    * @return         The responsible value
    */
   public boolean isResponsible (FElement element)
   {
      return isResponsible ((ASGElement) element);
   }


   /**
    * @param asgElement
    * @return            The responsible value
    * @deprecated        use {@link #isResponsible(de.uni_paderborn.fujaba.metamodel.FElement)}
    *      instead
    */
   public boolean isResponsible (ASGElement asgElement)
   {
      throw new AbstractMethodError (getClass().getName() + " must implement isResponsible(de.uni_paderborn.fujaba.metamodel.FElement)");
   }


   /**
    * Generate source code for the element. Subclasses must override this method. For backward
    * compatibility this method is not abstract but invokes the old implementation.
    *
    * @param prevToken  No description provided
    * @param param      No description provided
    * @param element    No description provided
    * @return           No description provided
    */
   public OOGenToken generateSourceCode (FElement element, OOGenToken prevToken, Object param[])
   {
      return generateSourceCode ((ASGElement) element, prevToken, param);
   }


   /**
    * @param asgElement
    * @param prevToken
    * @param param
    * @return            No description provided
    * @deprecated        use {@link #generateSourceCode(de.uni_paderborn.fujaba.metamodel.FElement,
    *      OOGenToken, Object[])} instead
    */
   public OOGenToken generateSourceCode (ASGElement asgElement, OOGenToken prevToken, Object param[])
   {

      throw new AbstractMethodError (getClass().getName() + " must implement generateSourceCode(de.uni_paderborn.fujaba.metamodel.FElement, OOGenToken, Object[])");
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public abstract boolean needToken();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param methodName  No description provided
    * @param param       No description provided
    * @param asgElement  No description provided
    * @return            No description provided
    */
   public final Object generateCode (FElement asgElement, String methodName, Object param[])
   {
      Object result = getClientOfChain().generateCode (asgElement, methodName, param);

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param prevToken   No description provided
    * @param methodName  No description provided
    * @param param       No description provided
    * @param asgElement  No description provided
    */
   public final void appendOOStatement (OOGenToken prevToken, FElement asgElement, String methodName, Object param[])
   {
      prevToken.appendStatement (generateCode (asgElement, methodName, param));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param prevToken  No description provided
    * @param statement  No description provided
    */
   public final void appendOOStatement (OOGenToken prevToken, OOStatement statement)
   {
      prevToken.appendStatement (statement);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param prevToken   No description provided
    * @param statements  No description provided
    */
   public final void appendOOStatement (OOGenToken prevToken, OOStatement statements[])
   {
      prevToken.appendStatement (statements);
   }


   /**
    * Get the clientOfChain attribute of the OOGenStrategyHandler object
    *
    * @return   The clientOfChain value
    */
   public final CodeGenStrategy getClientOfChain()
   {
      return this.getChainMaster();
   }


   /**
    * <pre>
    *                  0..1         handlerChain          0..n
    * CodeGenStrategy ----------------------------------------- CodeGenStrategyHandler
    *                  chainMaster   {ordered,}   handlerChain
    * </pre>
    */

   private CodeGenStrategy chainMaster;



   /**
    * Get the chainMaster attribute of the CodeGenStrategyHandler object
    *
    * @return   The chainMaster value
    */
   public CodeGenStrategy getChainMaster()
   {

      return this.chainMaster;
   }



   /**
    * Sets the chainMaster attribute of the CodeGenStrategyHandler object
    *
    * @param value  The new chainMaster value
    * @return       No description provided
    */
   public boolean setChainMaster (CodeGenStrategy value)
   {

      boolean changed = false;

      if (this.chainMaster != value)
      {

         if (this.chainMaster != null)
         {

            CodeGenStrategy oldValue = this.chainMaster;

            this.chainMaster = null;

            oldValue.removeFromHandlerChain (this);

         }

         this.chainMaster = value;

         if (value != null)
         {

            value.addToHandlerChain (this);

         }

         changed = true;

      }

      return changed;
   }



   /**
    * <pre>
    *                       0..1                 0..1
    * OOGenStrategyHandler --------------------------- OOGenStrategyHandler
    *                       successor     prevHandler
    * </pre>
    */
   private CodeGenStrategyHandler prevHandler;


   /**
    * @param value  The new prevHandler value
    * @return       No description provided
    * @see          #prevHandler
    */
   public boolean setPrevHandler (CodeGenStrategyHandler value)
   {
      if (this.prevHandler != value)
      {
         if (this.prevHandler != null)
         {
            CodeGenStrategyHandler oldValue = this.prevHandler;
            this.prevHandler = null;
            oldValue.setSuccessor (null);
         }
         this.prevHandler = value;
         if (value != null)
         {
            this.prevHandler.setSuccessor (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The prevHandler value
    * @see      #prevHandler
    */
   public CodeGenStrategyHandler getPrevHandler()
   {
      return this.prevHandler;
   }


   /**
    * <pre>
    *                       0..1                 0..1
    * OOGenStrategyHandler --------------------------- OOGenStrategyHandler
    *                       prevHandler     successor
    * </pre>
    */
   private CodeGenStrategyHandler successor;


   /**
    * @param value  The new successor value
    * @return       No description provided
    * @see          #successor
    */
   public boolean setSuccessor (CodeGenStrategyHandler value)
   {
      if (this.successor != value)
      {
         if (this.successor != null)
         {
            CodeGenStrategyHandler oldValue = this.successor;
            this.successor = null;
            oldValue.setPrevHandler (null);
         }
         this.successor = value;
         if (value != null)
         {
            this.successor.setPrevHandler (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The successor value
    * @see      #successor
    */
   public CodeGenStrategyHandler getSuccessor()
   {
      return this.successor;
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "OOGenStrategyHandler[]";
   }
}

/*
 * $Log: CodeGenStrategyHandler.java,v $
 * Revision 1.12  2005/02/17 12:49:19  cschneid
 * Exception and Error handling improved - execptions in actions displayed, context detection for codegen; new libraries
 *
 */
