/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.UMLProject;


/**
 * Class EiffelGenVisitor
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.31.2.1 $
 */
public class EiffelGenVisitor
    extends OOGenVisitor
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (EiffelGenVisitor.class);


   /**
    * Default Constructor
    */
   public EiffelGenVisitor()
   {
      super ("e");

      //
      // overwrite some public attr
      //
      NULL_STRING = "Void";

      START_BLOCK = "begin";
      END_BLOCK = "end;";

      START_IF_BLOCK = "then";

      START_METHOD_BLOCK = "is";
      START_METHOD_BODY = "do";
      END_METHOD_BLOCK = "end";

      SDM_ENSURE_STRING = "EiffelSDM.sdmEnsure";
      SDM_EXCEPTION_STRING = "EiffelSDMException";

      // assignments
      ASSIGNMENT_OP_STRING = ":=";

      THIS_STRING = "Current";

      // constants for code generation
      START_BLOCK_COMMENTARY = "--";
      END_BLOCK_COMMENTARY = "--";
      LINE_IN_BLOCK_COMMENTARY = "--";
      LINE_COMMENTARY = "--";
   }

   //
   // begin assoc template section
   //

   /**
    * Get the referenceTemplateName attribute of the EiffelGenVisitor object
    *
    * @return   The referenceTemplateName value
    */
   public final String getReferenceTemplateName()
   {
      return "Templates/ReferenceTemplate.Eiffel.tpl";
   }


   /**
    * Get the assocTemplateName attribute of the EiffelGenVisitor object
    *
    * @return   The assocTemplateName value
    */
   public final String getAssocTemplateName()
   {
      return "Templates/AssocTemplateFCA.Eiffel.tpl";
   }

   //
   // end assoc template section
   //

   /**
    * get the full qualified containername for type <code>containerType</code>
    *
    * @param containerType  the <code>containerType</code>
    * @param bound          true, if assoc is bound
    * @return               the full qualified containername
    */
   public String getContainerName (OOCollectionEnum containerType, boolean bound)
   {
      StringBuffer containerName = new StringBuffer();

      if (OOCollectionEnum.LIST.equals (containerType))
      {
         containerName.append ("LINKED_LIST");
      }
      else if (OOCollectionEnum.SET.equals (containerType))
      {
         containerName.append ("<unkown>");
      }
      else if (OOCollectionEnum.MAP.equals (containerType))
      {
         containerName.append ("<unkown>");
      }
      else if (OOCollectionEnum.MULTI_MAP.equals (containerType))
      {
         containerName.append ("<unkown>");
      }

      return containerName.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initProgLangTypes()
   {
      addToProgLangTypes (FBaseTypes.INITIALIZER, "");
      addToProgLangTypes (FBaseTypes.BOOLEAN, "BOOLEAN");
      addToProgLangTypes (FBaseTypes.CHARACTER, "char");
      addToProgLangTypes (FBaseTypes.STRING, "STRING");
      addToProgLangTypes (FBaseTypes.INTEGER, "INTEGER");
      addToProgLangTypes (FBaseTypes.BYTE, "byte");
      addToProgLangTypes (FBaseTypes.SHORT_INTEGER, "short");
      addToProgLangTypes (FBaseTypes.LONG_INTEGER, "long");
      addToProgLangTypes (FBaseTypes.FLOAT, "float");
      addToProgLangTypes (FBaseTypes.DOUBLE, "double");
      addToProgLangTypes (FBaseTypes.VOID, "void");
      addToProgLangTypes (FBaseTypes.CONSTRUCTOR, "");

      addToProgLangTypes ("BooleanArray", "boolean[]");
      addToProgLangTypes ("CharacterArray", "char[]");
      addToProgLangTypes ("StringArray", "String[]");
      addToProgLangTypes ("IntegerArray", "int[]");
      addToProgLangTypes ("ByteArray", "byte[]");
      addToProgLangTypes ("ShortIntegerArray", "short[]");
      addToProgLangTypes ("LongIntegerArray", "long[]");
      addToProgLangTypes ("FloatArray", "float[]");
      addToProgLangTypes ("DoubleArray", "double[]");
      addToProgLangTypes ("ObjectArray", "Object[]");

      // fix me
      addToProgLangTypes ("FWIterator", "<iterator>");

   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initVisibilityStrings()
   {
      visibilityStrings = new String[]{
         "private",
         "public",
         "protected",
         "/*package*/",
         "/*user*/"
         };
   }

   //
   // begin visit OOStatement section
   //

   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public final String getSourceCode (OOThrowStatement statement)
   {
      StringBuffer result = new StringBuffer();

      result.append (createIndentString());
      result.append ("throw ");
      result.append (statement.getMyException().getSourceCode (this));

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public final String getSourceCode (OOStartMethodBodyStatement statement)
   {
      return createIndentString() + "do";
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOIfStatement statement)
   {
      if (log.isDebugEnabled())
      {
         log.debug (this + ".getSourceCode(" + statement + ")");
      }

      StringBuffer result = new StringBuffer();
      result.append (createIndentString());
      result.append (IF_STRING);
      result.append (" ");
      if (statement.getCondition() != null)
      {
         result.append (statement.getCondition().getSourceCode (this));
      }
      else
      {
         result.append (START_BLOCK_COMMENTARY +
            " oops, condition is missing " + END_BLOCK_COMMENTARY);
      }

      return result.toString();
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param statement  No description provided
    * @return           The sourceCode value
    */
   public String getSourceCode (OOStartBlockStatement statement)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info (this + ".getSourceCode(" + statement + ")");
         }
      }

      StringBuffer result = new StringBuffer();
      OOStatement prevStat = statement.getToken().getPreviousOfStatement (statement);

      result.append (createIndentString());

      if (prevStat instanceof OOIfStatement)
      {
         result.append ("then");
      }
      else
      {
         result.append (START_BLOCK);
      }

      incIndentNr();
      return result.toString();
   }

   //
   // end visit OOStatement section
   //

   //
   // begin visit OOExpression section
   //

   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOObjectOfTypeExpr expression)
   {
      return this + ".getSourceCode(" + expression + ") is empty !";
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOClassOfVariableExpr expression)
   {
      return this + ".getSourceCode(" + expression + ") is empty !";
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorOfExpr expression)
   {
      return this + ".getSourceCode(" + expression + ") is empty !";
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorGetExpr expression)
   {
      return this + ".getSourceCode(" + expression + ") is empty !";
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorHasNextExpr expression)
   {
      return this + ".getSourceCode(" + expression + ") is empty !";
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param expression  No description provided
    * @return            The sourceCode value
    */
   public final String getSourceCode (OOFWIteratorGotoNextExpr expression)
   {
      return this + ".getSourceCode(" + expression + ") is empty !";
   }


   /**
    * Get the sourceCode attribute of the EiffelGenVisitor object
    *
    * @param expr  No description provided
    * @return      The sourceCode value
    */
   public String getSourceCode (OONewArrayExpr expr)
   {
      return this + ".getSourceCode (" + expr + ") is empty !";
   }

   //
   // end visit OOExpression section
   //

   //
   // generate section
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theFile  No description provided
    */
   public void generatePackageBegin (FFile theFile) { }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theFile  No description provided
    */
   public void generatePackageEnd (FFile theFile) { }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void generateClassEndBlock()
   {
      getCurrentBuffer().append (END_BLOCK);
      getCurrentBuffer().append ("\n");

      decIndentNr();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void generateClassStartBlock()
   {
      getCurrentBuffer().append (START_BLOCK);
      getCurrentBuffer().append ("\n");

      incIndentNr();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param thePackage  No description provided
    * @return            No description provided
    */
   public String generateImportPackage (FPackage thePackage)
   {
      return "";
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String generateImportClass (FClass theClass)
   {
      return "";
   }

   //
   // create section
   //

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    * @return          No description provided
    */
   public String createClassInterfaceModifier (FClass theClass)
   {
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theAttr  No description provided
    * @return         No description provided
    */
   public String createAttrDeclaration (FAttr theAttr)
   {
      StringBuffer declString = new StringBuffer();

      declString.append ("feature ");
      declString.append (getAttrDeclVisibilityString (theAttr.getVisibility()));
      declString.append (" ");
      declString.append (getAttrDeclName (theAttr));

      declString.append (" : ");
      declString.append (getTypeAsString (theAttr.getFAttrType(), false));
      declString.append (";\n\n");

      return declString.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theMethod  No description provided
    * @return           No description provided
    */
   public String createMethodDeclaration (FMethod theMethod)
   {
      StringBuffer declString = new StringBuffer();

      declString.append ("feature ");

      if (getMethodDeclVisibilityString (theMethod.getVisibility()) != null)
      {
         declString.append (getMethodDeclVisibilityString (theMethod.getVisibility()));
         declString.append (" ");
      }

      // append methodname
      declString.append (getMethodDeclName (theMethod));

      // generate parameters
      String methodParamList = createMethodParamList (theMethod);

      if (methodParamList.length() > 0)
      {
         declString.append (OPEN_NORM_BRACKET);
         declString.append (methodParamList);
         declString.append (CLOSE_NORM_BRACKET);
      }

      // append result type
      if (theMethod.getFResultType() != null)
      {
         if (theMethod.getFResultType() != UMLProject.get().getFromBaseTypes (FBaseTypes.VOID))
         {
            declString.append (" : ");
            declString.append (getTypeAsString (theMethod.getFResultType()));
         }
      }
      else
      {
         declString.append ("result_type_is_missing");
      }

      return indentText (declString.toString());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theParam  No description provided
    * @return          No description provided
    */
   public String createMethodParam (FParam theParam)
   {
      StringBuffer declString = new StringBuffer();

      if (theParam.getFParamType() == null)
      {
         if (log.isInfoEnabled())
         {
            log.info ("ERROR: parameter in method found without a type !");
         }
         if (theParam.getFRevParam().getFParent() != null)
         {
            if (log.isInfoEnabled())
            {
               log.info ("\tClass    : " + theParam.getFRevParam().getFParent().getID() + ": " +
                  theParam.getFRevParam().getFParent().getName());
            }
         }
         else
         {
            if (log.isInfoEnabled())
            {
               log.info ("\tClass    : NULL !!!");
            }
         }
         if (log.isInfoEnabled())
         {
            log.info ("\tMethod   : " + theParam.getFRevParam().getID() + ": " + theParam.getFRevParam().getName());
         }
         if (log.isInfoEnabled())
         {
            log.info ("\tParameter: " + theParam.getID() + ": " + theParam.getName());
         }
      }
      else
      {
         declString.append (theParam.getName());
         declString.append (" : ");
         declString.append (getTypeAsString (theParam.getFParamType()));
      }

      return declString.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param varType     No description provided
    * @param varName     No description provided
    * @param initString  No description provided
    * @return            No description provided
    */
   public String createVarDecl (String varType, String varName, String initString)
   {
      StringBuffer declString = new StringBuffer();

      declString.append (varName);
      declString.append (" : ");
      declString.append (varType);

      if (initString != null)
      {
         declString.append (" ");
         declString.append (ASSIGNMENT_OP_STRING);
         declString.append (" ");
         declString.append (initString);
      }

      declString.append (" ");
      declString.append (END_OF_STATEMENT_LINE);

      return declString.toString();
   }

   //
   // append section
   //

   //
   // get identifier, keywords, literals etc.
   //

   /**
    * Get the attrDeclName attribute of the EiffelGenVisitor object
    *
    * @param theAttr  No description provided
    * @return         The attrDeclName value
    */
   public String getAttrDeclName (FAttr theAttr)
   {
      return theAttr.getName();
   }


   /**
    * Get the methodDeclName attribute of the EiffelGenVisitor object
    *
    * @param theMethod  No description provided
    * @return           The methodDeclName value
    */
   public String getMethodDeclName (FMethod theMethod)
   {
      return theMethod.getName();
   }


   /**
    * Get the uMLAttrDeclVisibilityString attribute of the EiffelGenVisitor object
    *
    * @param visibility  No description provided
    * @return            The uMLAttrDeclVisibilityString value
    */
   public String getAttrDeclVisibilityString (int visibility)
   {
      switch (visibility)
      {
         case FDeclaration.PRIVATE:
            return "{}";
         default:
            return this.toString();
      }
   }


   /**
    * Get the uMLMethodDeclVisibilityString attribute of the EiffelGenVisitor object
    *
    * @param visibility  No description provided
    * @return            The uMLMethodDeclVisibilityString value
    */
   public String getMethodDeclVisibilityString (int visibility)
   {
      switch (visibility)
      {
         case FDeclaration.PRIVATE:
            return "{}";
         default:
            return this.toString();
      }
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "EiffelGenVisitor[]";
   }
}

/*
 * $Log: EiffelGenVisitor.java,v $
 * Revision 1.31.2.1  2005/09/30 18:56:54  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
