/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.Iterator;
import java.util.LinkedList;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.Utility;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FLinkedList;


/**
 * <h2> Associations </h2> <pre>
 *             0..1                             0..1
 * OOGenToken --------------------------------------- ASGElement
 *             firstOOGenToken     firstAsgElement
 *
 *             0..1                           0..1
 * OOGenToken ------------------------------------- ASGElement
 *             lastOOGenToken     lastAsgElement
 *
 *              n      hasStatements      0..1
 * OOStatement -------------------------------- OOGenToken
 *              statement   {ordered,}   token
 *
 *             0..1                       0..1
 * OOGenToken --------------------------------- OOGenToken
 *             prev                       next
 * </pre>
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.26.2.1 $
 */
public class OOGenToken
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (OOGenToken.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static transient int FIRST = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static transient int LAST = FIRST + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static transient int UNUSED = LAST + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static transient int FILEHEADER = UNUSED + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static transient int FILEFOOT = FILEHEADER + 1;

   /*
    *  Imports
    *  Package
    *  ClassDeclaration
    *  ClassHeader
    *  ClassExtends
    *  ClassImplements
    *  BeginBody
    *  EndBody
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String FIRST_TOKEN_NAME = "FIRSTTOKEN";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String LAST_TOKEN_NAME = "LASTTOKEN";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient String GENERATED_TOKEN_NAME = "GENERATEDTOKEN";


   /**
    * Get the firstToken attribute of the OOGenToken object
    *
    * @return   The firstToken value
    */
   public boolean isFirstToken()
   {
      return this.sectionName.equals (FIRST_TOKEN_NAME);
   }


   /**
    * Get the lastToken attribute of the OOGenToken object
    *
    * @return   The lastToken value
    */
   public boolean isLastToken()
   {
      return this.sectionName.equals (LAST_TOKEN_NAME);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final static OOGenToken createFirstToken()
   {
      return new OOGenToken (null, FIRST_TOKEN_NAME);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final static OOGenToken createLastToken()
   {
      return new OOGenToken (null, LAST_TOKEN_NAME);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final static OOGenToken createGeneratedToken()
   {
      return new OOGenToken (null, GENERATED_TOKEN_NAME);
   }


   /**
    * Constructor for class OOGenToken
    */
   public OOGenToken()
   {
      super();
   }


   /**
    * Constructor for class OOGenToken
    *
    * @param statement  No description provided
    */
   public OOGenToken (OOStatement statement)
   {
      super();
      //   setText (text);
      appendStatement (statement);
   }


   /**
    * Constructor for class OOGenToken
    *
    * @param statement    No description provided
    * @param sectionName  No description provided
    */
   public OOGenToken (OOStatement statement, String sectionName)
   {
      super();
      //setText (text);
      appendStatement (statement);
      setSectionName (sectionName);
   }


   /**
    * Creates a new token pair for with explicite first and last token and inserts it in the
    * list behind prevToken before: prevElement <---> nextElement after : prevElement <--->
    * this <---> nextElement
    *
    * @param prevToken  indicates the position in the list
    * @return           the first token
    * @see              #append
    * @see              #prepend
    */
   public static OOGenToken createNewList (OOGenToken prevToken)
   {
      OOGenToken oldNextToken;
      OOGenToken firstToken = createFirstToken();
      OOGenToken lastToken = createLastToken();

      if (prevToken != null)
      {
         oldNextToken = prevToken.getNext();
      }
      else
      {
         oldNextToken = null;
      }

      firstToken.setNext (lastToken);
      firstToken.setPrev (prevToken);
      lastToken.setNext (oldNextToken);

      return firstToken;
   }

   //    private String text;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param token  No description provided
    */
   public static void showStatements (OOGenToken token)
   {
      while (token != null)
      {
         token.showStatements();
         token = token.getNext();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void showStatements()
   {
      Iterator iter = iteratorOfStatement();

      while (iter.hasNext())
      {
         log.error (this + ".statement." +  (iter.next()));
      }
   }


   /**
    * Get the sourceCode attribute of the OOGenToken object
    *
    * @return   The sourceCode value
    */
   public String getSourceCode()
   {
      Iterator iter = iteratorOfStatement();
      StringBuffer text = new StringBuffer();
      CodeGenVisitor tmpVisitor = CodeGenFactory.get().getCurrentVisitor();
      OOGenVisitor v = null;
      OOStatement currentStatement = null;
      String tmpCode = null;

      if (tmpVisitor instanceof OOGenVisitor)
      {
         v = (OOGenVisitor) tmpVisitor;
      }

      if (v != null)
      {
         while (iter.hasNext())
         {
            currentStatement = (OOStatement) iter.next();

            tmpCode = currentStatement.getSourceCode (v);
            if (tmpCode != null)
            {
               text.append (tmpCode);
               text.append ("\n");
            }
         }
      }

      return text.toString();
   }


   /**
    * Get the sourceCodeForTokens attribute of the OOGenToken class
    *
    * @param token  No description provided
    * @param stop   No description provided
    * @return       The sourceCodeForTokens value
    */
   public static String getSourceCodeForTokens (OOGenToken token, OOGenToken stop)
   {
      StringBuffer buf = new StringBuffer();

      // skip leading null's and empty lines
      while (token != stop)
      {
         if (token.getSourceCode() != null &&
            token.getSourceCode().trim().length() != 0)
         {
            break;
         }
         token = token.getNext();
      }
      while (token != stop)
      {
         buf.append (token.getSourceCode());
         token = token.getNext();
      }
      return buf.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String sectionName = "";


   /**
    * Get the sectionName attribute of the OOGenToken object
    *
    * @return   The sectionName value
    */
   public String getSectionName()
   {
      return sectionName;
   }


   /**
    * Sets the sectionName attribute of the OOGenToken object
    *
    * @param sectionName  The new sectionName value
    * @return             No description provided
    */
   public String setSectionName (String sectionName)
   {
      this.sectionName = sectionName;
      return this.sectionName;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param width  No description provided
    * @return       No description provided
    */
   public String toString (int width)
   {
      StringBuffer buf = new StringBuffer (width);
      buf.append ("|" + this.sectionName + "|");

      if (this.getSourceCode() == null)
      {
         buf.append (">no text<|");
      }
      else
      {
         buf.append (Utility.textToWidth (this.getSourceCode(), width));
         buf.append ("|");
      }
      return buf.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int startRow = -1;


   /**
    * Get the startRow attribute of the OOGenToken object
    *
    * @return   The startRow value
    */
   public int getStartRow()
   {
      return startRow;
   }


   /**
    * Sets the startRow attribute of the OOGenToken object
    *
    * @param startRow  The new startRow value
    * @return          No description provided
    */
   public int setStartRow (int startRow)
   {
      this.startRow = startRow;
      return startRow;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int startColumn = -1;


   /**
    * Get the startColumn attribute of the OOGenToken object
    *
    * @return   The startColumn value
    */
   public int getStartColumn()
   {
      return startColumn;
   }


   /**
    * Sets the startColumn attribute of the OOGenToken object
    *
    * @param startColumn  The new startColumn value
    * @return             No description provided
    */
   public int setStartColumn (int startColumn)
   {
      this.startColumn = startColumn;
      return startColumn;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient OOGenToken prev; // reverse OOGenToken next;


   /**
    * Get the prev attribute of the OOGenToken object
    *
    * @return   The prev value
    */
   public OOGenToken getPrev()
   {
      return prev;
   }


   /**
    * Sets the prev attribute of the OOGenToken object
    *
    * @param prev  The new prev value
    */
   public void setPrev (OOGenToken prev)
   {
      if (this.prev != prev)
      { // new partner

         OOGenToken oldPrev = this.prev;
         if (this.prev != null)
         { // inform old partner

            this.prev = null;
            oldPrev.setNext (null);
         }
         this.prev = prev;
         if (prev != null)
         { // inform new partner

            this.prev.setNext (this);
         }
         // firePropertyChange ("prev", oldPrev, prev);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient OOGenToken next; // reverse OOGenToken prev;


   /**
    * Get the next attribute of the OOGenToken object
    *
    * @return   The next value
    */
   public OOGenToken getNext()
   {
      return next;
   }


   /**
    * Sets the next attribute of the OOGenToken object
    *
    * @param next  The new next value
    */
   public void setNext (OOGenToken next)
   {
      if (this.next != next)
      { // new partner

         OOGenToken oldNext = this.next;
         if (this.next != null)
         { // inform old partner

            this.next = null;
            oldNext.setPrev (null);
         }
         this.next = next;
         if (next != null)
         { // inform new partner

            this.next.setPrev (this);
         }
         // firePropertyChange ("next", oldNext, next);
      }
   }


   /**
    * <pre>
    *             0..1                             0..1
    * OOGenToken --------------------------------------- ASGElement
    *             firstOOGenToken     firstAsgElement
    * </pre>
    */
   private ASGElement firstAsgElement;


   /**
    * @param value  The new firstAsgElement value
    * @return       No description provided
    * @see          #firstAsgElement
    */
   public boolean setFirstAsgElement (ASGElement value)
   {
      if (this.firstAsgElement != value)
      {
         if (this.firstAsgElement != null)
         {
            ASGElement oldValue = this.firstAsgElement;
            this.firstAsgElement = null;
            oldValue.setFirstOOGenToken (null);
         }
         this.firstAsgElement = value;
         if (value != null)
         {
            this.firstAsgElement.setFirstOOGenToken (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The firstAsgElement value
    * @see      #firstAsgElement
    */
   public ASGElement getFirstAsgElement()
   {
      return this.firstAsgElement;
   }


   /**
    * <pre>
    *             0..1                           0..1
    * OOGenToken ------------------------------------- ASGElement
    *             lastOOGenToken     lastAsgElement
    * </pre>
    */
   private ASGElement lastAsgElement;


   /**
    * @param value  The new lastAsgElement value
    * @return       No description provided
    * @see          #lastAsgElement
    */
   public boolean setLastAsgElement (ASGElement value)
   {
      if (this.lastAsgElement != value)
      {
         if (this.lastAsgElement != null)
         {
            ASGElement oldValue = this.lastAsgElement;
            this.lastAsgElement = null;
            oldValue.setLastOOGenToken (null);
         }
         this.lastAsgElement = value;
         if (value != null)
         {
            this.lastAsgElement.setLastOOGenToken (this);
         }

         return true;
      }

      return false;
   }


   /**
    * @return   The lastAsgElement value
    * @see      #lastAsgElement
    */
   public ASGElement getLastAsgElement()
   {
      return this.lastAsgElement;
   }


   /**
    * <pre>
    *              n      hasStatements      0..1
    * OOStatement -------------------------------- OOGenToken
    *              statement   {ordered,}   token
    * </pre>
    */
   private FLinkedList statement;


   /**
    * @param value  No description provided
    * @return       No description provided
    * @see          #statement
    */
   public boolean hasInStatement (OOStatement value)
   {
      return  ( (this.statement != null) &&
          (value != null) &&
         this.statement.contains (value));
   }


   /**
    * @return   No description provided
    * @see      #statement
    */
   public Iterator iteratorOfStatement()
   {
      return  ( (this.statement == null)
         ? FEmptyIterator.get()
         : this.statement.iterator());
   }


   /**
    * @return   No description provided
    * @see      #statement
    */
   public int sizeOfStatement()
   {
      return  ( (this.statement == null)
         ? 0
         : this.statement.size());
   }


   /**
    * @param value  The object added.
    * @return       No description provided
    * @see          #statement
    */
   public boolean addToStatement (OOStatement value)
   {
      boolean changed = false;
      if (value != null && !hasInStatement (value))
      {
         if (this.statement == null)
         {
            this.statement = new FLinkedList(); // or FTreeSet () or FLinkedList ()
         }
         changed = this.statement.add (value);
         if (changed)
         {
            value.setToken (this);
         }
      }
      return changed;
   }


   /**
    * @param value  No description provided
    * @return       No description provided
    * @see          #statement
    */
   public boolean prependStatement (OOStatement value)
   {
      boolean changed = false;
      if (value != null && !hasInStatement (value))
      {
         if (this.statement == null)
         {
            this.statement = new FLinkedList(); // or FTreeSet () or FLinkedList ()
         }
         this.statement.addFirst (value);
         if (changed)
         {
            value.setToken (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    */
   public void appendStatement (Object value)
   {
      if (value instanceof OOStatement)
      {
         appendStatement ((OOStatement) value);
      }
      else if (value instanceof OOStatement[])
      {
         appendStatement ((OOStatement[]) value);
      }
      else if (value instanceof LinkedList)
      {
         appendStatement (OOStatement.toArray ((LinkedList) value));
      }
      else
      {
         throw new UnsupportedOperationException ("value=" + value);
      }
   }


   /**
    * append all statements
    *
    * @param values  an <code>OOStatement[]</code> value
    */
   public void appendStatement (OOStatement[] values)
   {
      int i;

      for (i = 0; i < values.length; i++)
      {
         appendStatement (values[i]);
      }
   }


   /**
    * @param value  No description provided
    * @return       No description provided
    * @see          #statement
    */
   public boolean appendStatement (OOStatement value)
   {
      return addToStatement (value);
   }


   /**
    * @param value  No description provided
    * @return       No description provided
    * @see          #statement
    */
   public boolean removeFromStatement (OOStatement value)
   {
      boolean changed = false;
      if ( (this.statement != null) &&  (value != null))
      {
         changed = this.statement.remove (value);
         if (changed)
         {
            value.setToken (null);
         }
      }
      return changed;
   }


   /**
    * @see   #statement
    */
   public void removeAllFromStatement()
   {
      OOStatement tmpValue;
      Iterator iter = this.iteratorOfStatement();
      while (iter.hasNext())
      {
         tmpValue = (OOStatement) iter.next();
         this.removeFromStatement (tmpValue);
      }

   }


   /**
    * @param index  No description provided
    * @return       The statementAt value
    * @see          #statement
    */
   public OOStatement getStatementAt (int index)
   {
      if (index >= 0 && index < sizeOfStatement())
      {
         return (OOStatement) this.statement.get (index);
      }
      else
      {
         throw new IllegalArgumentException ("getStatementAt(" + index + ")");
      }

   }


   /**
    * @param elem  No description provided
    * @return      No description provided
    * @see         #statement
    */
   public int indexOfStatement (OOStatement elem)
   {
      return  ( (this.statement == null)
         ? -1
         : this.statement.indexOf (elem));
   }


   /**
    * @param elem  No description provided
    * @return      No description provided
    * @see         #statement
    */
   public int lastIndexOfStatement (OOStatement elem)
   {
      return  ( (this.statement == null)
         ? -1
         : this.statement.lastIndexOf (elem));
   }


   /**
    * @param lowerBound  No description provided
    * @return            No description provided
    * @see               #statement
    */
   public Iterator iteratorOfStatement (OOStatement lowerBound)
   {
      Iterator result = FEmptyIterator.get();
      if (statement == null)
      {
         result = FEmptyIterator.get();
      }
      else if (statement != null && lowerBound != null)
      {
         int index = statement.indexOf (lowerBound) + 1;
         result = statement.listIterator (index);
      }
      else if (statement != null && lowerBound == null)
      {
         result = statement.listIterator (0);
      }
      return result;
   }


   /**
    * @param leftObject   No description provided
    * @param rightObject  No description provided
    * @return             The beforeOfStatement value
    * @see                #statement
    */
   public boolean isBeforeOfStatement (OOStatement leftObject, OOStatement rightObject)
   {
      if (statement == null)
      {
         return false;
      }
      else
      {
         return statement.isBefore (leftObject, rightObject);
      }

   }


   /**
    * @param leftObject   No description provided
    * @param rightObject  No description provided
    * @return             The afterOfStatement value
    * @see                #statement
    */
   public boolean isAfterOfStatement (OOStatement leftObject, OOStatement rightObject)
   {
      if (statement == null)
      {
         return false;
      }
      else
      {
         return statement.isAfter (leftObject, rightObject);
      }

   }


   /**
    * @return   The firstOfStatement value
    * @see      #statement
    */
   public OOStatement getFirstOfStatement()
   {
      if (statement == null)
      {
         return null;
      }
      else
      {
         return (OOStatement) statement.getFirst();
      }

   }


   /**
    * @return   The lastOfStatement value
    * @see      #statement
    */
   public OOStatement getLastOfStatement()
   {
      if (statement == null)
      {
         return null;
      }
      else
      {
         return (OOStatement) statement.getLast();
      }

   }


   /**
    * @param object  No description provided
    * @return        The nextOfStatement value
    * @see           #statement
    */
   public OOStatement getNextOfStatement (OOStatement object)
   {
      if (statement == null)
      {
         return null;
      }
      else
      {
         return (OOStatement) statement.getNextOf (object);
      }

   }


   /**
    * @param object  No description provided
    * @param index   No description provided
    * @return        The nextIndexOfStatement value
    * @see           #statement
    */
   public OOStatement getNextIndexOfStatement (OOStatement object, int index)
   {
      if (statement == null)
      {
         return null;
      }
      else
      {
         return (OOStatement) statement.getNextOf (object, index);
      }

   }


   /**
    * @param object  No description provided
    * @return        The previousOfStatement value
    * @see           #statement
    */
   public OOStatement getPreviousOfStatement (OOStatement object)
   {
      if (statement == null)
      {
         return null;
      }
      else
      {
         return (OOStatement) statement.getPreviousOf (object);
      }

   }


   /**
    * @param object  No description provided
    * @param index   No description provided
    * @return        The previousIndexOfStatement value
    * @see           #statement
    */
   public OOStatement getPreviousIndexOfStatement (OOStatement object, int index)
   {
      if (statement == null)
      {
         return null;
      }
      else
      {
         return (OOStatement) statement.getPreviousOf (object, index);
      }

   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param sectionName  No description provided
    * @return             No description provided
    */
   public OOGenToken findSection (String sectionName)
   {
      OOGenToken startToken = this;
      while ( (startToken != null)
         &&  ( (startToken.getSectionName() == null)
         ||  (!startToken.getSectionName().equals (sectionName))))
      {
         startToken = startToken.getNext();
      }
      return startToken;
   }


   /**
    * Return the last token of the section named sectionName.
    *
    * @param sectionName  the name of the new section
    * @param lastToken    marks the end
    * @return             the last token of the section
    */
   public OOGenToken skipSection (String sectionName, OOGenToken lastToken)
   {
      OOGenToken currentToken = this;
      OOGenToken nextToken = currentToken.getNext();

      while ( (nextToken != null)
         &&  (nextToken != lastToken)
         &&  (nextToken.getSectionName().equals (sectionName))
         &&  (nextToken.getFirstAsgElement() == null))
      {
         currentToken = nextToken;
         nextToken = currentToken.getNext();
      }
      return currentToken;
   }


   /**
    * Returns the token just in front of the next Firsttoken or in front of the lastToken.
    *
    * @param lastToken  marks the end
    * @return           the new position
    */

   public OOGenToken skipToPrevNextFirstToken (OOGenToken lastToken)
   {
      OOGenToken currentToken = this;
      OOGenToken nextToken = currentToken.getNext();

      while ( (nextToken != null)
         &&  (nextToken != lastToken)
         &&  (nextToken.getFirstAsgElement() == null))
      {
         currentToken = nextToken;
         nextToken = currentToken.getNext();
      }
      return currentToken;
   }


   /**
    * Returns the token in front of the specified section and deletes all tokens on the way
    * are deleted.
    *
    * @param sectionName  the name of the new section
    * @param lastToken    the last token in the current list
    */
   public void deleteToSection (String sectionName, OOGenToken lastToken)
   {
      OOGenToken currentToken = this;
      OOGenToken nextToken = currentToken.getNext();

      if (sectionName == null)
      {
         // deletes all to the last token
         sectionName = LAST_TOKEN_NAME;
      }

      while ( (nextToken != null)
         &&  (nextToken != lastToken)
         && ! (nextToken.getSectionName().equals (sectionName))
         &&  (nextToken.getFirstAsgElement() == null))
      {
         nextToken.removeYouFromList();
         nextToken.removeYou();
         nextToken = currentToken.getNext();
      }
   }


   /**
    * Starts the new section
    *
    * @param section    the name of the new section
    * @param lastToken  the last token of the list
    * @return           the sections name
    */
   public String newSection (String section, OOGenToken lastToken)
   {
      deleteToSection (section, lastToken);
      return section;
   }


   /**
    * Creates two new tokens behind the current and sets the section name of the first to 'sectionName'Top
    * and of the last to 'sectionName'Bottom.
    *
    * @param sectionName  the name of the section
    * @return             the sections name
    */
   public String insertTopAndBottomToken (String sectionName)
   {
      this.append (new OOGenToken (null, sectionName + "Top"));
      this.getNext().append (new OOGenToken (null, sectionName + "Bottom"));
      return sectionName;
   }


   /**
    * Get the topToken attribute of the OOGenToken object
    *
    * @param sectionName  No description provided
    * @return             The topToken value
    */
   public OOGenToken getTopToken (String sectionName)
   {
      if (!this.getNext().getSectionName().equals (sectionName + "Top"))
      {
         throw new RuntimeException ("Error generating top token");
      }
      return this.getNext();
   }


   /**
    * Get the bottomToken attribute of the OOGenToken object
    *
    * @param sectionName  No description provided
    * @return             The bottomToken value
    */
   public OOGenToken getBottomToken (String sectionName)
   {
      if (!this.getNext().getSectionName().equals (sectionName + "Bottom"))
      {
         throw new RuntimeException ("Error generating top token");
      }
      return this.getNext();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param sectionName  No description provided
    * @return             No description provided
    */
   public OOGenToken insertNewToken (String sectionName)
   {
      this.append (new OOGenToken (null, sectionName));
      return this.getNext();
   }


   /**
    * Gets the next token in the section or creates a new one behind me. This method is useful
    * because at many times you need it. For example see generateJava() methods.
    *
    * @param sectionName  the name of the section
    * @param lastToken    the last token in the current list
    * @return             the next token if exists or a new one
    */
   public OOGenToken getNextOrAppend (String sectionName, OOGenToken lastToken)
   {
      OOGenToken nextToken = this.getNext();
      if ( (nextToken == null)
         ||  (nextToken == lastToken)
         || ! (nextToken.getSectionName().equals (sectionName))
         ||  (nextToken.getFirstAsgElement() != null))
      {
         append (new OOGenToken (null, sectionName));
      }
      return getNext();
   }


   /**
    * Cuts all tokens from firstToken to lastToken out of the list and inserts them behind
    * prevToken.
    *
    * @param firstToken  marks the first token in the list
    * @param lastToken   marks the last token in the list
    * @param prevToken   marks the new position for the list
    */
   public static void cutAndPaste (OOGenToken firstToken, OOGenToken lastToken, OOGenToken prevToken)
   {
      OOGenToken oldPrevFirstToken = firstToken.getPrev();
      OOGenToken oldNextLastToken = lastToken.getNext();
      OOGenToken oldNextPrevToken;

      if (prevToken != null)
      {
         oldNextPrevToken = prevToken.getNext();
      }
      else
      {
         oldNextPrevToken = null;
      }

      // cut out the list
      if (oldPrevFirstToken != null)
      {
         oldPrevFirstToken.setNext (oldNextLastToken);
      }
      else
      {
         if (oldNextLastToken != null)
         {
            oldNextLastToken.setPrev (oldPrevFirstToken);
         }
      }

      // paste the tokens behind prevToken
      firstToken.setPrev (prevToken);
      lastToken.setNext (oldNextPrevToken);
   }


   /**
    * Inserts the new token behind me. before: prevElement <---> this <---> nextElement after
    * : prevElement <---> this <---> newToken <---> nextElement
    *
    * @param newToken  the new token to insert in the list
    */
   public void append (OOGenToken newToken)
   {
      if (newToken != null)
      {
         newToken.setNext (getNext());
      }
      setNext (newToken);
   }


   /**
    * Inserts the new token in front of me. before: prevElement <---> this <---> nextElement
    * after : prevElement <---> newToken <---> this <---> nextElement
    *
    * @param newToken  the new token to insert in the list
    */
   public void prepend (OOGenToken newToken)
   {
      if (newToken != null)
      {
         newToken.setPrev (getPrev());
      }
      setPrev (newToken);
   }


   /**
    * Removes this element from the list. before: prevElement <---> this <---> nextElement
    * after : prevElement <--------------> nextElement this
    */
   public void removeYouFromList()
   {
      OOGenToken prev = getPrev();
      OOGenToken next = getNext();

      if (prev != null)
      {
         // has a previous element
         prev.setNext (next);
      }
      else
      {
         if (next != null)
         {
            // or has a next element
            next.setPrev (prev);
         }
         else
         {
            // or I am the only token in the list
            setPrev (null);
            setNext (null);
         }
      }
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "OOGenToken[" + this.hashCode() + "," + this.sectionName + "]";
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      Iterator iter = iteratorOfStatement();
      OOStatement tmpStatement = null;

      while (iter.hasNext())
      {
         tmpStatement = (OOStatement) iter.next();
         removeFromStatement (tmpStatement);

         tmpStatement.removeYou();
      }

      setFirstAsgElement (null);
      setLastAsgElement (null);
      setNext (null);
      setPrev (null);
   }
}

/*
 * $Log: OOGenToken.java,v $
 * Revision 1.26.2.1  2005/05/02 11:51:03  creckord
 * Fix for Multilinks
 * Small change in CoObRA change filtering
 *
 */
