/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.*;

import de.uni_paderborn.fujaba.basic.SourceCodeFactory;
import de.uni_paderborn.fujaba.basic.UnifiedClassLoader;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.StandardComparator;


/**
 * @author    <a href="mailto:creckord@upb.de">Carsten Reckord</a>
 * @version   $Revision: 1.13.2.2 $
 */
public class UMLClassBodyRemoveYouStrategy extends UMLClassBodyStrategy
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlClass  No description provided
    */
   public void createContents (FClass umlClass)
   {
      if (mustCreateRemoveYouMethod (umlClass))
      {
         generateRemoveYouMethod (umlClass);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param clazz  No description provided
    * @return       No description provided
    */
   public static boolean mustCreateRemoveYouMethod (FClass clazz)
   {
      boolean result = false;

      FStereotype interfaceType = UMLStereotypeManager.get().
         getFromStereotypes (UMLStereotypeManager.INTERFACE);

      FStereotype referenceType = UMLStereotypeManager.get().
         getFromStereotypes (UMLStereotypeManager.REFERENCE);

      FStereotype importType = UMLStereotypeManager.get().
         getFromStereotypes (UMLStereotypeManager.IMPORT);

      // don't create method if class is an interface or a reference
      if (clazz != null &&
         ! (clazz.hasInStereotypes (interfaceType) ||
         clazz.hasInStereotypes (referenceType) ||
         clazz.hasInStereotypes (importType)))
      {
         result =  (getRemoveRoles (clazz).size() > 0) &&  (clazz.getFromFMethodsByShortName ("removeYou") == null);
      }
      return result;
   } // mustCreateRemoveYouMethod


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param clazz  No description provided
    */
   protected void generateRemoveYouMethod (FClass clazz)
   {
      StringBuffer methodBody = new StringBuffer();
      SourceCodeFactory factory = UMLProject.getSourceCodeFactory();

      FRole tmpRole;
      FRole tmpPartnerRole;

      boolean toOneAccess;
      boolean isQualified;
      boolean isPartnerQualified;
      boolean isInternalQualified;
      boolean callSuperMethod = false;

      Collection removeRoles = getRemoveRoles (clazz);
      Iterator iter = removeRoles.iterator();
      while (iter.hasNext())
      {
         callSuperMethod = true;
         tmpRole = (FRole) iter.next();

         tmpPartnerRole = tmpRole.getFPartnerRole();
         FQualifier qualifier = tmpRole.getFQualifier();
         FQualifier partnerQualifier = tmpPartnerRole.getFQualifier();

         if (qualifier != null)
         {
            isQualified = true;
         }
         else
         {
            isQualified = false;
         } // end of else

         isPartnerQualified = false;

         if (partnerQualifier != null)
         {
            isPartnerQualified = true;

            isInternalQualified = !partnerQualifier.isExternalQualifier();
         }
         else
         {
            isInternalQualified = false;
         }

         if (isQualified)
         {
            toOneAccess = false;
         }
         else
         {
            toOneAccess =  ( (tmpPartnerRole.getFCard() == null) ||
                ( (tmpPartnerRole.getFCard() != null) &&
                (tmpPartnerRole.getFCard().getUpperBound() == 1)));
         }

         FClass target = tmpPartnerRole.getFTarget();
         if (target != null)
         {
            methodBody.append (factory.removeYouBody
                (tmpPartnerRole.getAttrName(),
               target.getName(),
               toOneAccess,
               isQualified,
               isPartnerQualified,
               isInternalQualified,
               tmpRole.getAdornment()
               == FRole.COMPOSITION));
         }
      }
      callSuperMethod = callSuperMethod && alreadyImplementsRemoveYou (clazz);

      if (callSuperMethod)
      {
         methodBody.append ("super.").
            append (factory.getAccessMethodName ("", SourceCodeFactory.REMOVE_YOU_METHOD)).
            append ( (" (")).
            append ( (")")).
            append (";\n\n");
      }

      FMethod removeYouMethod = new UMLMethod (false);
      removeYouMethod.setGenerated (true);
      removeYouMethod.setName (factory.getAccessMethodName ("", SourceCodeFactory.REMOVE_YOU_METHOD));
      removeYouMethod.setVisibility (FDeclaration.PUBLIC);
      removeYouMethod.setResultType (UMLProject.get().getFromBaseTypes (UMLBaseTypes.VOID));

      removeYouMethod.setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
      removeYouMethod.setMethodBody (methodBody.toString(), true);

      getClassHandler().generateFMethod (removeYouMethod);
      removeYouMethod.removeYou();
   }


   /**
    * Get the removeRoles attribute of the UMLClassBodyRemoveYouStrategy class
    *
    * @param umlClass  No description provided
    * @return          The removeRoles value
    */
   protected static Collection getRemoveRoles (FClass umlClass)
   {
      Enumeration enumeration = umlClass.elementsOfRoles();
      FRole tmpRole;

      Collection removeRoles = new TreeSet (new FRoleComparator());
      TreeSet roleNames = new TreeSet (new StandardComparator());
      while (enumeration.hasMoreElements())
      {
         tmpRole = (FRole) enumeration.nextElement();
         String tmpName = tmpRole.getFPartnerRole().getName();
         if (tmpName != null && !roleNames.contains (tmpName))
         {
            roleNames.add (tmpName);

            if (!tmpRole.getFAssoc().isVirtualPath() &&
               tmpRole.getAdornment() != FRole.REFERENCE &&
               tmpRole.getFPartnerRole() != null)
            {
               removeRoles.add (tmpRole);
            }
         }
      }
      return removeRoles;
   }


   /**
    * @param clazz  FClass of interest
    * @return       true if a superclass of the class already implements a removeYou method
    */
   public static boolean alreadyImplementsRemoveYou (FClass clazz)
   {
      Enumeration subclassEnum = clazz.elementsOfRevSubclass();
      while (subclassEnum.hasMoreElements())
      {
         FClass tmpClass =  ((UMLGeneralization) subclassEnum.nextElement()).getSuperclass();

         if (mustCreateRemoveYouMethod (tmpClass))
         {
            return true;
         }
         else if (tmpClass.getFromFMethodsByShortName ("removeYou") != null)
         {
            return true;
         }
         else if (tmpClass.isReference())
         {
            //maybe we know the class (is in classpath)
            boolean maybe = false;
            try
            {
               maybe = UnifiedClassLoader.get().loadClass (tmpClass.getFullClassName()).getMethod ("removeYou", null) != null;
            }
            catch (NoSuchMethodException e)
            {
               maybe = false;
            }
            catch (ClassNotFoundException e)
            {
               maybe = false;
            }
            if (maybe)
            {
               return true;
            }
         }
         else if (alreadyImplementsRemoveYou (tmpClass))
         {
            return true;
         }
      }
      return false;
   }


   /**
    * This Comparator can be used to compare two <code>Object</code>s of type
    * <code>FRole</code>.
    * This <code>Comparator</code> sorts roles the following way:
    * <ul>
    * <li> <code>FRole</code>s adjacent to a assoc qulified by a role lower than <code>FRole</code>s of normal assocs
    * <li> <code>FRole</code>s
    *  </ul>
    *
    * @author    basilb
    * @version   $Revision: 1.13.2.2 $ $Date: 2006/06/07 09:36:39 $
    */
   protected static class FRoleComparator implements Comparator
   {

      public int compare (Object o1, Object o2)
      {
         if (o1.equals (o2))
         {
            return 0;
         }
         UMLRole role1 = (UMLRole) o1;
         UMLRole role2 = (UMLRole) o2;

         //a role is a role of special interest if its partnerrole is    qualified by a role
         boolean role1IsSpecial = role1.getPartnerRole().getQualifier()
            != null && role1.getPartnerRole().getQualifier().getQualifiedRole() != null;
         boolean role2IsSpecial = role2.getPartnerRole().getQualifier()
            != null && role2.getPartnerRole().getQualifier().getQualifiedRole() != null;

         if (role1IsSpecial && !role2IsSpecial)
         {
            return -1;
         }
         else if (!role1IsSpecial && role2IsSpecial)
         {
            return 1;
         }
         else if (! (role1IsSpecial || role2IsSpecial))
         {
            //both roles belong to unqualified Assocs, in this case the   sorting order
            //is not relevant but must be unique
            return role1.hashCode() - role2.hashCode();
         }
         else
         {
            UMLRole qualiRole1 =
               role1.getPartnerRole().getQualifier().getQualifiedRole();
            UMLRole qualiRole2 =
               role2.getPartnerRole().getQualifier().getQualifiedRole();
            if (qualiRole2 == role1.getFPartnerRole() && qualiRole1 !=
               role2.getFPartnerRole())
            {
               //role1 is role2' assoc's qualifier. We have to remove  role2 first
               return 1;
            }
            else if (qualiRole1 == role2.getPartnerRole() &&
               qualiRole2 != role1.getPartnerRole())
            {
               //role2 is role1' assoc's qualifier. We have to remove   role1 first
               return -1;
            }
            else if (qualiRole1 != role2.getPartnerRole() &&
               qualiRole2 != role1.getPartnerRole())
            {
               //the roles aren't associated by a direct qualification relation
               //but perhaps the roles they are qualified by, are.
               return compare (qualiRole1.getPartnerRole(), qualiRole2.getPartnerRole());
            }
            else if (qualiRole1 == role2.getPartnerRole() && qualiRole2 == role1.getPartnerRole())
            {
               //role1 is qualified by role2 and vice versa
               //don't know how to resolve this problem
               return role1.hashCode() - role2.hashCode();
            }
         }
         return role1.hashCode() - role2.hashCode();
      }

   }
}

/*
 * $Log: UMLClassBodyRemoveYouStrategy.java,v $
 * Revision 1.13.2.2  2006/06/07 09:36:39  mtt
 * according to the uml specification, the removeYou()-call should be generated for a composition (not an aggregation)
 *
 */
