/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen;

import java.util.Iterator;

import org.apache.log4j.Logger;

import de.tu_bs.coobra.ObjectChange;
import de.tu_bs.coobra.ObjectChangeAwareHelper;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.UMLStereotypeManager;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FLinkedList;


/**
 * Class UMLClassOOHandler
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.31.2.6 $
 */
public class UMLClassOOHandler extends OOGenStrategyHandler
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLClassOOHandler.class);


   /**
    * Default Constructor
    */
   public UMLClassOOHandler()
   {
      super();
   }


   /**
    * Get the responsible attribute of the UMLClassOOHandler object
    *
    * @param incr  No description provided
    * @return      The responsible value
    */
   public boolean isResponsible (FElement incr)
   {
      return  (incr instanceof FClass);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final boolean needToken()
   {
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr       No description provided
    * @param prevToken  No description provided
    * @param param      No description provided
    * @return           No description provided
    */
   public OOGenToken generateSourceCode (FElement incr,
                                         OOGenToken prevToken,
                                         Object param[])
   {
      FClass theClass = (FClass) incr;

      ObjectChange.pushCause (ObjectChangeAwareHelper.CAUSE_UNDOREDO);
      try
      {
         //create nasty stuff that will not be saved
      }
      finally
      {
         ObjectChange.popCause (ObjectChangeAwareHelper.CAUSE_UNDOREDO);
      }

      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();

      if (log.isDebugEnabled())
      {
         log.debug (this + ".generate(theClass=" + theClass + ")");
      }

      // generate class declaration
      client.newLine();

      generateFClassDeclaration (theClass);

      client.generateClassStartBlock();

      generateClassBody (theClass);

      client.generateClassEndBlock();

      // has no tokens
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    */
   public final void generateFClassDeclaration (FClass theClass)
   {
      if (log.isInfoEnabled())
      {
         log.info (this + ".generateFClassDeclaration(" + theClass + ")");
      }

      OOGenStrategyClient client = (OOGenStrategyClient) getClientOfChain();

      // generate class commentary
      FCommentary fCommentary = theClass.getFComment();
      if (fCommentary != null)
      {
         String commentaryText;
         if (theClass.isParsed())
         {
            commentaryText = fCommentary.getText();
         }
         else
         {
            commentaryText = client.createFClassCommentary (fCommentary);
         }
         client.appendFClassCommentary (commentaryText);
      }

      // generate class declaration
      String declString = null;
      FStereotype interfaceStereotype = UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.INTERFACE);
      if (theClass.hasInStereotypes (interfaceStereotype))
      {
         declString = client.createInterfaceDeclaration (theClass);
      }
      else
      {
         declString = client.createClassDeclaration (theClass);
      }

      client.appendFClassDeclaration (declString);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theClass  No description provided
    */
   public void generateClassBody (FClass theClass)
   {
      if (log.isInfoEnabled())
      {
         log.info (this + ".generateClassBody(" + theClass + ")");
      }

      // generate code for attributes and methods in the following order:
      // 1. constants (static final) in alphabetical order
      // 2. constructors in alphabetical order
      // 3. static attributes in alphabetical order
      // 4. static methods in alphabetical order
      // 5. attributes in alphabetical order
      // 5.1 access methods of current attribute in alphabetical order
      // 6. roles in alphabetical order
      // 6.1 access methods of current role in alphabetical order
      // 7. all missing methods in alphabetical order
      // 8. last but not least removeYou() method
      Iterator strategies = iteratorOfBodyStrategies();
      while (strategies.hasNext())
      {
         UMLClassBodyStrategy strategy = (UMLClassBodyStrategy) strategies.next();
         strategy.createContents (theClass);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theAttr  No description provided
    */
   public final void generateFAttr (FAttr theAttr)
   {
      getClientOfChain().generateSourceCodeFor (theAttr, null, null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theRole  No description provided
    */
   public final void generateFRole (FRole theRole)
   {
      getClientOfChain().generateSourceCodeFor (theRole, null, null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param theMethod  No description provided
    */
   public final void generateFMethod (FMethod theMethod)
   {
      getClientOfChain().generateSourceCodeFor (theMethod, null, null);
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return "UMLClassOOHandler[]";
   }


   /**
    * <pre>
    *                       n                Assoc                0..1
    * UMLClassBodyStrategy -------------------------------------------- UMLClassOOHandler
    *                       bodyStrategies   {ordered,}   classHandler
    * </pre>
    */
   private FLinkedList bodyStrategies;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    */
   public void appendStrategy (UMLClassBodyStrategy value)
   {
      addToBodyStrategies (value);
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToBodyStrategies (UMLClassBodyStrategy value)
   {
      boolean changed = false;
      if (value != null && !hasInBodyStrategies (value))
      {
         if (this.bodyStrategies == null)
         {
            this.bodyStrategies = new FLinkedList(); // or FTreeSet () or FLinkedList ()
         }
         changed = this.bodyStrategies.add (value);
         if (changed)
         {
            value.setClassHandler (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInBodyStrategies (UMLClassBodyStrategy value)
   {
      return  ( (this.bodyStrategies != null) &&
          (value != null) &&
         this.bodyStrategies.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfBodyStrategies()
   {
      return  ( (this.bodyStrategies == null)
         ? FEmptyIterator.get()
         : this.bodyStrategies.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromBodyStrategies()
   {
      UMLClassBodyStrategy tmpValue;
      Iterator iter = this.iteratorOfBodyStrategies();
      while (iter.hasNext())
      {
         tmpValue = (UMLClassBodyStrategy) iter.next();
         this.removeFromBodyStrategies (tmpValue);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromBodyStrategies (UMLClassBodyStrategy value)
   {
      boolean changed = false;
      if ( (this.bodyStrategies != null) &&  (value != null))
      {
         changed = this.bodyStrategies.remove (value);
         if (changed)
         {
            value.setClassHandler (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfBodyStrategies()
   {
      return  ( (this.bodyStrategies == null)
         ? 0
         : this.bodyStrategies.size());
   }
}

/*
 * $Log: UMLClassOOHandler.java,v $
 * Revision 1.31.2.6  2006/06/07 09:13:31  creckord
 * - UMLTransitionGuard can be null instead of UMLTransitionGuard.NONE
 * - old "repair assoc" code removed (access methods are not kept, so repairing them is no longer needed)
 * - loop bends for assocs are removed when loop is removed
 *
 */
