/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.codegen.digester.rules;

import java.lang.reflect.Method;

import de.uni_paderborn.fujaba.codegen.digester.XDigester;
import de.uni_paderborn.fujaba.codegen.digester.XRule;


/**
 * Rule implementation that calls a method on the top (parent) object, passing the (top-1)
 * (child) object as an argument.
 *
 * @author    Craig McClanahan
 * @version   $Revision: 1.9 $ $Date: 2004/10/29 13:46:53 $
 */
public class XSetTopRule extends XRule
{
   /**
    * Construct a "set parent" rule with the specified method name. The parent method's argument
    * type is assumed to be the class of the child object.
    *
    * @param digester    The associated Digester
    * @param methodName  Method name of the parent method to call
    */
   public XSetTopRule (XDigester digester, String methodName)
   {
      this (digester, methodName, null);
   }


   /**
    * Construct a "set parent" rule with the specified method name.
    *
    * @param digester    The associated Digester
    * @param methodName  Method name of the parent method to call
    * @param paramType   Java class of the parent method's argument (if you wish to use a primitive
    *      type, specify the corresonding Java wrapper class instead, such as <code>java.lang.Boolean</code>
    *      for a <code>boolean</code> parameter)
    */
   public XSetTopRule (XDigester digester, String methodName,
                       String paramType)
   {
      super (digester);
      this.methodName = methodName;
      this.paramType = paramType;
   }


   /**
    * The method name to call on the parent object.
    */
   protected String methodName = null;

   /**
    * The Java class name of the parameter type expected by the method.
    */
   protected String paramType = null;


   // --------------------------------------------------------- Public Methods


   /**
    * Process the end of this element.
    *
    * @throws Exception  Exception description not provided
    */
   public void end() throws Exception
   {

      // Identify the objects to be used
      Object child = digester.peek (1);
      Object parent = digester.peek (0);
      if (digester.getDebug() >= 1)
      {
         digester.log ("Call " + parent.getClass().getName() + "." +
            methodName + "(" + child + ")");
      }

      // Call the specified method
      Class paramTypes[] = new Class[1];
      if (paramType != null)
      {
         paramTypes[0] = digester.getClassLoader().loadClass (paramType);
      }
      else
      {
         paramTypes[0] = child.getClass();
      }
      Method method = parent.getClass().getMethod (methodName, paramTypes);
      method.invoke (parent, new Object[]
         {
         child
         }
         );

   }


   /**
    * Clean up after parsing is complete.
    *
    * @throws Exception  Exception description not provided
    */
   public void finish() throws Exception
   {

      methodName = null;
      paramType = null;

   }

}

/*
 * $Log: XSetTopRule.java,v $
 * Revision 1.9  2004/10/29 13:46:53  lowende
 * Some improvements for parsed code
 *
 */
