/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.coobra.actions;

import java.awt.event.ActionEvent;
import java.io.IOException;

import javax.swing.*;

import de.tu_bs.coobra.LocalRepository;
import de.tu_bs.coobra.RemoteRepositoryManager;
import de.tu_bs.coobra.errors.ServerIdentifierChangedException;
import de.tu_bs.coobra.remote.RemoteRepository;
import de.tu_bs.coobra.remote.lightweight.*;
import de.tu_bs.coobra.util.Password;
import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.uml.UMLProject;


/**
 * Restore all changes from server
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.25.2.4 $
 */
public class CheckoutAction extends AbstractAction implements LocalRepository.ProgressCallback
{
   /**
    * For choosing the server to checkout from
    */
   SelectServerRepositoryDialog dialog;
   /**
    * true if no update should be called
    */
   private boolean connectOnly;

   /**
    * dialog for displaying progress
    */
   ProgressDialog progress = new ProgressDialog (FrameMain.get());


   /**
    * called when update/restore/store progress
    *
    * @param percent  progress percentage
    */
   public void progress (int percent)
   {
      if (progress != null)
      {
         //progress.setVisible( percent > 0 && percent < 100 );
         progress.setPercentage (percent);
      }
   }


   /**
    * Perform the action
    *
    * @param event  the ActionEvent for performing the action
    */
   public void actionPerformed (ActionEvent event)
   {
      String status;
      connectOnly = shouldConnectOnly();

      //ask where to checkout from
      if (dialog == null)
      {
         dialog = new SelectServerRepositoryDialog (FrameMain.get(), true);
      }

      if (connectOnly)
      {
         dialog.setTitle ("Connect");
         progress.setTitle ("Connect");
      }
      else
      {
         dialog.setTitle ("Checkout");
         progress.setTitle ("Checkout");
      }
      dialog.show();

      progress.setPercentage (0);

      if (dialog.isOk())
      {
         connectOnly |= dialog.isConnectOnlyChecked();
         status = "connecting";
         FrameMain.get().setStatusLabel (status);

            new Thread()
            {
               /**
                * @see   Runnable#run()
                */
               public void run()
               {
                  while (!progress.isVisible())
                  {
                     try
                     {
                        Thread.sleep (10);
                     }
                     catch (InterruptedException e)
                     {
                     }
                  }
                  try
                  {
                     connect();
                  }
                  finally
                  {
                     progress.hide();
                  }
               }

            }.start();
         progress.show();
      }
   }


   /**
    * @return
    */
   protected boolean shouldConnectOnly()
   {
      return FujabaChangeManager.getVMRepository() != null && FujabaChangeManager.getVMRepository().getServerIdentifier() != null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void connect()
   {
      try
      {
         if (dialog.isByName())
         {
            connect (dialog.isNameServerChecked() ? dialog.getRepositoryName() : null, null, dialog.getLogin(), dialog.getPassword(),
               dialog.getNameServer(), dialog.getNameServerPort(), dialog.isLightweightChecked());
         }
         else
         {
            connect (dialog.getIOR());
         }
         dialog.rememberCurrentDataInHistoryList();
      }
      catch (Exception e)
      {
         FrameMain.get().setStatusLabel ("connect failed: " + e.getMessage());
         if (e.getMessage() != null)
         {
            if (e.getMessage().indexOf (LightWeightServerImplWithAuthentication.ERROR_AUTH) >= 0)
            {
               JOptionPane.showMessageDialog (FrameMain.get(), "Username and password not accepted!"
                  , "Connect", JOptionPane.ERROR_MESSAGE);
            }
            else if (e.getMessage().indexOf (LightWeightServerImplWithAuthentication.ERROR_DENIED_PREFIX) >= 0)
            {
               JOptionPane.showMessageDialog (FrameMain.get(), "User " + dialog.getLogin() + " is not allowed to connect to repository " + dialog.getRepositoryName()
                  , "Connect", JOptionPane.ERROR_MESSAGE);
            }
         }
         e.printStackTrace();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param ior                     No description provided
    * @throws NumberFormatException  Exception description not provided
    * @throws IOException            Exception description not provided
    */
   public void connect (String ior) throws NumberFormatException, IOException
   {
      connect (null, ior, null, null, null, null, false);
   }


   /**
    * perform connect (and update)
    *
    * @param repName                 No description provided
    * @param ior                     No description provided
    * @param login                   No description provided
    * @param password                No description provided
    * @param nameServer              No description provided
    * @param nameServerPort          No description provided
    * @param lightweight             No description provided
    * @throws NumberFormatException  Exception description not provided
    * @throws IOException            Exception description not provided
    */
   public void connect (String repName, String ior, String login, String password, String nameServer, String nameServerPort, boolean lightweight) throws NumberFormatException, IOException
   {
      String status;
      LocalRepository repository;
      if (!connectOnly)
      {
         createNewProject();
         repository = createRepository();
      }
      else
      {
         repository = FujabaChangeManager.getVMRepository();
      }

      RemoteRepository remote;
      if (ior == null)
      {
         if (!lightweight)
         {
            RemoteRepositoryManager manager = new RemoteRepositoryManager (nameServer, nameServerPort, null);
            remote = manager.connectByName (repName);
         }
         else
         {
            if (repName != null)
            {
               LightWeightNameService.ServiceInfo info =
                  LightWeightNameService.query (nameServer,
                  Integer.parseInt (nameServerPort), repName);
               final String passwordHash = Password.passwordHash (password);
               remote = new RemoteRepository (new LightWeightServer (info.getHost(),
                  Integer.parseInt (info.getPort()),
                  login,
                  passwordHash));
            }
            else
            {
               final String passwordHash = Password.passwordHash (password);
               remote = new RemoteRepository (new LightWeightServer (nameServer,
                  Integer.parseInt (nameServerPort),
                  login,
                  passwordHash));
            }
         }
      }
      else
      {
         RemoteRepositoryManager manager = new RemoteRepositoryManager (null);
         remote = manager.connectByIOR (ior);
      }

      try
      {
         repository.connect (remote);
      }
      catch (ServerIdentifierChangedException e)
      {
         e.printStackTrace();
         JOptionPane.showMessageDialog (FrameMain.get(), "The server does not contain the Project you are working on. Connect aborted!"
            , "Connect", JOptionPane.ERROR_MESSAGE);
         return;
      }
      status = "connect succeeded";

      if (!connectOnly)
      {
         repository.update (this);

         //find project
         if ( (repository.getUpdateFilter() == null) || !repository.getUpdateFilter().isPatch())
         {

            UMLProject project = FujabaChangeManager.getUMLProject();
            if (project != null)
            {
               UMLProject.setNewProject (project);
               FrameMain.get().setUMLProject (project);
               project.refreshDisplay();
               repository.moveChangesIntoChangesBase();
               project.setSaved (true);
            }
            else
            {
               status = "No project was found in the loaded repository!";
               //repository.removeYou();
            }
         }
      }

      FrameMain.get().setStatusLabel (status);
   }


   /**
    */
   protected void createNewProject()
   {
      UMLProject.setNewProject (null);
   }


   /**
    * @return   new LocalRepository()
    */
   protected LocalRepository createRepository()
   {
      LocalRepository oldRepo = FujabaChangeManager.getVMRepository();
      LocalRepository localRepository = new LocalRepository();
      FujabaChangeManager.setVMRepository (localRepository);
      if (oldRepo != null)
      {
         oldRepo.removeYou();
      }

      return localRepository;
   }
}

/*
 * $Log: CheckoutAction.java,v $
 * Revision 1.25.2.4  2006/03/30 15:51:34  lowende
 * Removed compile warnings.
 *
 * Revision 1.25.2.3  2006/03/07 17:49:59  l3_g5
 * added coobra action "apply as patch"
 *
 */
