/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.coobra.actions;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.util.*;
import java.util.List;

import javax.swing.*;

import org.apache.log4j.Logger;

import de.tu_bs.coobra.*;
import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.app.action.UpdateProjectTreeAction;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.UMLMethod;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.upb.tools.fca.FHashMap;
import de.upb.tools.fca.FLinkedList;


/**
 * Paste some ASGElements via CoObRA
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.11.2.1 $
 */
public class PasteAction extends AbstractAction implements CopyManager.CopyChangeCallbackInterface
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (PasteAction.class);

   /**
    * map for new object name (ASGElement, "newName")
    */
   private Map newNames;
   /**
    * data for copying
    */
   private CopyAction.CopyData copyData;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private CopyManager copyManager;


   /**
    * Defines an <code>Action</code> object with a default description string and default icon.
    */
   public PasteAction()
   {
      this (CopyAction.getClipboard());
   }


   /**
    * @param copyData  what data to use for pasting
    */
   public PasteAction (CopyAction.CopyData copyData)
   {
      this.copyData = copyData;
   }


   /**
    * @return   the data for copying the objects
    */
   public CopyAction.CopyData getCopyData()
   {
      return copyData;
   }


   /**
    * Invoked when the action occurs.
    *
    * @param e  event that caused the action
    */
   public void actionPerformed (ActionEvent e)
   {
      if (getCopyData().getSelectedObjects() != null)
      {
         UMLProject proj = UMLProject.get();
         Object pasteTarget = e.getSource();
         if (pasteTarget instanceof Iterator)
         {
            Iterator iter = (Iterator) pasteTarget;
            if (iter.hasNext())
            {
               pasteTarget = iter.next();
            }
            else
            {
               pasteTarget = proj.getCurrentDiagram();
            }
         }

         if (pasteTarget instanceof FElement)
         {
            paste ((FElement) pasteTarget);
         }
         else
         {
            Toolkit.getDefaultToolkit().beep();
            FrameMain.get().setStatusLabel ("Select a target for pasting first!");
         }
      }
      else
      {
         Toolkit.getDefaultToolkit().beep();
         FrameMain.get().setStatusLabel ("Nothing copied/cut!");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pasteTarget  No description provided
    * @return             No description provided
    */
   public List paste (FElement pasteTarget)
   {
      return paste (pasteTarget, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pasteTarget      No description provided
    * @param doUpdateDisplay  No description provided
    * @return                 No description provided
    */
   public List paste (FElement pasteTarget, boolean doUpdateDisplay)
   {
      newNames = new FHashMap();
      List copies = new FLinkedList();
      if (askForClassNames() && askForMethodNames (pasteTarget) && askForAttrNames (pasteTarget))
      {
         copyManager = new CopyManager (FujabaChangeManager.getVMRepository(), getCopyData().getCopyObjects());
         copyManager.setCopyChangeCallback (this);
         copyManager.copy (getCopyData().getChangesToBeCopied());

         Iterator it = getCopyData().getSelectedObjects().iterator();
         while (it.hasNext())
         {
            FElement element = (FElement) copyManager.getCopiedObject (it.next());
            element.setCutCopyPasteParent (pasteTarget);
            copies.add (element);
         }
      }

      if (doUpdateDisplay)
      {
         UMLProject.get().refreshDisplay();
         new UpdateProjectTreeAction().actionPerformed (null);
      }
      return copies;
   }


   /**
    * display a dialog to enter a new name for all copied methods
    *
    * @param pasteTarget  No description provided
    * @return             false when user clicked cancel
    */
   private boolean askForAttrNames (FElement pasteTarget)
   {
      Iterator it = getCopyData().getChangesToBeCopied().iterator();
      while (it.hasNext())
      {
         ObjectChange change = (ObjectChange) it.next();
         if (!newNames.containsKey (change.getAffectedObject())
            && change.getAffectedObject() instanceof FAttr)
         {
            FAttr attr = (FAttr) change.getAffectedObject();
            String newName = attr.getName();
            while (!isNameOk (newName, attr, pasteTarget))
            {
               newName = (String) JOptionPane.showInputDialog (FrameMain.get()
                  , "Please enter new name for Attribute " + attr.getName() + " (must be unique in class)."
                  , "Copy Attribute", JOptionPane.QUESTION_MESSAGE, null, null, newName);
               if (newName == null)
               {
                  return false; // user cancelled operation
               }
            }
            newNames.put (attr, newName);
         }
      }
      return true;
   }


   /**
    * checks wether the name for an attribute is unique and not empty
    *
    * @param newName      name to be checked
    * @param attr         attr that will get the name
    * @param pasteTarget  No description provided
    * @return             true when ok
    */
   private boolean isNameOk (String newName, FAttr attr, FElement pasteTarget)
   {
      boolean nameIsOk = newName != null && newName.length() >= 1;
      if (nameIsOk)
      {
         if (getCopyData().getCopyObjects().contains (attr.getFParent()))
         {
            return true;
         }

         //don't use same name twice
         FClass newParent = attr.getFParent();
         if (pasteTarget instanceof FClass && getCopyData().getSelectedObjects().contains (attr))
         {
            newParent = (FClass) pasteTarget;
         }
         if (newParent != null && newParent.getFromFAttrs (newName) != null)
         {
            nameIsOk = false;
         }
      }
      return nameIsOk;
   }


   /**
    * display a dialog to enter a new name for all copied methods
    *
    * @param pasteTarget  No description provided
    * @return             false when user clicked cancel
    */
   private boolean askForMethodNames (FElement pasteTarget)
   {
      Iterator it = getCopyData().getChangesToBeCopied().iterator();
      while (it.hasNext())
      {
         ObjectChange change = (ObjectChange) it.next();
         if (!newNames.containsKey (change.getAffectedObject())
            && change.getAffectedObject() instanceof FMethod)
         {
            FMethod method = (FMethod) change.getAffectedObject();
            String newName = method.getName();
            while (!isNameOk (newName, method, pasteTarget))
            {
               newName = (String) JOptionPane.showInputDialog (FrameMain.get()
                  , "Please enter new name for Method " + method.getName() + " (must be unique in class)."
                  , "Copy Method", JOptionPane.QUESTION_MESSAGE, null, null, newName);
               if (newName == null)
               {
                  return false; // user cancelled operation
               }
            }
            newNames.put (method, newName);
         }
      }
      return true;
   }


   /**
    * checks wether the name for a method is unique and not empty
    *
    * @param newName      name to be checked
    * @param method       method that will get the name
    * @param pasteTarget  No description provided
    * @return             true when ok
    */
   private boolean isNameOk (String newName, FMethod method, FElement pasteTarget)
   {
      boolean nameIsOk = newName != null && newName.length() >= 1;
      if (nameIsOk)
      {
         if (getCopyData().getCopyObjects().contains (method.getFParent()))
         {
            return true;
         }

         String fullName = UMLMethod.constructFullMethodName (newName, method.iteratorOfParam());
         //don't use same name twice
         FClass newParent = method.getFParent();
         if (pasteTarget instanceof FClass && getCopyData().getSelectedObjects().contains (method))
         {
            newParent = (FClass) pasteTarget;
         }
         if (newParent != null && newParent.getFromFMethods (fullName) != null)
         {
            nameIsOk = false;
         }
      }
      return nameIsOk;
   }


   /**
    * display a dialog to enter a new name for all methods when pasting into a not copied class
    *
    * @return   false when the user clicked cancel
    */
   private boolean askForClassNames()
   {
      Iterator it = getCopyData().getChangesToBeCopied().iterator();
      while (it.hasNext())
      {
         ObjectChange change = (ObjectChange) it.next();
         if (!newNames.containsKey (change.getAffectedObject())
            && change.getAffectedObject() instanceof FClass)
         {
            FClass cls = (FClass) change.getAffectedObject();
            String newName;
            boolean nameIsOk;
            do
            {
               newName = (String) JOptionPane.showInputDialog (FrameMain.get()
                  , "Please enter new classname for Class " + cls.getName() + " (must be unique)."
                  , "Copy class", JOptionPane.QUESTION_MESSAGE, null, null, cls.getName());
               if (newName == null)
               {
                  return false; // user cancelled operation
               }
               nameIsOk = newName.length() >= 1;
               if (nameIsOk)
               {
                  //don't use same name twice
                  if (UMLProject.get().getFromClasses (newName) != null)
                  {
                     nameIsOk = false;
                  }
               }
            } while (!nameIsOk);
            if (log.isInfoEnabled())
            {
               log.info ("putting " + cls + ": " + newName);
            }
            newNames.put (cls, newName);
         }
      }
      return true;
   }


   /**
    * called upon copy of each ObjectChange to provide an ability to alter the copied data
    *
    * @param change  values of the copied ObjectChange
    */
   public void reviewCopiedChange (MutableObjectChange change)
   {
      if (newNames.containsKey (change.getAffectedObject()))
      {
         if ("parent".equals (change.getFieldName())
            &&
             (change.getAffectedObject() instanceof FMethod
            || change.getAffectedObject() instanceof FAttr)
            && copyData.getSelectedObjects().contains (change.getAffectedObject()))
         {
            //ignore parent of directly copied methods
            change.setNewValue (null);
         }
         else if ("name".equals (change.getFieldName()))
         {
            change.setNewValue (newNames.get (change.getAffectedObject()));
         }
      }
   }
}

/*
 * $Log: PasteAction.java,v $
 * Revision 1.11.2.1  2005/09/30 18:57:18  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
