/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.listener;

import java.awt.*;
import java.awt.event.MouseEvent;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.*;

import de.tu_bs.coobra.*;
import de.uni_paderborn.fujaba.fsa.FSABend;
import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.fsa.FSAPolyLine;
import de.uni_paderborn.fujaba.fsa.SelectionManager;
import de.uni_paderborn.fujaba.fsa.swing.JBend;
import de.uni_paderborn.fujaba.fsa.swing.JBendLine;
import de.uni_paderborn.fujaba.fsa.swing.JPolyLine;
import de.uni_paderborn.fujaba.fsa.swing.LayerManager;


/**
 * Standard mouselistener for Components displaying DiagramItems</p> Handles dragging and selecting
 * of Objects as well as putting selected Objects on top of others
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.21.2.1 $
 */
public class DragMouseListener extends ComponentMouseListener implements AscendDescendMouseHandler.Descend
{
   /**
    * The default constructor is empty.
    */
   protected DragMouseListener() { }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static DragMouseListener singleton = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public static DragMouseListener get()
   {
      if (singleton == null)
      {
         singleton = new DragMouseListener();
      }

      return singleton;
   }


   /**
    * Inner attribute for calculating a mouse move.
    */
   private Point lastPoint = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected boolean dragging = false;


   /**
    * Overwrites the mouseClicked method of ComponentMouseListener. It sets the lastPoint attribute.
    *
    * @param event  No description provided
    */
   public void mouseClicked (MouseEvent event)
   {
      if (event.getClickCount() == 1)
      {
         lastPoint = event.getPoint();
      }
   }


   /**
    * Overwrites the mousePressed method of ComponentMouseListener. It sets the lastPoint attribute.
    *
    * @param event  No description provided
    */
   public void mousePressed (MouseEvent event)
   {
      if (event.getClickCount() == 1)
      {
         JComponent comp = (JComponent) event.getSource();
         setOrientation (comp, event);
         lastComponent = comp;
         lastPoint = event.getPoint();
         event.consume();
      }
   }


   /**
    * Overwrites the mouseReleased method of ComponentMouseListener. It sets the lastPoint
    * attribute to null.
    *
    * @param event  No description provided
    */
   public void mouseReleased (MouseEvent event)
   {
      JComponent comp = (JComponent) event.getSource();

      // store undo-information
      // TODO: check if we should do this only if we have dragged something
      ObjectChangeCause saveLocationsCause = new ObjectChangeStringCause ("drag");
      ObjectChange.pushCause (saveLocationsCause);
      try
      {
         Iterator iter = SelectionManager.get().iteratorOfSelection();
         while (iter.hasNext())
         {
            FSAObject object = (FSAObject) iter.next();
            object.setTransientProperties (false);
            object.saveLocation();

            // special handling for saving location of additional bends
            // in a polyline
            saveLocationInPolyLine (object);
         }
      }
      finally
      {
         ObjectChange.popCause (saveLocationsCause);
      }

      if (dragging)
      {
         setDragging (false);
         Container parent = comp.getParent();
         if (parent != null && parent instanceof JLayeredPane)
         {
            JLayeredPane pane = (JLayeredPane) parent;

            pane.setPosition (comp, 0);
            pane.revalidate();
            pane.repaint();
         }
      }
      lastPoint = null;
      lastComponent = null;
      event.consume();
   }


   /**
    * Overwrites the mouseClicked method of ComponentMouseListener. It moves the event's source
    * component to the new calculated position.
    *
    * @param event  No description provided
    */
   public void mouseDragged (MouseEvent event)
   {
      JComponent source = (JComponent) event.getSource();

      Point eventPoint = event.getPoint();

      if (getOrientation() == INNER)
      {
         if (!dragging)
         {
            setDragging (true);
         }

         if (lastPoint == null)
         {
            lastPoint = eventPoint;
         }

         int deltaX = (int)  (eventPoint.getX() - lastPoint.getX());
         int deltaY = (int)  (eventPoint.getY() - lastPoint.getY());

         boolean sourceFound = false;

         FSAObject object;
         JComponent jObject;

         Iterator selIter = SelectionManager.get().iteratorOfSelection();
         while (selIter.hasNext())
         {
            object = (FSAObject) selIter.next();
            FSAObject parent = object.getParent();

            if (parent == null || !parent.isSelected())
            {
               jObject = object.getJComponent();
               dragByDelta (jObject, deltaX, deltaY);

               if (jObject == source)
               {
                  sourceFound = true;
               }
            }
         }

         // drag the source if it hasn't already been dragged
         if (!sourceFound)
         {
            dragByDelta (source, deltaX, deltaY);
         }
         event.consume();
      }
   }


   /**
    * Drag FSAObject by specified amount.
    * Allows FSAObject to react on dragging.
    *
    *
    * @param fsaObject  FSAObject to be moved.
    * @param deltaX     x-amount
    * @param deltaY     y-amount
    */
   protected void dragByDelta (FSAObject fsaObject, int deltaX, int deltaY)
   {
      if (fsaObject == null)
      {
         return;
      }

      Point currentLocation = fsaObject.getJComponent().getLocation();
      Point destinationPoint = new Point (currentLocation.x + deltaX, currentLocation.y + deltaY);

      // don't let user drag object 'outside' the diagram
      if (destinationPoint.x < 0)
      {
         if (currentLocation.x > 0)
         {
            // adjust delta so it will drag component to zero-point
            deltaX = -currentLocation.x;
         }
         else
         {
            deltaX = 0;
         }
      }
      if (destinationPoint.y < 0)
      {
         if (currentLocation.y > 0)
         {
            // adjust delta so it will drag component to zero-point
            deltaY = -currentLocation.y;
         }
         else
         {
            deltaY = 0;
         }
      }

      // return if both x and y values won't change
      if (deltaX == 0 && deltaY == 0)
      {
         return;
      }

      fsaObject.dragByDelta (deltaX, deltaY);

      JComponent parent = (JComponent) fsaObject.getJComponent().getParent();
      if (parent != null)
      {
         parent.revalidate();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    *
    * @param comp    No description provided
    * @param deltaX  No description provided
    * @param deltaY  No description provided
    */
   protected void dragByDelta (JComponent comp, int deltaX, int deltaY)
   {
      Point destinationPoint = comp.getLocation();
      destinationPoint.x += deltaX;
      destinationPoint.y += deltaY;

      // don't let user drag object 'outside' the diagram
      if (destinationPoint.x < 0)
      {
         destinationPoint.x = 0;
      }
      if (destinationPoint.y < 0)
      {
         destinationPoint.y = 0;
      }

      // return if both x and y values won't change
      // check is also done in Component, so don't do it here, too
//		Point currentLocation = comp.getLocation();
//		if (destinationPoint.x == currentLocation.x && destinationPoint.y == currentLocation.y)
//			return;
      comp.setLocation (destinationPoint);

      JComponent parent = (JComponent) comp.getParent();
      if (parent != null)
      {
         parent.revalidate();
      }
   }


   /**
    * If drag is true move all selected components to the drag layer. If drag is
    * false move all selected components to the default layer. Repaint all
    * modified LayeredPanes afterwards.
    *
    *
    * @param drag  The new dragging value
    */
   protected void setDragging (boolean drag)
   {
      if (this.dragging != drag)
      {
         Vector panes = new Vector();
         Iterator selIter = SelectionManager.get().iteratorOfSelection();
         int layer;
         while (selIter.hasNext())
         {
            FSAObject selected = (FSAObject) selIter.next();
            JComponent selectedComp = selected.getJComponent();
            Container parent;

            if (selectedComp != null &&
                (parent = selectedComp.getParent()) != null &&
               parent instanceof JLayeredPane)
            {
               JLayeredPane pane = (JLayeredPane) parent;

               if (drag)
               {
                  layer = LayerManager.get().getDragLayer (selectedComp);
                  pane.setLayer (selectedComp, layer);
               }
               else
               {
                  layer = LayerManager.get().getLayer (selectedComp);
                  pane.setLayer (selectedComp, layer);
                  pane.setPosition (selectedComp, 0);
               }

               panes.addElement (pane);
            }
         }
         Iterator paneIter = panes.iterator();
         while (paneIter.hasNext())
         {
             ((JLayeredPane) paneIter.next()).repaint();
         }
         dragging = drag;
      }
   } // selectedComp


   /**
    * Sets the orientation attribute of the DragMouseListener object
    *
    * @param comp   The new orientation value
    * @param event  The new orientation value
    */
   protected void setOrientation (Component comp, MouseEvent event)
   {
      super.setOrientation (comp, event);
      int orientation = getOrientation();

      if ( ( (orientation &  (EAST | WEST)) != 0) && comp.getWidth() <= 4 * DISTANCE)
      {
         orientation = orientation & ~ (EAST | WEST);
      }
      if ( ( (orientation &  (NORTH | SOUTH)) != 0) && comp.getHeight() <= 4 * DISTANCE)
      {
         orientation = orientation & ~ (NORTH | SOUTH);
      }
      if (orientation == 0)
      {
         orientation = INNER;
      }
      setOrientation (orientation);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param object  No description provided
    */
   protected void saveLocationInPolyLine (FSAObject object)
   {
      if (! (object instanceof FSABend))
      {
         return;
      }

      FSABend fsaBend = (FSABend) object;
      JBend jBend = (JBend) fsaBend.getJComponent();

      if (jBend == null)
      {
         return;
      }

      Iterator connectedLines = jBend.iteratorOfLines();
      JBendLine line;

      while (connectedLines.hasNext())
      {
         line = (JBendLine) connectedLines.next();

         // found an 'additional' bend of a polyline
         if (line instanceof JPolyLine.LineSegment)
         {
            JPolyLine jPolyLine =  ((JPolyLine.LineSegment) line).getPolyLine();
            if (jPolyLine == null)
            {
               continue;
            }

            FSAPolyLine polyLine = (FSAPolyLine) FSAObject.getFSAObjectFromJComponent (jPolyLine);
            polyLine.saveAdditionalBendLocation();
         }
      }
   }
}

/*
 * $Log: DragMouseListener.java,v $
 * Revision 1.21.2.1  2005/05/25 12:47:47  fklar
 * + finegranular saving of location after dragging is complete
 * + storing location of additional bends in a FSAPolyLine
 *
 */
