/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.listener;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.event.EventListenerList;

import de.uni_paderborn.fujaba.fsa.FSAContainer;
import de.uni_paderborn.fujaba.fsa.FSAObject;


/**
 * Notifier for Parent changes in FSA hierarchy<p>
 *
 * Notifies the listeners whenever changes in the Swing-component hierarchy or the association
 * to their fsa components result in a change of the parent of the watched fsa component
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.16 $
 */
public class ParentNotifier implements PropertyChangeListener
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   FSAObject root = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   FSAContainer parent = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   Component end = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   EventListenerList listenerList = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   Component rootComponent = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JComponent realRootComponent = null;


   /**
    * Constructor for class ParentNotifier
    *
    * @param root  No description provided
    */
   public ParentNotifier (FSAObject root)
   {
      setRoot (root);
   }


   /**
    * Sets the root attribute of the ParentNotifier object
    *
    * @param root  The new root value
    */
   void setRoot (FSAObject root)
   {
      if (this.root != root)
      {
         if (this.root != null)
         {
            removeAllListeners();
         }
         this.root = root;
         if (root != null)
         {
            addRootListeners();
         }
      }
   }


   /**
    * Get the root attribute of the ParentNotifier object
    *
    * @return   The root value
    */
   public FSAObject getRoot()
   {
      return this.root;
   }


   /**
    * Access method for an one to n association.
    *
    * @param l  The object added.
    */
   public void addListener (ParentListener l)
   {
      if (listenerList == null)
      {
         listenerList = new EventListenerList();
      } // end of if ()
      listenerList.add (ParentListener.class, l);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param l  No description provided
    */
   public void removeListener (ParentListener l)
   {
      if (listenerList != null)
      {
         listenerList.remove (ParentListener.class, l);
      } // end of if ()
   }


   /*
    *  Notify all listeners that have registered interest for
    *  notification on this event type.  The event instance
    *  is lazily created using the parameters passed into
    *  the fire method.
    *  @see EventListenerList
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param source     No description provided
    * @param oldParent  No description provided
    * @param newParent  No description provided
    */
   protected void fireParentChanged (FSAObject source, FSAContainer oldParent, FSAContainer newParent)
   {
      if (listenerList != null)
      {
         // Guaranteed to return a non-null array
         Object[] listeners = listenerList.getListenerList();
         // Process the listeners last to first, notifying
         // those that are interested in this event
         for (int i = listeners.length - 2; i >= 0; i -= 2)
         {
            if (listeners[i] == ParentListener.class)
            {
               // Lazily create the event:
               ParentEvent parentEvent =
                  new ParentEvent (source, oldParent, newParent);
                ((ParentListener) listeners[i + 1]).parentChanged (parentEvent);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      removeAllListeners();
      root = null;
      if (listenerList != null)
      {
         Object[] listeners = listenerList.getListenerList();

         for (int i = listeners.length - 2; i >= 0; i -= 2)
         {
            listenerList.remove ((Class) listeners[i], (java.util.EventListener) listeners[i + 1]);
         }
         listenerList = null;
      }
      end = null;
   }


   /**
    * Access method for an one to n association.
    */
   void addRootListeners()
   {
      realRootComponent = root.getJComponent();
      Container comp = realRootComponent;

      parent = null;
      if (comp == null)
      {
         comp = root.getParentOfJComponent();
         //root.addPropertyChangeListener ("parentOfJComponent", this);
      }
      //root.addPropertyChangeListener ("jComponent", this);
      if (comp != null)
      {
         rootComponent = comp;
         addListeners (comp, true);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void removeRootListeners()
   {
      //root.removePropertyChangeListener ("parentOfJComponent", this);
      //root.removePropertyChangeListener ("jComponent", this);
   }


   /**
    * Access method for an one to n association.
    *
    * @param ancestor    The object added.
    * @param addToFirst  The object added.
    */
   void addListeners (Component ancestor, boolean addToFirst)
   {
      Component a;

      end = null;
      parent = null;
      for (a = ancestor;
         end == null;
         a = a.getParent())
      {
         if (addToFirst || a != ancestor)
         {
            if (a instanceof JComponent)
            {
               JComponent jAncestor = (JComponent) a;

               if (a != realRootComponent)
               {
                  parent = (FSAContainer) FSAObject.getFSAObjectFromJComponent (jAncestor);
                  if (parent != null)
                  {
                     end = a;
                  }
               }
               if (end == null)
               {
                  jAncestor.addPropertyChangeListener ("parent", this);
                  jAncestor.addPropertyChangeListener ("ancestor", this);
               }
               jAncestor.addPropertyChangeListener (FSAObject.JCOMPONENT_CLIENT_PROPERTY, this);
            }
         }
         if (end == null && a.getParent() == null)
         {
            end = a;
         }
      }
      if (end instanceof Window)
      {
         end = null;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param ancestor  No description provided
    */
   void removeListeners (Component ancestor)
   {
      Component a;
      for (a = ancestor; a != null; a = a.getParent())
      {
         if (a instanceof JComponent)
         {
            JComponent jAncestor = (JComponent) a;
            jAncestor.removePropertyChangeListener ("parent", this);
            jAncestor.removePropertyChangeListener ("ancestor", this);
            jAncestor.removePropertyChangeListener (FSAObject.JCOMPONENT_CLIENT_PROPERTY, this);
         }
         if (a == end)
         {
            break;
         }

      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void removeAllListeners()
   {
      if (rootComponent != null)
      {
         removeListeners (rootComponent);
         rootComponent = null;
      }
      removeRootListeners();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param evt  No description provided
    */
   public void propertyChange (PropertyChangeEvent evt)
   {
      String s = evt.getPropertyName();

      if (s != null)
      {
         Object source = evt.getSource();
         if (source instanceof JComponent)
         {
            //log.info("Source is JComponent "+source);
            JComponent component = (JComponent) evt.getSource();
            if (s.equals ("parent") || s.equals ("ancestor"))
            {
               if (evt.getNewValue() != null)
               {
                  if (component == end && parent == null)
                  {
                     addListeners (component, false);
                     if (parent != null)
                     {
                        fireParentChanged (root, null, parent);
                     }
                  }
               }
               else
               {
                  boolean needsNotify = parent != null;
                  Container oldParent = (Container) evt.getOldValue();

                  removeListeners (oldParent);

                  end = component;
                  if (needsNotify)
                  {
                     fireParentChanged (root, parent, null);
                     parent = null;
                  }
               }
            }
            else if (s.equals (FSAObject.JCOMPONENT_CLIENT_PROPERTY))
            {
               if (evt.getNewValue() != null)
               {
                  removeListeners (component);
                  fireParentChanged (root, parent, (FSAContainer) evt.getNewValue());
                  parent = (FSAContainer) evt.getNewValue();
                  end = component;
                  addListeners (end, true);
               }
               else if (parent != null && evt.getOldValue() == parent)
               {
                  component.removePropertyChangeListener (FSAObject.JCOMPONENT_CLIENT_PROPERTY, this);
                  addListeners (component, true);
                  fireParentChanged (root, (FSAContainer) evt.getOldValue(), parent);
               }
            }
         }
         else
         {
            FSAObject fsa = (FSAObject) source;

            if (s.equals ("jComponent") ||  (s.equals ("parentOfJComponent") && fsa.getJComponent() == null))
            {
               removeAllListeners();
               FSAContainer oldParent = parent;
               addRootListeners();
               if (oldParent != parent)
               {
                  fireParentChanged (root, oldParent, parent);
               }
            }
         }
      }
   }
}

/*
 * $Log: ParentNotifier.java,v $
 * Revision 1.16  2004/11/03 10:17:55  lowende
 * Javadoc warnings removed.
 *
 */
