/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.util.Iterator;

import javax.swing.*;


/**
 * THE choice for JCircle-Targets Cannot layout anything else
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.11 $
 */
public class CircleGrabLayouter extends GrabLayouter
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static CircleGrabLayouter layouter = null;


   /**
    * Constructor for class CircleGrabLayouter
    */
   private CircleGrabLayouter()
   {
      super();
   }


   /**
    * @return   the singleton instance
    */
   public static GrabLayouter get()
   {
      if (layouter == null)
      {
         layouter = new CircleGrabLayouter();
      }

      return layouter;
   }


   /**
    * does the layout. <br>
    * the target of the manager must be a JCircle otherwise an Exception is thrown
    *
    * @param manager                              No description provided
    * @see                                        de.uni_paderborn.fujaba.fsa.swing.GrabManager#getTarget
    * @see                                        #layoutGrab
    * @throws java.lang.IllegalArgumentException  if the target of manager is anything else
    *      than a JCircle
    */
   public void layout (GrabManager manager)
   {
      if (! (manager.getTarget() instanceof JCircle))
      {
         throw new IllegalArgumentException (
            "A CircleGrabLayouter can only layout grabs with a JCircle as target");
      }

      Iterator grabIter = manager.iteratorOfGrabs();
      while (grabIter.hasNext())
      {
         JGrab grab = (JGrab) grabIter.next();
         layoutGrab (manager, grab);
      }
   }


   /**
    * layout the Grab around the Circle
    *
    * @param manager  No description provided
    * @param grab     No description provided
    * @see            #getPreferredLayoutInformation
    */
   protected void layoutGrab (GrabManager manager, JGrab grab)
   {
      Rectangle bounds = manager.getTarget().getBounds();
      double[] info = getPreferredLayoutInformation (grab);
      Direction orientation = Direction.fromInt ((int) info[0]);
      double align = info[1];

      align =  (1 - 2 * align) * Math.PI / 4;

      int rad =  ((JCircle) manager.getTarget()).getRadius();
      int centerX = bounds.x + rad - 1;
      int centerY = bounds.y + rad - 1;

      Point grabPoint = new Point();
      Rectangle grabBounds = grab.getBounds();

      if (orientation == Direction.TOP)
      {
         grabPoint.x = (int)  (centerX - Math.sin (align) * rad -  ((double) grabBounds.width) / 2);
         grabPoint.y = (int)  (centerY - Math.cos (align) * rad) - grabBounds.height + 1;
      }
      else if (orientation == Direction.BOTTOM)
      {
         grabPoint.x = (int)  (centerX - Math.sin (align) * rad -  ((double) grabBounds.width) / 2);
         grabPoint.y = (int)  (centerY + Math.cos (align) * rad);
      }
      else if (orientation == Direction.LEFT)
      {
         grabPoint.x = (int)  (centerX - Math.cos (align) * rad) - grabBounds.width + 1;
         grabPoint.y = (int)  (centerY - Math.sin (align) * rad -  ((double) grabBounds.height / 2));
      }
      else
      {
         grabPoint.x = (int)  (centerX + Math.cos (align) * rad);
         grabPoint.y = (int)  (centerY - Math.sin (align) * rad -  ((double) grabBounds.height / 2));
      }
      Container targetParent = manager.getTarget().getParent();
      Container grabParent = grab.getParent();
      if (targetParent != null && grabParent != null)
      {
         grabPoint = SwingUtilities.convertPoint (targetParent, grabPoint, grabParent);
      }

      grab.setLocation (grabPoint);
   }


   /**
    * the preferred orientation and alignment are determined by the angle of the first line
    * of the grab to the center of the circle
    *
    * @param grab  No description provided
    * @return      The preferredLayoutInformation value
    */
   protected double[] getPreferredLayoutInformation (JGrab grab)
   {
      Direction orientation = grab.getOrientation();
      double align = grab.getAlignment();
      JCircle target = (JCircle) grab.getTarget();
      if (grab.sizeOfLines() == 0 || target == null)
      {
         if (grab.isAutoOrientation())
         {
            orientation = Direction.LEFT;
            grab.setLayoutOrientation (orientation);
         }
         if (grab.isAutoAlignment())
         {
            align = 0.5;
            grab.setLayoutAlignment (align);
         }
         return new double[]
            {
            orientation.asInt(), align
            };
      }

      Point mid = target.getLocation();
      mid.x += target.getRadius() - 1;
      mid.y += target.getRadius() - 1;

      Iterator lineIter = grab.iteratorOfLines();
      JBendLine line = (JBendLine) lineIter.next();
      JBend bend = line.getStartBend();

      if (bend == grab)
      {
         bend = line.getEndBend();
      }

      Point p = bend.getPoint();
      if (bend.getParent() != null && target.getParent() != null)
      {
         p = SwingUtilities.convertPoint (bend.getParent(), p, target.getParent());
      }

      Direction tmpOrient = Direction.LEFT;
      double tmpAlign = 0.5;

      if (!p.equals (mid))
      {
         int angle = (int)  (Math.acos ( (p.x - mid.x) / Math.sqrt ( (p.x - mid.x) *  (p.x - mid.x) +
             (p.y - mid.y) *  (p.y - mid.y))) * 180 / Math.PI);

         if (p.y > mid.y)
         {
            angle = 360 - angle;
         }

         if (angle <= 45)
         {
            tmpOrient = Direction.RIGHT;
            tmpAlign = 1 -  (angle + 45.0) / 90;
         }
         else if (angle > 315)
         {
            tmpOrient = Direction.RIGHT;
            tmpAlign = 1 -  ((double)  (angle - 360 + 45)) / 90;
         }
         else if (45 < angle && angle <= 135)
         {
            tmpOrient = Direction.TOP;
            tmpAlign = 1 -  ((double)  (angle - 45)) / 90;
         }
         else if (135 < angle && angle <= 225)
         {
            tmpOrient = Direction.LEFT;
            tmpAlign =  ((double)  (angle - 135)) / 90;
         }
         else
         {
            tmpOrient = Direction.BOTTOM;
            tmpAlign =  ((double)  (angle - 225)) / 90;
         }
      }

      if (grab.isAutoOrientation())
      {
         grab.setLayoutOrientation (tmpOrient);
         orientation = tmpOrient;
      }

      if (grab.isAutoAlignment())
      {
         grab.setLayoutAlignment (tmpAlign);
         align = tmpAlign;
      }
      return new double[]
         {
         orientation.asInt(), align
         };
   }
}

/*
 * $Log: CircleGrabLayouter.java,v $
 * Revision 1.11  2004/11/03 10:17:56  lowende
 * Javadoc warnings removed.
 *
 */
