/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.util.Iterator;
import java.util.Map;

import javax.swing.*;

import de.upb.tools.fca.FDuplicatedTreeMap;


/**
 * THE choice for rectangular shaped targets
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.20 $
 */
public class DefaultGrabLayouter extends GrabLayouter
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static transient int MAX_PAD = 4;


   /**
    * layout all the grabs of manager<p>
    *
    * currently the grabs are simply set to their preferred location as determined by layoutGrab
    * and getPreferredLayoutInformation.<p>
    *
    * Overlapping is not handled <pre>
    *    |             A            |
    * ...|---|------------------|---|...
    *      b           c          b
    * <pre>
    *
    * A = available space for each Grab, equally distributed along the appropriate dimension of the target's bounds<br>
    * b = left/right padding, 1/8 of A<br>
    * c = layout space for the grab. It can be positioned according to its preferred alignment
    * within this interval.
    *
    * @param manager  No description provided
    * @see            de.uni_paderborn.fujaba.fsa.swing.GrabLayouter#getPreferredLayoutInformation
    * @see            #layoutGrab
    */
   public void layout (GrabManager manager)
   {
      Iterator grabIter = manager.iteratorOfGrabs();
      FDuplicatedTreeMap topGrabs = new FDuplicatedTreeMap();
      int topAutoLayoutGrabs = 0;
      FDuplicatedTreeMap rightGrabs = new FDuplicatedTreeMap();
      int rightAutoLayoutGrabs = 0;
      FDuplicatedTreeMap bottomGrabs = new FDuplicatedTreeMap();
      int bottomAutoLayoutGrabs = 0;
      FDuplicatedTreeMap leftGrabs = new FDuplicatedTreeMap();
      int leftAutoLayoutGrabs = 0;

      while (grabIter.hasNext())
      {
         JGrab grab = (JGrab) grabIter.next();

         double[] info = getPreferredLayoutInformation (grab);
         Direction orientation = Direction.fromInt ((int) info[0]);
         double align = info[1];

         EnumKey key = EnumKey.create (grab, orientation, align);
         if (orientation == Direction.TOP)
         {
            topGrabs.put (key, grab);
            if (grab.isAutoAlignment())
            {
               topAutoLayoutGrabs++;
            }
         }
         else if (orientation == Direction.RIGHT)
         {
            rightGrabs.put (key, grab);
            if (grab.isAutoAlignment())
            {
               rightAutoLayoutGrabs++;
            }
         }
         else if (orientation == Direction.BOTTOM)
         {
            bottomGrabs.put (key, grab);
            if (grab.isAutoAlignment())
            {
               bottomAutoLayoutGrabs++;
            }
         }
         else
         {
            leftGrabs.put (key, grab);
            if (grab.isAutoAlignment())
            {
               leftAutoLayoutGrabs++;
            }
         }
      }

      layoutAll (manager, topGrabs, topAutoLayoutGrabs, Direction.TOP);
      layoutAll (manager, rightGrabs, rightAutoLayoutGrabs, Direction.RIGHT);
      layoutAll (manager, bottomGrabs, bottomAutoLayoutGrabs, Direction.BOTTOM);
      layoutAll (manager, leftGrabs, leftAutoLayoutGrabs, Direction.LEFT);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param manager      No description provided
    * @param grabs        No description provided
    * @param size         No description provided
    * @param orientation  No description provided
    */
   private void layoutAll (GrabManager manager, Map grabs, int size, Direction orientation)
   {
      Iterator grabIter = grabs.entrySet().iterator();
      double space =  (size > 0 ? 1.0 / size : 0);

      double pad = space * 0.125;
      double maxPadRel;
      int dim;

      if (orientation == Direction.TOP || orientation == Direction.BOTTOM)
      {
         dim = manager.getTarget().getBounds().width;
         maxPadRel =  (dim > 0 ? MAX_PAD /  (2.0 * dim) : 1);
      }
      else if (orientation == Direction.LEFT || orientation == Direction.RIGHT)
      {
         dim = manager.getTarget().getBounds().height;
         maxPadRel =  (dim > 0 ? MAX_PAD /  (2.0 * dim) : 1);
      }
      else
      {
         maxPadRel = 1;
      }

      pad = Math.min (pad, maxPadRel);
      double layoutSpace = space - 2 * pad;
      double offset = 0;
      double start;
      double end;
      double[] used = new double[2];

      int count = 0;

      Map.Entry[] reverse = new Map.Entry[size];
      Map.Entry entry;
      EnumKey key;
      JGrab grab;

      while (grabIter.hasNext())
      {
         entry = (Map.Entry) grabIter.next();
         key = (EnumKey) entry.getKey();
         grab = (JGrab) entry.getValue();
         if (grab.isAutoAlignment())
         {
            count++;
            start = offset + pad;
            end = start + layoutSpace;
            layoutGrab (manager, grab, Direction.fromInt (key.orientation), key.alignment, start, end, used);
            offset = used[1];
            key.start = used[0];
            key.end = used[1];

            layoutSpace +=  (end - offset) /  (size - count); // distribute unused space for next grab
            offset += pad;

            reverse[count - 1] = entry;
         }
         else
         {
            layoutGrab (manager, grab, Direction.fromInt (key.orientation), key.alignment, 0, 1, used);
         } // else
      } // while

      Map.Entry prevEntry = null;
      EnumKey prevKey = null;
      double align;
      end = 1;
      for (int i = size - 1; i >= 0; i--)
      {
         entry = reverse[i];
         key = (EnumKey) entry.getKey();
         grab = (JGrab) entry.getValue();
         align = grab.getAlignment();

         if (i < size - 1 && align < key.alignment && align <= end)
         {
            if (i > 0 && i < size - 1)
            {
               prevEntry = reverse[i - 1];
               prevKey = (EnumKey) prevEntry.getKey();
               start = prevKey.end + 2 * pad;
            }
            else
            {
               start = key.start;
            } // else
            layoutGrab (manager, grab, Direction.fromInt (key.orientation), key.alignment, start, end, used);
            end = used[0] - 2 * pad;
         }
         else
         {
            end = key.start - 2 * pad;
         } // else
      } // for
   }


   /**
    * layout the grab
    *
    * @param manager      No description provided
    * @param grab         No description provided
    * @param orientation  No description provided
    * @param align        No description provided
    * @param start        No description provided
    * @param end          No description provided
    * @param used         No description provided
    * @see                de.uni_paderborn.fujaba.fsa.swing.GrabLayouter#getPreferredLayoutInformation
    * @see                #layout
    */
   protected void layoutGrab (GrabManager manager, JGrab grab, Direction orientation, double align, double start, double end, double[] used)
   {
      Rectangle bounds = manager.getTarget().getBounds();

      Point grabPoint = new Point();
      Rectangle grabBounds = grab.getBounds();
      int grabStart = grab.getTouchOffset();
      int grabEnd = grabStart + grab.getTouchLength();
      int grabAlign = grab.getAlignmentOffset();
      int grabPad = grab.getPadOffset();

      if (start > align)
      {
         align = start;
      }
      else if (end < align)
      {
         align = end;
      }

      int leftBorder;
      int rightBorder;
      int divStart;
      int divEnd;

      if (orientation == Direction.TOP)
      {
         grabPoint.x = bounds.x + (int)  (bounds.width * align - grabAlign);
         grabPoint.y = bounds.y - grabBounds.height + 1 - grabPad;

         leftBorder = bounds.x + (int)  (bounds.width * start);
         rightBorder = bounds.x + (int)  (bounds.width * end);
         divStart = grabPoint.x + grabStart - leftBorder;
         divEnd = rightBorder -  (grabPoint.x + grabEnd);

         if (divEnd < 0 || divStart < 0)
         {
            // distribute overlapping space
            if (divEnd < 0 && divStart < 0)
            {
               divEnd -= divStart;
               divEnd = divEnd / 2;
               grabPoint.x += divEnd;
            }
            else if (divStart < 0)
            {
               divEnd += divStart;
               if (divEnd < 0)
               {
                  grabPoint.x -= divEnd / 2;
               }
               else
               {
                  grabPoint.x -= divStart;
               } // else
            } // else
            else
            { //divEnd < 0
               divStart += divEnd;
               if (divStart < 0)
               {
                  grabPoint.x += divStart / 2;
               }
               else
               {
                  grabPoint.x += divEnd;
               } // else
            } // else
         }
      }
      else if (orientation == Direction.BOTTOM)
      {
         grabPoint.x = bounds.x + (int)  (bounds.width * align - grabAlign);
         grabPoint.y = bounds.y + bounds.height - 1 + grabPad;
         leftBorder = bounds.x + (int)  (bounds.width * start);
         rightBorder = bounds.x + (int)  (bounds.width * end);
         divStart = grabPoint.x + grabStart - leftBorder;
         divEnd = rightBorder -  (grabPoint.x + grabEnd);

         if (divEnd < 0 || divStart < 0)
         {
            // distribute overlapping space
            if (divEnd < 0 && divStart < 0)
            {
               divEnd -= divStart;
               divEnd = divEnd / 2;
               grabPoint.x += divEnd;
            }
            else if (divStart < 0)
            {
               divEnd += divStart;
               if (divEnd < 0)
               {
                  grabPoint.x -= divEnd / 2;
               }
               else
               {
                  grabPoint.x -= divStart;
               } // else
            } // else
            else
            { //divEnd < 0
               divStart += divEnd;
               if (divStart < 0)
               {
                  grabPoint.x += divStart / 2;
               }
               else
               {
                  grabPoint.x += divEnd;
               } // else
            } // else
         }
      }
      else if (orientation == Direction.LEFT)
      {
         grabPoint.x = bounds.x - grabBounds.width + 1 - grabPad;
         grabPoint.y = bounds.y + (int)  (bounds.height * align - grabAlign);
         leftBorder = bounds.y + (int)  (bounds.height * start);
         rightBorder = bounds.y + (int)  (bounds.height * end);
         divStart = grabPoint.y + grabStart - leftBorder;
         divEnd = rightBorder -  (grabPoint.y + grabEnd);

         if (divEnd < 0 || divStart < 0)
         {
            // distribute overlapping space
            if (divEnd < 0 && divStart < 0)
            {
               divEnd -= divStart;
               divEnd = divEnd / 2;
               grabPoint.y += divEnd;
            }
            else if (divStart < 0)
            {
               divEnd += divStart;
               if (divEnd < 0)
               {
                  grabPoint.y -= divEnd / 2;
               }
               else
               {
                  grabPoint.y -= divStart;
               } // else
            } // else
            else
            { //divEnd < 0
               divStart += divEnd;
               if (divStart < 0)
               {
                  grabPoint.y += divStart / 2;
               }
               else
               {
                  grabPoint.y += divEnd;
               } // else
            } // else
         }
      }
      else
      {
         grabPoint.x = bounds.x + bounds.width - 1 + grabPad;
         grabPoint.y = bounds.y + (int)  (bounds.height * align - grabAlign);
         leftBorder = bounds.y + (int)  (bounds.height * start);
         rightBorder = bounds.y + (int)  (bounds.height * end);
         divStart = grabPoint.y + grabStart - leftBorder;
         divEnd = rightBorder -  (grabPoint.y + grabEnd);
         if (divEnd < 0 || divStart < 0)
         {
            // distribute overlapping space
            if (divEnd < 0 && divStart < 0)
            {
               divEnd -= divStart;
               divEnd = divEnd / 2;
               grabPoint.y += divEnd;
            }
            else if (divStart < 0)
            {
               divEnd += divStart;
               if (divEnd < 0)
               {
                  grabPoint.y -= divEnd / 2;
               }
               else
               {
                  grabPoint.y -= divStart;
               } // else
            } // else
            else
            { //divEnd < 0
               divStart += divEnd;
               if (divStart < 0)
               {
                  grabPoint.y += divStart / 2;
               }
               else
               {
                  grabPoint.y += divEnd;
               } // else
            } // else
         }
      }

      Container targetParent = manager.getTarget().getParent();
      Container grabParent = grab.getParent();

      if (targetParent != null && grabParent != null)
      {
         grabPoint = SwingUtilities.convertPoint (targetParent, grabPoint, grabParent);
      }

      grab.setLocation (grabPoint);

      if (orientation == Direction.TOP || orientation == Direction.BOTTOM)
      {
         if (bounds.width != 0)
         {
            used[0] = Math.max (start, Math.abs (grabPoint.x + grabStart - bounds.x) / (double) bounds.width);
            used[1] = Math.min (end, Math.abs (grabPoint.x + grabEnd - bounds.x) / (double) bounds.width);
            grab.setLayoutAlignment ( (grabPoint.x + grabAlign - bounds.x) / (double) bounds.width);
         }
         else
         {
            used[0] = used[1] =
            /*
             *  0.5
             */
               align *  (start + end);
            grab.setLayoutAlignment (used[0]);
         } // else
      }
      else if (orientation == Direction.LEFT || orientation == Direction.RIGHT)
      {
         if (bounds.height != 0)
         {
            used[0] = Math.max (start, Math.abs (grabPoint.y + grabStart - bounds.y) / (double) bounds.height);
            used[1] = Math.min (end, Math.abs (grabPoint.y + grabEnd - bounds.y) / (double) bounds.height);
            grab.setLayoutAlignment ( (grabPoint.y + grabAlign - bounds.y) / (double) bounds.height);
         }
         else
         {
            used[0] = used[1] =
            /*
             *  0.5
             */
               align *  (start + end);
            grab.setLayoutAlignment (used[0]);
         } // else
      }
      else
      {
         used[0] = used[1] =
         /*
          *  0.5
          */
            align *  (start + end);
         grab.setLayoutAlignment (used[0]);
      }

      grab.setLayoutOrientation (orientation);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.20 $
    */
   private final static class EnumKey implements Comparable
   {
      /**
       * Constructor for class EnumKey
       *
       * @param orientation  No description provided
       * @param alignment    No description provided
       * @param angle        No description provided
       * @param hashcode     No description provided
       */
      private EnumKey (int orientation, double alignment, double angle, int hashcode)
      {
         this.angle = angle;
         this.hashcode = hashcode;
         this.orientation = orientation;
         this.alignment = alignment;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final transient double angle;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final transient int hashcode;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final transient int orientation;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public final transient double alignment;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public transient double start;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public transient double end;


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param o  No description provided
       * @return   No description provided
       */
      public int compareTo (Object o)
      {
         if (o == null)
         {
            return 1;
         }
         else if (o == this)
         {
            return 0;
         }
         else if (o instanceof EnumKey)
         {
            EnumKey key = (EnumKey) o;

            if (this.orientation != key.orientation)
            {
               return  (this.orientation < key.orientation ? -1 : 1);
            }
            if (this.alignment != key.alignment)
            {
               return  (this.alignment < key.alignment ? -1 : 1);
            }
            if (this.angle != key.angle)
            {
               return  (this.angle < key.angle ? -1 : 1);
            }
            if (this.hashcode != key.hashcode)
            {
               return  (this.hashcode < key.hashcode ? -1 : 1);
            }
            return 0;
         }
         else
         {
            return -1;
         } // else
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param grab         No description provided
       * @param orientation  No description provided
       * @param align        No description provided
       * @return             No description provided
       */
      public static EnumKey create (JGrab grab, Direction orientation, double align)
      {
         JBendLine line = grab.getFirstFromLines();
         double angle = 0;
         int hashcode = 0;
         if (line != null)
         {
            angle = getAngle (grab, line, orientation);
            hashcode = line.hashCode();
         }
         return new EnumKey (orientation.asInt(), align, angle, hashcode);
      }


      /**
       * Get the angle attribute of the EnumKey class
       *
       * @param grab         No description provided
       * @param line         No description provided
       * @param orientation  No description provided
       * @return             The angle value
       */
      private static double getAngle (JGrab grab, JBendLine line, Direction orientation)
      {
         double angle = 0;

         JBend bend = line.getNextFromBends (grab);
         if (bend == null)
         {
            bend = line.getPrevFromBends (grab);
         }

         Point to = null;

         if (bend instanceof JGrab)
         {
            to = getGrabCoordinates ((JGrab) bend);
         }
         else if (bend != null)
         {
            to = bend.getPoint();
         } // else

         if (to != null)
         {
            Point from = getGrabCoordinates (grab);

            int dx = to.x - from.x;
            int dy = to.y - from.y;

            angle = Math.atan2 (dx, dy);
            if (orientation == Direction.TOP)
            {
               angle = -angle;
            }
            else if (orientation == Direction.RIGHT)
            {
               if (angle < Math.PI)
               {
                  angle += 2 * Math.PI;
               }
               angle = -angle;
            }

         } // end of if ()

         return angle;
      }


      /**
       * Get the grabCoordinates attribute of the EnumKey class
       *
       * @param grab  No description provided
       * @return      The grabCoordinates value
       */
      private static Point getGrabCoordinates (JGrab grab)
      {
         Point result = null;
         JComponent target = grab.getTarget();

         if (target != null)
         {
            Rectangle bounds = target.getBounds();
            result = new Point (bounds.x + bounds.width / 2, bounds.y + bounds.height / 2);
         }
         else
         {
            result = grab.getPoint();
         } // else
         return result;
      }
   }
}

/*
 * $Log: DefaultGrabLayouter.java,v $
 * Revision 1.20  2004/11/03 10:17:56  lowende
 * Javadoc warnings removed.
 *
 */
