/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.util.Iterator;

import javax.swing.*;


/**
 * Called by a GrabManager to layout the Grabs of the Manager around the border of the Managers
 * target.<p>
 *
 * The GrabLayouter used by the GrabManager is determined as follows:<br>
 * If the target has a GrabLayouter defined as client property with name GrabLayouter.TARGET_PROPERTY
 * (which is the same as the result of <pre>GrabLayouter.getLayouter(target)</pre>) that one
 * is used. Otherwise GrabLayouter.get() is called to get the default Layouter.
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.13 $
 */
public abstract class GrabLayouter
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TARGET_PROPERTY = "GrabLayouter";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static Class defaultLayouterClass = DefaultGrabLayouter.class;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static GrabLayouter layouter = null;


   /**
    * @return   the singleton instance
    */
   public static GrabLayouter getDefaultLayouter()
   {
      if (layouter == null && defaultLayouterClass != null)
      {
         try
         {
            layouter = (GrabLayouter) defaultLayouterClass.newInstance();
         }
         catch (IllegalAccessException iae)
         {
            iae.printStackTrace();
            throw new RuntimeException ("Exception in instantiation: " + iae.getMessage());
         }
         catch (InstantiationException iex)
         {
            iex.printStackTrace();
            throw new RuntimeException ("Exception in instantiation: " + iex.getMessage());
         }
         catch (ExceptionInInitializerError ei)
         {
            ei.printStackTrace();
            Throwable ex = ei.getException();
            ex.printStackTrace();
            throw new RuntimeException ("Exception in instantiation: " + ex.getMessage());
         }
      }
      return layouter;
   }


   /**
    * Sets the defaultLayouterClass attribute of the GrabLayouter class
    *
    * @param layouterClass  The new defaultLayouterClass value
    */
   public static void setDefaultLayouterClass (Class layouterClass)
   {
      if (defaultLayouterClass != layouterClass)
      {
         if (GrabLayouter.class.isAssignableFrom (layouterClass))
         {
            defaultLayouterClass = layouterClass;
            layouter = null;
         }
         else
         {
            throw new IllegalArgumentException (layouterClass + " is not a subclass of " + GrabLayouter.class);
         }
      }
   }


   /**
    * Get the defaultLayouterClass attribute of the GrabLayouter class
    *
    * @return   The defaultLayouterClass value
    */
   public static Class getDefaultLayouterClass()
   {
      return defaultLayouterClass;
   }


   /**
    * The result of this method is either the layouter specified as client property of comp
    * or if none is found the result of GrabLayouter.get()
    *
    * @param comp  the target for which a layouter is needed
    * @return      the Layouter to use for the target comp
    * @see         #registerLayouterWithJComponent
    * @see         #unregisterLayouterFromJComponent
    * @see         #getDefaultLayouter
    * @see         javax.swing.JComponent#getClientProperty
    */
   public static GrabLayouter getLayouter (JComponent comp)
   {
      if (comp == null)
      {
         return null;
      }

      GrabLayouter result = (GrabLayouter) comp.getClientProperty (TARGET_PROPERTY);

      if (result == null)
      {
         result = getDefaultLayouter();
      }
      return result;
   }


   /**
    * Set this Layouter as default for comp by adding it to comps client properties
    *
    * @param comp  No description provided
    * @see         #TARGET_PROPERTY
    * @see         #unregisterLayouterFromJComponent
    * @see         #getLayouter
    */
   public void registerLayouterWithJComponent (JComponent comp)
   {
      if (comp == null)
      {
         return;
      }

      comp.putClientProperty (TARGET_PROPERTY, this);
   }


   /**
    * clears the default for comp by the field from the client properties
    *
    * @param comp  No description provided
    * @see         #TARGET_PROPERTY
    * @see         #registerLayouterWithJComponent
    * @see         #getLayouter
    */
   public static void unregisterLayouterFromJComponent (JComponent comp)
   {
      if (comp == null)
      {
         return;
      }

      comp.putClientProperty (TARGET_PROPERTY, null);
   }


   /**
    * Assigns every Grab of manager a position on the border of the managers target
    *
    * @param manager  the GrabManager for which the work is to be done
    */
   public abstract void layout (GrabManager manager);


   /**
    * Retrieves the preferred orientation and alignment of the Grab and sets these values in
    * the Grab.<p>
    *
    * This default implementation only inspects the first line of the Grab returned by grab.iteratorOfLines().
    * It gets the orientation and alignment that are closest to the other point of the line
    *
    * @param grab  the Grab to inspect
    * @return      the preferred orientation and alignment of the grab
    * @see         de.uni_paderborn.fujaba.fsa.swing.JGrab#setLayoutOrientation
    * @see         de.uni_paderborn.fujaba.fsa.swing.JGrab#setLayoutAlignment
    * @see         de.uni_paderborn.fujaba.fsa.swing.JBend#iteratorOfLines
    */
   protected double[] getPreferredLayoutInformation (JGrab grab)
   {
      Direction orientation = grab.getOrientation();
      double align = grab.getAlignment();
      JComponent target = grab.getTarget();
      if (grab.sizeOfLines() == 0 || target == null)
      {
         if (grab.isAutoOrientation())
         {
            orientation = Direction.LEFT;
            grab.setLayoutOrientation (orientation);
         }
         if (grab.isAutoAlignment())
         {
            align = 0.5;
            grab.setLayoutAlignment (align);
         }
         return new double[]
            {
            orientation.asInt(), align
            };
      }
      Rectangle bounds = target.getBounds();
      Iterator lineIter = grab.iteratorOfLines();
      JBendLine line = (JBendLine) lineIter.next();
      JBend bend = line.getNextFromBends (grab);
      if (bend == null)
      {
         bend = line.getPrevFromBends (grab);
      }

      if (bend != null)
      {
         Point p = bend.getPoint();
         Container targetParent = target.getParent();
         if (targetParent != null && bend.getParent() != null)
         {
            p = SwingUtilities.convertPoint (bend.getParent(), p, targetParent);
         }

         Direction tmpOrient = Direction.LEFT;
         double tmpAlign = 0.5;

         if (bounds.contains (p))
         {
            int dist = p.x - bounds.x;
            tmpOrient = Direction.LEFT;

            if (bounds.x + bounds.width - p.x - 1 < dist)
            {
               dist = bounds.x + bounds.width - p.x - 1;
               tmpOrient = Direction.RIGHT;
            }
            if (p.y - bounds.y < dist)
            {
               dist = p.y - bounds.y;
               tmpOrient = Direction.TOP;
            }
            if (bounds.y + bounds.height - p.y - 1 < dist)
            {
               tmpOrient = Direction.BOTTOM;
            }
            if (tmpOrient == Direction.TOP || tmpOrient == Direction.BOTTOM)
            {
               tmpAlign =  ((double)  (p.x - bounds.x)) / bounds.width;
            }
            else
            {
               tmpAlign =  ((double)  (p.y - bounds.y)) / bounds.height;
            }
         }
         else if (p.x <= bounds.x)
         {
            if (p.y < bounds.y)
            {
               tmpAlign = 0;
               if (bounds.x - p.x < bounds.y - p.y)
               {
                  tmpOrient = Direction.TOP;
               }
               else
               {
                  tmpOrient = Direction.LEFT;
               }
            }
            else if (p.y >= bounds.y + bounds.height)
            {
               if (bounds.x - p.x < p.y - bounds.y - bounds.height)
               {
                  tmpOrient = Direction.BOTTOM;
                  tmpAlign = 0;
               }
               else
               {
                  tmpOrient = Direction.LEFT;
                  tmpAlign = 1;
               }
            }
            else
            {
               tmpOrient = Direction.LEFT;
               tmpAlign =  ((double)  (p.y - bounds.y)) / bounds.height;
            }
         }
         else if (p.x >= bounds.x + bounds.width)
         {
            if (p.y < bounds.y)
            {
               if (p.x - bounds.x - bounds.width < bounds.y - p.y)
               {
                  tmpOrient = Direction.TOP;
                  tmpAlign = 1;
               }
               else
               {
                  tmpOrient = Direction.RIGHT;
                  tmpAlign = 0;
               }
            }
            else if (p.y >= bounds.y + bounds.height)
            {
               tmpAlign = 1;
               if (p.x - bounds.x - bounds.width < p.y - bounds.y - bounds.height)
               {
                  tmpOrient = Direction.BOTTOM;
               }
               else
               {
                  tmpOrient = Direction.RIGHT;
               }
            }
            else
            {
               tmpOrient = Direction.RIGHT;
               tmpAlign =  ((double)  (p.y - bounds.y)) / bounds.height;
            }
         }
         else
         {
            tmpAlign =  ((double)  (p.x - bounds.x)) / bounds.width;
            if (p.y <= bounds.y)
            {
               tmpOrient = Direction.TOP;
            }
            else
            {
               tmpOrient = Direction.BOTTOM;
            }
         }

         if (grab.isAutoOrientation())
         {
            orientation = tmpOrient;
            grab.setLayoutOrientation (orientation);
         }
         if (grab.isAutoAlignment())
         {
            align = tmpAlign;
            grab.setLayoutAlignment (align);
         }
      } // if (bend != null)

      return new double[]
         {
         orientation.asInt(), align
         };
   }
}

/*
 * $Log: GrabLayouter.java,v $
 * Revision 1.13  2004/10/20 17:49:45  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
