/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashSet;
import java.util.Iterator;

import javax.swing.*;

import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashSet;


/**
 * The GrabManager associates a number of Grabs to a target JComponent, manages events of the
 * target and calls the GrabLayouter to relayout the Grabs when necessary.<p>
 *
 * The association to the JComponent is implemented using the clientProperty-Methods of the
 * JComponent. This means that the association to the JComponent is implemented with the traditional
 * access methods, but the backward association is implemented as the client property with
 * name <pre>GrabManager.TARGET_PROPERTY</pre> of the target component. It is accessible via:
 * <p>
 *
 * <pre>
 *    (GrabManager)targetComponent.getClientProperty(GrabManager.TARGET_PROPERTY)
 * </pre> <p>
 *
 * <h2>Associations</h2> <pre>
 *              0..1                      N
 * GrabManager ----------------------------- JGrab
 *              manager               grabs
 *              0..1                                 N
 * GrabManager ---------------------------------------- JComponent
 *              getClientProperty               target
 * </pre>
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.28 $
 * @see       javax.swing.JComponent#getClientProperty
 * @see       javax.swing.JComponent#putClientProperty
 * @see       de.uni_paderborn.fujaba.fsa.swing.JGrab
 * @see       de.uni_paderborn.fujaba.fsa.swing.GrabLayouter
 */
public class GrabManager extends ComponentAdapter implements PropertyChangeListener,
   HierarchyListener, HierarchyBoundsListener
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String TARGET_PROPERTY = "GrabManager";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static int recursionDepth = 0;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static boolean postPoneing = false;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static HashSet postPonedManagers = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public static void postPoneGrabLayout()
   {
      if (!postPoneing)
      {
         postPoneing = true;
         postPonedManagers = new HashSet();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public static void doPostPonedGrabLayout()
   {
      if (postPoneing)
      {
         postPoneing = false;

         GrabManager tmpMan = null;
         Iterator iter = postPonedManagers.iterator();
         while (iter.hasNext())
         {
            tmpMan = (GrabManager) iter.next();
            tmpMan.revalidate (2);
         }
      }
   }


   /**
    * Constructor for class GrabManager
    */
   public GrabManager() { }


   /**
    * Constructor for class GrabManager
    *
    * @param target  No description provided
    */
   public GrabManager (JComponent target)
   {
      this();
      setTarget (target);
   }


   /**
    * <pre>
    *              0..1                                 N
    * GrabManager ---------------------------------------- JComponent
    *              getClientProperty               target
    *              (GrabManager.TARGET_PROPERTY)
    * </pre>
    */
   private JComponent target = null;


   /**
    * Set the target of this Manager (and all of its Grabs) to value.<p>
    *
    * Reverse Link is <br>
    * <pre>
    *    (GrabManager)value.getClientProperty(GrabManager.TARGET_PROPERTY)
    * </pre>
    *
    * @param value  The new target
    * @return       true if target was changed, false otherwise
    * @see          #getTarget
    */
   public boolean setTarget (JComponent value)
   {
      boolean changed = false;
      if (this.target != value)
      {
         GrabManager oldManager = null;
         JComponent oldValue = this.target;
         if (this.target != null)
         {
            unregisterTarget();
            if (value == null)
            {
               unregisterGrabs();
            }

            this.target = null;
            oldManager = (GrabManager) oldValue.getClientProperty (TARGET_PROPERTY);

            if (oldManager == this)
            {
               oldValue.putClientProperty (TARGET_PROPERTY, null);
            }

            oldValue.removePropertyChangeListener (TARGET_PROPERTY, this);
         }
         this.target = value;
         if (value != null)
         {
            oldManager = (GrabManager) value.getClientProperty (TARGET_PROPERTY);
            if (oldManager != null)
            {
               oldManager.setTarget (null);
            }
            value.putClientProperty (TARGET_PROPERTY, this);
            value.addPropertyChangeListener (TARGET_PROPERTY, this);
            if (sizeOfGrabs() != 0)
            {
               registerTarget();
               if (oldValue == null)
               {
                  registerGrabs();
               }
            }
         }
         revalidate();
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void unregisterTarget()
   {
      this.target.removeComponentListener (this);
      this.target.removePropertyChangeListener (GrabLayouter.TARGET_PROPERTY, this);
      this.target.removeHierarchyListener (this);
      this.target.removeHierarchyBoundsListener (this);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void registerTarget()
   {
      if (this.target != null)
      {
         this.target.addPropertyChangeListener (GrabLayouter.TARGET_PROPERTY, this);
         this.target.addComponentListener (this);
         this.target.addHierarchyListener (this);
         this.target.addHierarchyBoundsListener (this);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void registerGrabs()
   {
      Iterator grabIter = iteratorOfGrabs();
      while (grabIter.hasNext())
      {
         JGrab grab = (JGrab) grabIter.next();
         registerGrab (grab);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void unregisterGrabs()
   {
      Iterator grabIter = iteratorOfGrabs();
      while (grabIter.hasNext())
      {
         JGrab grab = (JGrab) grabIter.next();
         unregisterGrab (grab);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param grab  No description provided
    */
   private void registerGrab (JGrab grab)
   {
      grab.addHierarchyListener (this);
      grab.addComponentListener (this);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param grab  No description provided
    */
   private void unregisterGrab (JGrab grab)
   {
      grab.removeComponentListener (this);
      grab.removeHierarchyListener (this);
   }


   /**
    * @return   the JComponent this GrabManager is connected to
    */
   public JComponent getTarget()
   {
      return target;
   }


   /**
    * <pre>
    *              0..1                      N
    * GrabManager ----------------------------- JGrab
    *              manager               grabs
    * </pre>
    */
   private FHashSet grabs = null;


   /**
    * Add a Grab to the list of Grabs that are managed by this Object. Update the common ancestor
    * of all Grabs accordingly
    *
    * @param value  the Grab to add
    * @return       true if the Grab was added, false otherwise
    * @see          #updateCommonAncestorOfGrabs(java.awt.Container, boolean)
    */
   public boolean addToGrabs (JGrab value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (grabs == null)
         {
            grabs = new FHashSet();
         }
         changed = grabs.add (value);
         if (changed)
         {
            value.setManager (this);
            if (grabs.size() == 1)
            {
               registerTarget();
            }
            updateCommonAncestorOfGrabs (value.getParent(), false);
            registerGrab (value);
            revalidate();
         }
      }
      return changed;
   }


   /**
    * @param value
    * @return       No description provided
    */
   public boolean hasInGrabs (JGrab value)
   {
      return  ( (grabs != null) &&
          (value != null) &&
         grabs.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfGrabs()
   {
      return  ( (grabs == null)
         ? FEmptyIterator.get()
         : grabs.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfGrabs()
   {
      return  ( (grabs == null)
         ? 0
         : grabs.size());
   }


   /**
    * Removes the Grab value from the list of Grabs and updates the common ancestor of all
    * grabs
    *
    * @param value  the grab in question
    * @return       true if the list was changed
    * @see          #updateCommonAncestorOfGrabs(java.awt.Container, boolean)
    */
   public boolean removeFromGrabs (JGrab value)
   {
      boolean changed = false;

      if ( (grabs != null) &&  (value != null))
      {
         changed = grabs.remove (value);
         if (changed)
         {
            value.setManager (null);
            unregisterGrab (value);
            updateCommonAncestorOfGrabs (value.getParent(), true);
            if ( (grabs.size() == 0) &&  (target != null))
            {
               unregisterTarget();
            }
            revalidate();
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromGrabs()
   {
      JGrab tmpValue;
      Iterator iter = iteratorOfGrabs();

      commonAncestorOfGrabs = null;

      while (iter.hasNext())
      {
         tmpValue = (JGrab) iter.next();
         removeFromGrabs (tmpValue);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Container commonAncestorOfGrabs = null;


   /**
    * recalculate common ancestor from scratch by iterating over all grabs
    *
    * @param hint  No description provided
    * @see         #updateCommonAncestorOfGrabs(java.awt.Container, boolean)
    */
   private void updateCommonAncestorOfGrabs (Container hint)
   {
      commonAncestorOfGrabs = null;
      Iterator grabIter = iteratorOfGrabs();
      while (grabIter.hasNext())
      {
         updateCommonAncestorOfGrabs ( ((JGrab) grabIter.next()).getParent(), false);
         if (hint != null && commonAncestorOfGrabs == hint)
         {
            break;
         }
      }
   }


   /**
    * Update the common ancestor of all grabs by checking the parent value and comparing it
    * to the current value.<p>
    *
    * The commonAncestorOfGrabs is needed to decrease the work that is to be done when an AncestorEvent
    * occurs on the JComponent. If the ancestor in question is an ancestor of commonAncestorOfGrabs
    * nothing is to be done. Otherwise a revalidate() is necessary
    *
    * @param parent
    * @param removed
    * @see            #updateCommonAncestorOfGrabs(Container)
    */
   private void updateCommonAncestorOfGrabs (Container parent, boolean removed)
   {
      if (parent == null)
      {
         return;
      }

      if (!removed)
      {
         if (commonAncestorOfGrabs == null)
         {
            commonAncestorOfGrabs = parent;
         }
         else
         {
            commonAncestorOfGrabs = JBendLine.getClosestCommonAncestor (parent, commonAncestorOfGrabs);
         }
      }
      else
      {
         if (commonAncestorOfGrabs != null && SwingUtilities.isDescendingFrom (parent, commonAncestorOfGrabs))
         {
            updateCommonAncestorOfGrabs (commonAncestorOfGrabs);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean revalidating = false;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
//   private transient LayoutWorker lastLayoutWorker = null;

   /**
    * Checks the Target if a specific GrabLayouter is set as client property with name GrabLayouter.TARGET_PROPERTY.
    * If none is found the default Layouter is the result of GrabLayouter.get().<p>
    *
    * Then the layout-Method of the Layouter is called to reposition the grabs around the target
    * JComponent
    *
    * @see   de.uni_paderborn.fujaba.fsa.swing.GrabLayouter
    * @see   de.uni_paderborn.fujaba.fsa.swing.GrabLayouter#getDefaultLayouter
    * @see   de.uni_paderborn.fujaba.fsa.swing.GrabLayouter#layout(de.uni_paderborn.fujaba.fsa.swing.GrabManager)
    */
   public void revalidate()
   {
      revalidate (1);
   }


   /**
    * Checks the Target if a specific GrabLayouter is set as client property with name GrabLayouter.TARGET_PROPERTY.
    * If none is found the default Layouter is the result of GrabLayouter.get().<p>
    *
    * Then the layout-Method of the Layouter is called to reposition the grabs around the target
    * JComponent
    *
    * @param depthIncrease  recursion is limitted to 2.<br>
    *      depthIncrease 0 layouts with full recursion<br>
    *      depthIncrease 1 layouts this frame and its direct neighbors<br>
    *      depthIncrease 2 or greater layouts only this frame
    * @see                  de.uni_paderborn.fujaba.fsa.swing.GrabLayouter
    * @see                  de.uni_paderborn.fujaba.fsa.swing.GrabLayouter#getDefaultLayouter
    * @see                  de.uni_paderborn.fujaba.fsa.swing.GrabLayouter#layout(de.uni_paderborn.fujaba.fsa.swing.GrabManager)
    */
   public void revalidate (int depthIncrease)
   {
      if (!revalidating)
      {
         // trying to increase performance
         if (GrabManager.postPoneing)
         {
            GrabManager.postPonedManagers.add (this);
            return;
         }

         if (recursionDepth < 2)
         {
            recursionDepth += depthIncrease;
            revalidating = true;

            try
            {
               JComponent target = getTarget();
               if (target == null)
               {
                  return;
               }

               GrabLayouter layouter = GrabLayouter.getLayouter (target);
               layouter.layout (this);
            }
            catch (Exception ex)
            {
               ex.printStackTrace();
            }
            finally
            {
               revalidating = false;
               recursionDepth -= depthIncrease;
            }
         }
      }
   }


   // -------------------------  The Listener - Methods ----------------------
   /**
    * Listens for changes to Position or size of the target to revalidate the grabs and for
    * changes to the parents of the Grabs to update the common ancestor
    *
    * @param e  the event
    * @see      #revalidate()
    * @see      #updateCommonAncestorOfGrabs(java.awt.Container, boolean)
    */
   public void propertyChange (PropertyChangeEvent e)
   {
      /*
       *  log.info("PropertyChange: "+e.getPropertyName()+" from "+e.getSource());
       */
      if (TARGET_PROPERTY.equals (e.getPropertyName()))
      {
         if (e.getOldValue() == this && e.getNewValue() != this)
         {
            setTarget (null);
         }
      }
      revalidate();
   }


   /**
    * @param e  the event
    */
   public void componentMoved (ComponentEvent e)
   {
      if (e.getComponent() == getTarget())
      {
         revalidate();
      }
   }


   /**
    * @param e  the event
    */
   public void componentResized (ComponentEvent e)
   {
      Component source = e.getComponent();
      if (source == getTarget() || source instanceof JGrab)
      {
         revalidate();
      }
   }


   /**
    * @param e  the event
    */
   public void hierarchyChanged (HierarchyEvent e)
   {
      long flags = e.getChangeFlags();
      if ( (flags & HierarchyEvent.PARENT_CHANGED) != 0)
      {
         Object source = e.getSource();
         Container parent = e.getChangedParent();

         if (e.getChanged().getParent() == null)
         {
            //ancestor removed
            if (source != target)
            {
               //Grab
               updateCommonAncestorOfGrabs (parent, false);
            }
         }
         else
         {
            //ancestor added
            if (source == target)
            {
               if (commonAncestorOfGrabs != null && SwingUtilities.isDescendingFrom (parent, commonAncestorOfGrabs))
               {
                  revalidate();
               }
            }
            else
            {
               //Grab
               Component oldValue = commonAncestorOfGrabs;
               updateCommonAncestorOfGrabs (parent, false);
               if (oldValue != commonAncestorOfGrabs)
               {
                  revalidate();
               }
            }
         }
      }
   }


   /**
    * Listen for changes to ancestors of the target.<p>
    *
    * If a parent of the common ancestor of grabs is changed there is nothing to do, because
    * the changes affect the grabs, too. Otherwise the grabs have to be relayouted
    *
    * @param e  The event
    * @see      #revalidate()
    * @see      #updateCommonAncestorOfGrabs(Container)
    */
   public void ancestorMoved (HierarchyEvent e)
   {
      if (commonAncestorOfGrabs == null)
      {
         return;
      }

      Component ancestor = e.getChanged();

      if (!SwingUtilities.isDescendingFrom (commonAncestorOfGrabs, ancestor))
      {
         revalidate();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void ancestorResized (HierarchyEvent e)
   {
   }
}


/**
 * No comment provided by developer, please add a comment to improve documentation. GrabManager.java
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.28 $
 */
class LayoutWorker implements Runnable
{


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   GrabLayouter layouter;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   GrabManager manager;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int INITIALIZED = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int RUNNING = 2;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int FINISHED = 3;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int state = INITIALIZED;


   /**
    * Constructor for class LayoutWorker
    *
    * @param layouter  No description provided
    * @param manager   No description provided
    */
   public LayoutWorker (GrabLayouter layouter, GrabManager manager)
   {
      this.layouter = layouter;
      this.manager = manager;
   }


   /**
    * Main processing method for the LayoutWorker object
    */
   public void run()
   {
      synchronized (this)
      {
         this.state = RUNNING;
      }
      if (layouter != null && manager != null)
      {
         layouter.layout (manager);
      }
      synchronized (this)
      {
         this.state = FINISHED;
      }
   }


   /**
    * Get the state attribute of the LayoutWorker object
    *
    * @return   The state value
    */
   public synchronized int getState()
   {
      return this.state;
   }

}

/*
 * $Log: GrabManager.java,v $
 * Revision 1.28  2004/10/20 17:49:45  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
