/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.util.ArrayList;

import javax.swing.*;


/**
 * <pre>
 * Prototype class for arrow head styles to define shape, fill style, tip,
 * and base point. The arrowhead will be rotated around its head point.
 * The tip of the arrowhead is connected to the grab of the adjacent object.
 * The line of the arrow is attached to the base of the arrowhead.
 * The arrowhead can be filled or hollow.
 *
 * TODO: limitations: width > 0, height >= width
 *
 * example:
 *
 *     width +-------+
 *           |       |
 *               o <----------- head point
 *        +-     *
 *        |     ***
 * height |    *****
 *        |   *******
 *        +- ****o****
 *               |
 *               +------------- base point
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.18 $
 */
public abstract class JArrowHead extends JComponent implements Cloneable
{
   /**
    * Caches the shapes of this arrow. This list may contain subclasses of the java.awt.Shape
    * interface or a int [2][n] array containing a point list. The first element of the first
    * dimension contains the x coordinates and the second element of the second dimension contains
    * the y coordinates.
    */
   private ArrayList shapeList = new ArrayList (2);

   /**
    * The arrow head is filled when true, hollow otherwise.
    */
   private boolean filled;

   /**
    * Direction of the Arrow
    */
   private Direction direction = Direction.TOP;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected AffineTransform at = new AffineTransform();


   /**
    * Constructor for class JArrowHead
    */
   JArrowHead() { }


   /**
    * @param style  No description provided
    */
   JArrowHead (int style)
   {
      super();
      setStyle (style);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   The basePoint value
    */
   public Point getBasePoint()
   {
      return (Point) at.transform (new Point ( (getHeight() - 1) / 2, getHeight() - 1), new Point());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   The headPoint value
    */
   public Point getHeadPoint()
   {
      return (Point) at.transform (new Point ( (getHeight() - 1) / 2, -1), new Point());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   The shapeList value
    */
   public ArrayList getShapeList()
   {
      return shapeList;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param filled  The new filled value
    */
   protected void setFilled (boolean filled)
   {
      this.filled = filled;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   The filled value
    */
   public boolean getFilled()
   {
      return filled;
   }


   /**
    * Set the direction attribute of the JArrowHead object
    *
    * @param direction  The new direction value
    * @return           No description provided
    */
   public boolean setDirection (Direction direction)
   {
      if (direction == Direction.UNDEFINED)
      {
         throw new IllegalArgumentException ("Direction is undefined");
      }
      else
      {
         if (direction != this.direction)
         {
            this.direction = direction;
            if (direction == Direction.TOP)
            {
               at.setToIdentity();
            }
            else if (direction == Direction.LEFT)
            {
               at.setToIdentity();
               at.translate (getHeight() / 2, getHeight() / 2);
               at.rotate (Math.toRadians (-90.0));
               at.translate (-getHeight() / 2, -getHeight() / 2);
            }
            else if (direction == Direction.BOTTOM)
            {
               at.setToIdentity();
               at.setToIdentity();
               at.translate (getHeight() / 2, getHeight() / 2);
               at.rotate (Math.toRadians (180.0));
               at.translate (-getHeight() / 2, -getHeight() / 2);
            }
            else
            {
               at.setToIdentity();
               at.translate (getHeight() / 2, getHeight() / 2);
               at.rotate (Math.toRadians (90.0));
               at.translate (-getHeight() / 2, -getHeight() / 2);
            }

            return true;
         }
         else
         {
            return false;
         }
      }
   }


   /**
    * Get the direction attribute of the JArrowHead object
    *
    * @return   The direction value
    */
   public Direction getDirection()
   {
      return direction;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    */
   public void paintComponent (Graphics g)
   {
      super.paintComponent (g);
      paintArrow (g);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    */
   public abstract void paintArrow (Graphics g);


   /**
    * Clone the arrow head for using the Prototype pattern. See Gamma et. al: Design Patterns,
    * pp. 117ff
    *
    * @return   No description provided
    */
   public abstract Object clone();


   // Here are the deprecated old-style definitions
   /**
    * @deprecated   Use specializations and attributes instead
    */
   private int style = 0;

   /**
    * @deprecated   Use attribute size or width and height instead
    */
   public final static int BIG = 1;

   /**
    * @deprecated   Use attribute size or width and height instead
    */
   public final static int SMALL = 0;

   /**
    * @deprecated   Use attribute filled=true instead
    */
   public final static int FILLED = 2;

   /**
    * @deprecated   Use attribute filled=false instead
    */
   public final static int HOLLOW = 0;

   /**
    * @deprecated   Use class JSimpleArrow instead
    */
   public final static int OPEN_ARROW = 0;

   /**
    * @deprecated   Use class JTriangleHead instead
    */
   public final static int TRIANGLE = 0;

   /**
    * @deprecated   Use class JArrowDoubleLine instead
    */
   public final static int DOUBLE_LINE = 8;


   /**
    * @param style  The new style value
    * @deprecated   Clone prototype of JArrowHead specialization instead
    */
   public void setStyle (int style)
   {
      // @todo This method does nothing. Lookup your prototype here
      switch (style)
      {
         case 0:
         case 8:
            // this = new JTriangleHead (13,13,false);
            break;
         case 1:
         case 9:
            // this = new JTriangleHead (17,17, false);
            break;
         case 2:
         case 6:
         case 10:
         case 14:
            // this = new JTriangleHead(13,13,true);
            break;
         case 3:
         case 7:
         case 11:
         case 15:
            // this = new JTriangleHead(17,17,true);
            break;
         case 4:
            // this = new JSimpleArrowHead (13,13);
            break;
         case 5:
            // this = new JSimpleArrowHead (17,17);
            break;
         case 12:
            // this = new JArrowDoubleLineHead (13,13);
            break;
         case 13:
            // this = new JArrowDoubleLineHead (17,17);
            break;
         default:
         // this = new JSimpleArrowHead (17,17);
      }
      this.style = style;
   }


   /**
    * @return       The style value
    * @deprecated   Obsolete
    */
   int getStyle()
   {
      return style;
   }

}

/*
 * $Log: JArrowHead.java,v $
 * Revision 1.18  2004/11/22 19:04:54  lowende
 * Some javadoc corrections.
 *
 */
