/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.geom.Rectangle2D;
import java.util.Comparator;

import javax.swing.*;

import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.fsa.listener.AscendDescendMouseHandler;
import de.uni_paderborn.fujaba.fsa.listener.CollapsableMouseListener;
import de.uni_paderborn.fujaba.preferences.ColorsPreferences;


/**
 * A JCollapsable is a panel with a small button in the lower right corner of the panel.
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.21 $
 */
public class JCollapsable extends JScrollPane
{
   /**
    * The value of the scrollbar Width. When the width of an existing class is smaller than
    * this size the horizontal scrollbar is not visible.
    */
   private static int COLLAPSABLE_WIDTH = 150;

   /**
    * The value of the scrollbar Height. When the height of an existing class is smaller than
    * this size the horizontal scrollbar is not visible.
    */
   private static int COLLAPSABLE_HEIGHT = 60;


   /**
    * constructs the internal used JPanel and the internal used JScrollPane.
    */
   public JCollapsable()
   {
      super();
      viewPanel = new JPanel();
      viewPanel.setLayout (new ColumnRowLayout (0, ColumnRowLayout.COLUMN));

      setViewportView (viewPanel);
      setBorder (null);

      JScrollBar bar = getHorizontalScrollBar();
      Dimension dim = bar.getPreferredSize();
      dim.height = 9;
      bar.setPreferredSize (dim);

      JScrollBar bar2 = getVerticalScrollBar();
      dim = bar.getPreferredSize();
      dim.width = 9;
      bar2.setPreferredSize (dim);

      // create the internal scrollpane.
      setVerticalScrollBarPolicy (VERTICAL_SCROLLBAR_AS_NEEDED);
      setHorizontalScrollBarPolicy (HORIZONTAL_SCROLLBAR_AS_NEEDED);
      AscendDescendMouseHandler.addMouseListener (this, CollapsableMouseListener.get());
      AscendDescendMouseHandler.registerADMouseInputHandler (this);
   }


   /**
    * Get the layoutComparator attribute of the JCollapsable object
    *
    * @return   The layoutComparator value
    */
   public Comparator getLayoutComparator()
   {
      Comparator result = null;
      LayoutManager layout =  (this.viewPanel == null ? null : this.viewPanel.getLayout());
      if (layout instanceof SortedLayout)
      {
         result =  ((SortedLayout) layout).getComparator();
      }
      return result;
   }


   /**
    * Sets the layoutComparator attribute of the JCollapsable object
    *
    * @param comp  The new layoutComparator value
    */
   public void setLayoutComparator (Comparator comp)
   {
      LayoutManager layout =  (this.viewPanel == null ? null : this.viewPanel.getLayout());
      if (layout instanceof SortedLayout)
      {
          ((SortedLayout) layout).setComparator (comp);
      }
      else
      {
         throw new IllegalStateException ("LayoutManager is no SortedLayout");
      }
   }


   /**
    * Internal used attribute to store the minimum size of the panel.
    */
   private Dimension minSize = null;

   /**
    * the internal used JPanel.
    */
   private JPanel viewPanel;


   /**
    * sets the background of the JCollapsable, the JScrollPane and the JPanel.
    *
    * @param color  The new background value
    */
   public void setBackground (Color color)
   {
      super.setBackground (color);
      if (viewPanel != null)
      {
         viewPanel.setBackground (color);
      } // end of if ()
   }


   /**
    * Sets the foreground of the JCollapsable, the JScrollPane and the JPanel.
    *
    * @param color  The new foreground value
    */
   public void setForeground (Color color)
   {
      super.setForeground (color);
      if (viewPanel != null)
      {
         viewPanel.setForeground (color);
      } // end of if ()
   }


   /**
    * Sets the layout attribute of the JCollapsable object
    *
    * @param layout  The new layout value
    */
   public void setLayout (LayoutManager layout)
   {
      if (viewPanel == null)
      {
         super.setLayout (layout);
      } // end of if ()
      else
      {
         viewPanel.setLayout (layout);
      } // end of else
   }


   /**
    */
   private int insetWidth = 0;


   /**
    * get the value of insetWidth.
    *
    * @return   the value of insetWidth
    */
   public int getInsetWidth()
   {
      return this.insetWidth;
   } // getInsetWidth


   /**
    * set the value of insetWidth.
    *
    * @param value  The new insetWidth value
    */
   public void setInsetWidth (int value)
   {
      if (this.insetWidth != value)
      {
         this.insetWidth = value;
      }
   }


   /**
    * Overwrites the paint method of the JScrollPane.
    *
    * @param g  No description provided
    */
   public void paint (Graphics g)
   {
      super.paint (g);

      Rectangle bounds = getBounds();
      int x = bounds.width - 9 - getInsetWidth();
      int y = bounds.height - 9;
      int width = 9;
      int height = 9;

      Color oldColor = g.getColor();

      g.setColor (FSAObject.COLOR_BACKGROUND);
      g.fill3DRect (x, y, width, height, true);

      g.setColor (ColorsPreferences.get().COLLAPSE_ANCHOR);
      g.draw3DRect (x, y, width, height, true);
      int[] xCoordsCollapse = {x + 2,
         x + 6};

      int[] yCoordsCollapse = {y + 4,
         y + 4};

      g.drawPolygon (xCoordsCollapse, yCoordsCollapse, 2);
      if (isCollapsed())
      {
         int[] xCoordsCollapse2 = {x + 4,
            x + 4};

         int[] yCoordsCollapse2 = {y + 2,
            y + 6};
         g.drawPolygon (xCoordsCollapse2, yCoordsCollapse2, 2);

         // write "Collapsed" into compartment
         String collapsed = "collapsed";
         Font currentFont = g.getFont();
         Font italicFont = currentFont.deriveFont (Font.ITALIC);

         g.setColor (Color.BLACK);
         g.setFont (italicFont);
         Rectangle2D stringBounds = g.getFontMetrics().getStringBounds (collapsed, g);
         g.drawString (collapsed, (int)  ( (bounds.width - stringBounds.getWidth()) / 2), (int)  ( (bounds.height / 2) +  (stringBounds.getHeight() / 2)));
         g.setFont (currentFont);
      } // end of if ()

      g.setColor (oldColor);
   }


   /**
    * status flag if the JCollapsable is collapsed or not.
    */
   private boolean collapsed = false;


   /**
    * returns true if the JCollapsable is collapsed.
    *
    * @return   The collapsed value
    */
   public boolean isCollapsed()
   {
      return collapsed;
   }


   /**
    * changes the collapsed flag.
    *
    * @param collapsed  The new collapsed value
    */
   public void setCollapsed (boolean collapsed)
   {
      if (this.collapsed != collapsed)
      {
         boolean oldValue = this.collapsed;
         this.collapsed = collapsed;
         firePropertyChange ("collapsed", oldValue, collapsed);
         viewPanel.setVisible (!collapsed);
         getViewport().setVisible (!collapsed);
         if (collapsed)
         {
            setHorizontalScrollBarPolicy (ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
            setVerticalScrollBarPolicy (ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER);
         }
         else
         {
            setHorizontalScrollBarPolicy (ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
            setVerticalScrollBarPolicy (ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);
         }
      } // end of if ()
   }


   /**
    * changes the collapsed flag to !isCollapsed().
    */
   public void toggleCollapsed()
   {
      setCollapsed (!isCollapsed());
   }


   /**
    * delegates the add call to its internal JPanel.
    *
    * @param comp  No description provided
    * @return      No description provided
    */
   public Component add (Component comp)
   {
      if (viewPanel == null)
      {
         return super.add (comp);
      } // end of if ()
      return viewPanel.add (comp);
   }


   /**
    * delegates the add call to its internal JPanel.
    *
    * @param name  No description provided
    * @param comp  No description provided
    * @return      No description provided
    */
   public Component add (String name, Component comp)
   {
      if (viewPanel == null)
      {
         return super.add (name, comp);
      } // end of if ()
      return viewPanel.add (name, comp);
   }


   /**
    * delegates the add call to its internal JPanel.
    *
    * @param comp   No description provided
    * @param index  No description provided
    * @return       No description provided
    */
   public Component add (Component comp, int index)
   {
      if (viewPanel == null)
      {
         return super.add (comp, index);
      } // end of if ()
      return viewPanel.add (comp, index);
   }


   /**
    * delegates the add call to its internal JPanel.
    *
    * @param comp         No description provided
    * @param constraints  No description provided
    */
   public void add (Component comp, Object constraints)
   {
      if (viewPanel == null)
      {
         super.add (comp, constraints);
      } // end of if ()
      else
      {
         viewPanel.add (comp, constraints);
      } // end of else
   }


   /**
    * delegates the add call to its internal JPanel.
    *
    * @param comp         No description provided
    * @param constraints  No description provided
    * @param index        No description provided
    */
   public void add (Component comp, Object constraints, int index)
   {
      if (viewPanel == null)
      {
         super.add (comp, constraints, index);
      } // end of if ()
      else
      {
         viewPanel.add (comp, constraints, index);
      } // end of else
   }


   /**
    * delegates the addImpl call to its internal JPanel.
    *
    * @param comp         The object added.
    * @param constraints  The object added.
    * @param index        The object added.
    */
   public void addImpl (Component comp, Object constraints, int index)
   {
      if (viewPanel == null)
      {
         super.addImpl (comp, constraints, index);
      } // end of if ()
      else
      {
         viewPanel.add (comp, constraints, index);
      } // end of else
   }


   /**
    * Overwrites the getPreferredSize of JComponent. if the status of this component is collapsed
    * the collapsed size is given back, the not collapsed size otherwise.
    *
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize()
   {
      Dimension result = super.getPreferredSize();

      if (viewPanel != null)
      {
         result = (Dimension) viewPanel.getPreferredSize().clone();
         result.width += 12;
      }

      if (isCollapsed())
      {
         result.height = 15;
         result.width = 9;
      } // end of if()

      return result;
   }


   /**
    * Overwrites the method getMinimumSize of JComponent.
    *
    * @return   the collapsed size if this panel is collapsed or the minimum size of its father
    *      JScrollbar.
    */
   public Dimension getMinimumSize()
   {
      if (minSize == null)
      {
         minSize = new Dimension();
      } // end of if ()

      minSize.width = COLLAPSABLE_WIDTH;
      minSize.height = COLLAPSABLE_HEIGHT;

      Dimension maxDim = getMaximumSize();

      minSize.width = Math.min (minSize.width, maxDim.width);
      minSize.height = Math.min (minSize.height, maxDim.height);

      if (isCollapsed())
      {
         minSize.height = 15;
         minSize.width = 9;
      } // end of if ()

      return minSize;
   }


   /**
    * Overwrites the method getMaximumSize of JComponent.
    *
    * @return   the collapsed size if this panel is collapsed or the preferred size of the
    *      internal used viewPanel.
    */
   public Dimension getMaximumSize()
   {
      Dimension tmpDim = null;

      if (isCollapsed())
      {
         tmpDim = getPreferredSize();
      } // end of if ()
      else
      {
         tmpDim = viewPanel.getPreferredSize();
      } // end of else
      return tmpDim;
   }
}

/*
 * $Log: JCollapsable.java,v $
 * Revision 1.21  2004/10/20 17:49:45  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
