/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;

import javax.swing.*;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.17 $
 */
public class JDiagramRootPane extends JComponent
{
   /**
    * The layered pane that manages the menu bar and content pane.
    */
   JComponent contentPane;

   /**
    * The glass pane that overlays the menu bar and content pane, so it can intercept mouse
    * movements and such.
    */
   JComponent glassPane;


   /**
    * Create a JDiagramPane, setting up its glassPane, and contentPane.
    */
   public JDiagramRootPane()
   {
      JComponent gp = createGlassPane();
      setGlassPane (gp);
      setContentPane (createContentPane());
      setLayout (createLayout());
      setDoubleBuffered (true);
      //setCatchFindComponents(true);
   }


   /**
    * Called by the constructor methods to create the default contentPane. By default it creates
    * a new JLayeredPane.
    *
    * @return   No description provided
    */
   protected JComponent createContentPane()
   {
      JComponent p = new JLayeredPane();
      p.setName (this.getName() + ".contentPane");
      return p;
   }


   /**
    * Called by the constructor methods to create the default glassPane. By default this method
    * creates a new JPanel with visibility set to false.
    *
    * @return   No description provided
    */
   protected JComponent createGlassPane()
   {
      JComponent c =
         new JComponent()
         {
            /**
             * does not paint anything
             *
             * @param g  the <code>Graphics</code> context in which to paint
             */
            public void paintComponent (Graphics g)
            {
            }
         };
      c.setName (this.getName() + ".glassPane");

      // set to false, cursor won't be changed else
      c.setVisible (false);

      c.setOpaque (false);

      return c;
   }


   /**
    * Called by the constructor methods to create the default layoutManager.
    *
    * @return   No description provided
    */
   protected LayoutManager createLayout()
   {
      return new DefaultLayout();
   }


   /**
    * Set the content pane for the pane.
    *
    * @param layered  the JComponent to use.
    */
   public void setContentPane (JComponent layered)
   {
      if (layered == null)
      {
         throw new IllegalComponentStateException ("contentPane cannot be set to null.");
      }

      if (contentPane != null && contentPane.getParent() == this)
      {
         this.remove (contentPane);
      }

      contentPane = layered;

      this.add (contentPane, -1);
   }


   /**
    * Get the content pane used by the pane.
    *
    * @return   the JComponent currently in use
    */
   public JComponent getContentPane()
   {
      return contentPane;
   }


   /**
    * Sets a specified Component to be the glass pane for this root pane. The glass pane should
    * normally be a lightweight, transparent component, because it will be made visible when
    * ever the root pane needs to grab input events. For example, only one JInternalFrame is
    * ever active when using a DefaultDesktop, and any inactive JInternalFrames' glass panes
    * are made visible so that clicking anywhere within an inactive JInternalFrame can activate
    * it.
    *
    * @param glass  the Component to use as the glass pane for this JRootPane.
    */
   public void setGlassPane (JComponent glass)
   {
      if (glass == null)
      {
         throw new IllegalComponentStateException ("glassPane cannot be set to null.");
      }

      boolean visible = false;
      if (glassPane != null && glassPane.getParent() == this)
      {
         this.remove (glassPane);
         visible = glassPane.isVisible();
      }

      glassPane = glass;
      this.add (glassPane, 0);
      if (visible || glass.isVisible())
      {
         repaint();
      }
   }


   /**
    * Returns the current glass pane for this JDiagramRootPane.
    *
    * @return   the current glass pane.
    * @see      #setGlassPane
    */
   public JComponent getGlassPane()
   {
      return glassPane;
   }


   /**
    * The GlassPane and ContentPane have the same bounds, which means JRootPane does not tiles
    * its children and this should return false. On the other hand, the GlassPane is normally
    * not visible, and so this can return true if the GlassPane isn't visible. Therefore, the
    * return value here depends upon the visiblity of the GlassPane.
    *
    * @return   true if this component's children don't overlap
    */
   public boolean isOptimizedDrawingEnabled()
   {
      return !glassPane.isVisible();
   }


   /**
    * Overridden to enforce the position of the glass component as the zero child.
    *
    * @param comp         The object added.
    * @param constraints  The object added.
    * @param index        The object added.
    */
   protected void addImpl (Component comp, Object constraints, int index)
   {
      super.addImpl (comp, constraints, index);

      // We are making sure the glassPane is on top.
      if (glassPane != null
         && glassPane.getParent() == this
         && getComponent (0) != glassPane)
      {
         add (glassPane, 0);
      }
   }


   /**
    * Make this be the root of a focus cycle. That means that, by default, tabbing within the
    * pane will move between components of the pane, but not out of the pane.
    *
    * @return   true
    * @see      JComponent#isFocusCycleRoot()
    */
   public boolean isFocusCycleRoot()
   {
      return true;
   }


   /**
    * A custom layout manager that is responsible for the layout of contentPane and glassPane.
    * <p>
    *
    * <strong>Warning:</strong> Serialized objects of this class will not be compatible with
    * future Swing releases. The current serialization support is appropriate for short term
    * storage or RMI between applications running the same version of Swing. A future release
    * of Swing will provide support for long term persistence.
    *
    * @author    $Author: schneider $
    * @version   $Revision: 1.17 $
    */
   protected class DefaultLayout implements LayoutManager2
   {
      /**
       * Returns the amount of space the layout would like to have.
       *
       * @param parent  the Container for which this layout manager is being used
       * @return        a Dimension object containing the layout's preferred size
       */
      public Dimension preferredLayoutSize (Container parent)
      {
         Dimension rd;
         Insets insets = getInsets();

         rd = parent.getSize();

         if (contentPane != null)
         {
            Dimension tmpDim = null;
            try
            {
               tmpDim = contentPane.getPreferredSize();
            }
            catch (Exception e)
            {
               // ignore
               tmpDim = new Dimension (200, 400);
            }
            tmpDim.width += insets.left + insets.right;
            tmpDim.height += insets.top + insets.bottom;

            rd.width = Math.max (rd.width, tmpDim.width);
            rd.height = Math.max (rd.height, tmpDim.height);
         }
         return rd;
      }


      /**
       * Returns the minimum amount of space the layout needs.
       *
       * @param parent  the Container for which this layout manager is being used
       * @return        a Dimension object containing the layout's minimum size
       */
      public Dimension minimumLayoutSize (Container parent)
      {
         Dimension rd;
         Insets insets = getInsets();

         rd = parent.getSize();

         if (contentPane != null)
         {
            Dimension tmpDim = contentPane.getMinimumSize();
            tmpDim.width += insets.left + insets.right;
            tmpDim.height += insets.top + insets.bottom;

            rd.width = Math.max (rd.width, tmpDim.width);
            rd.height = Math.max (rd.height, tmpDim.height);
         }
         return rd;
      }


      /**
       * Returns the maximum amount of space the layout can use.
       *
       * @param parent  No description provided
       * @return        a Dimension object containing the layout's maximum size
       */
      public Dimension maximumLayoutSize (Container parent)
      {
         Dimension rd;
         Insets insets = getInsets();

         if (contentPane != null)
         {
            rd = contentPane.getMaximumSize();
            if (rd.width <= Integer.MAX_VALUE -  (insets.left + insets.right))
            {
               rd.width += insets.left + insets.right;
            }

            if (rd.height <= Integer.MAX_VALUE -  (insets.top + insets.bottom))
            {
               rd.height += insets.top + insets.bottom;
            }
         }
         else
         {
            rd = parent.getSize();
         }

         return rd;
      }


      /**
       * Instructs the layout manager to perform the layout for the specified container.
       *
       * @param parent  No description provided
       */
      public void layoutContainer (Container parent)
      {
         Rectangle b = parent.getBounds();
         Insets i = getInsets();
         int w = b.width - i.right - i.left;
         int h = b.height - i.top - i.bottom;

         if (contentPane != null)
         {
            contentPane.setBounds (i.left, i.top, w, h);
         }
         if (glassPane != null)
         {
            glassPane.setBounds (i.left, i.top, w, h);
         }
      }


      /**
       * Access method for an one to n association.
       *
       * @param name  The object added.
       * @param comp  The object added.
       */
      public void addLayoutComponent (String name, Component comp) { }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param comp  No description provided
       */
      public void removeLayoutComponent (Component comp) { }


      /**
       * Access method for an one to n association.
       *
       * @param comp         The object added.
       * @param constraints  The object added.
       */
      public void addLayoutComponent (Component comp, Object constraints) { }


      /**
       * Get the layoutAlignmentX attribute of the DefaultLayout object
       *
       * @param target  No description provided
       * @return        The layoutAlignmentX value
       */
      public float getLayoutAlignmentX (Container target)
      {
         return 0.0f;
      }


      /**
       * Get the layoutAlignmentY attribute of the DefaultLayout object
       *
       * @param target  No description provided
       * @return        The layoutAlignmentY value
       */
      public float getLayoutAlignmentY (Container target)
      {
         return 0.0f;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param target  No description provided
       */
      public void invalidateLayout (Container target) { }
   }
}

/*
 * $Log: JDiagramRootPane.java,v $
 * Revision 1.17  2004/10/20 17:49:46  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
