/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.awt.geom.Point2D;
import java.util.*;

import javax.swing.*;
import javax.swing.plaf.ComponentUI;

import de.uni_paderborn.fujaba.basic.AbstractListIterator;


/**
 * A Line from startPoint to endPoint
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.22.2.1 $
 * @see       de.uni_paderborn.fujaba.fsa.swing.JBendLine
 */
public class JLine extends JComponent
{
   /**
    * Creates a Line from (0,0) to (0,0)
    */
   public JLine()
   {
      super();

      setUI (DefaultLineUI.createUI (this));
      setLayout (new DecoratorLayout (true, false));

      setOpaque (false);
      startPoint = new Point (0, 0);
      endPoint = new Point (0, 0);
      linePointsChanged();
   }


   /**
    * Creates a Line from Point start to Point end
    *
    * @param start  start of line
    * @param end    end of line
    */
   public JLine (Point start, Point end)
   {
      this();
      setStartPoint (start);
      setEndPoint (end);
   }


   /**
    * Get the optimizedDrawingEnabled attribute of the JLine object
    *
    * @return   The optimizedDrawingEnabled value
    */
   public boolean isOptimizedDrawingEnabled()
   {
      return getComponentCount() <= 1;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Point startPoint = null;


   /**
    * Get the value of startPoint.<p>
    *
    * Same as getStartPoint(null).
    *
    * @return   Value of startPoint.
    * @see      #getStartPoint(java.awt.Point)
    * @see      #setStartPoint(java.awt.Point)
    * @see      #setStartPoint(int, int)
    * @see      #getEndPoint(java.awt.Point)
    * @see      #getEndPoint()
    */
   public final Point getStartPoint()
   {
      return getStartPoint (null);
   }


   /**
    * Store the value of startPoint in p or return a new Point containing these coordinates
    * if p is null
    *
    * @param p  The point to store the coordinates in
    * @return   p if p was non-null, a new Point otherwise
    * @see      #getStartPoint()
    * @see      #setStartPoint(java.awt.Point)
    * @see      #setStartPoint(int, int)
    * @see      #getEndPoint(java.awt.Point)
    * @see      #getEndPoint()
    */
   public Point getStartPoint (Point p)
   {
      if (p == null)
      {
         if (startPoint == null)
         {
            return null;
         }

         return new Point (startPoint);
      }

      p.x = startPoint.x;
      p.y = startPoint.y;

      return p;
   }


   /**
    * Set the value of startPoint and recalculate bounds
    *
    * @param point  Value to assign to startPoint.
    * @see          #setStartPoint(int, int)
    * @see          #setEndPoint(int, int)
    * @see          #setEndPoint(java.awt.Point)
    * @see          #getStartPoint()
    * @see          #getStartPoint(java.awt.Point)
    * @see          #setBounds(int, int, int, int)
    */
   public final void setStartPoint (Point point)
   {
      setStartPoint (point.x, point.y);
   }


   /**
    * Set the value of startPoint and recalculate bounds
    *
    * @param x  x-coordinate
    * @param y  y-coordinate
    * @return   No description provided
    * @see      #setStartPoint(java.awt.Point)
    * @see      #setEndPoint(int, int)
    * @see      #setEndPoint(java.awt.Point)
    * @see      #getStartPoint()
    * @see      #getStartPoint(java.awt.Point)
    * @see      #setBounds(int, int, int, int)
    */
   public boolean setStartPoint (int x, int y)
   {
      if (startPoint == null || startPoint.x != x || startPoint.y != y)
      {
         if (startPoint == null)
         {
            startPoint = new Point (x, y);
         }
         else
         {
            startPoint.x = x;
            startPoint.y = y;
         }

         linePointsChanged();
         repaint();
         return true;
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Point endPoint = null;


   /**
    * Get the value of endPoint.<br>
    * Same as getEndPoint(null).
    *
    * @return   Value of endPoint.
    * @see      #getEndPoint(java.awt.Point)
    * @see      #setEndPoint(java.awt.Point)
    * @see      #setEndPoint(int, int)
    * @see      #getStartPoint(java.awt.Point)
    * @see      #getStartPoint()
    */
   public final Point getEndPoint()
   {
      return getEndPoint (null);
   }


   /**
    * Store the value of endPoint in p or return a new Point containing these coordinates if
    * p is null
    *
    * @param p  The point to store the coordinates in
    * @return   p if p was non-null, a new Point otherwise
    * @see      #getEndPoint()
    * @see      #setEndPoint(java.awt.Point)
    * @see      #setEndPoint(int, int)
    * @see      #getStartPoint(java.awt.Point)
    * @see      #getStartPoint()
    */
   public Point getEndPoint (Point p)
   {
      if (p == null)
      {
         if (endPoint == null)
         {
            return null;
         }

         return new Point (endPoint);
      }

      p.x = endPoint.x;
      p.y = endPoint.y;

      return p;
   }


   /**
    * Set the value of endPoint and recalculate bounds
    *
    * @param point  Value to assign to endPoint.
    * @see          #setEndPoint(int, int)
    * @see          #setStartPoint(int, int)
    * @see          #setStartPoint(java.awt.Point)
    * @see          #getEndPoint()
    * @see          #getEndPoint(java.awt.Point)
    * @see          #setBounds(int, int, int, int)
    */
   public final void setEndPoint (Point point)
   {
      setEndPoint (point.x, point.y);
   }


   /**
    * Set the value of startPoint and recalculate bounds
    *
    * @param x  x-coordinate
    * @param y  y-coordinate
    * @return   No description provided
    * @see      #setEndPoint(java.awt.Point)
    * @see      #setStartPoint(int, int)
    * @see      #setStartPoint(java.awt.Point)
    * @see      #getEndPoint()
    * @see      #getEndPoint(java.awt.Point)
    * @see      #setBounds(int, int, int, int)
    */
   public boolean setEndPoint (int x, int y)
   {
      if (endPoint == null || endPoint.x != x || endPoint.y != y)
      {
         if (endPoint == null)
         {
            endPoint = new Point (x, y);
         }
         else
         {
            endPoint.x = x;
            endPoint.y = y;
         }

         linePointsChanged();
         repaint();
         return true;
      }
      return false;
   }


   /**
    * Sets the pointAt attribute of the JLine object
    *
    * @param index  The new pointAt value
    * @param p      The new pointAt value
    * @return       No description provided
    */
   public final boolean setPointAt (int index, Point p)
   {
      return setPointAt (index, p.x, p.y);
   }


   /**
    * Sets the pointAt attribute of the JLine object
    *
    * @param index  The new pointAt value
    * @param x      The new pointAt value
    * @param y      The new pointAt value
    * @return       No description provided
    */
   public boolean setPointAt (int index, int x, int y)
   {
      boolean changed = false;
      if (index == 0)
      {
         changed = setStartPoint (x, y);
      }
      else if (index == 1)
      {
         changed = setEndPoint (x, y);
      }
      else
      {
         throw new ArrayIndexOutOfBoundsException (index);
      }

      return changed;
   }


   /**
    * Get the fromPoints attribute of the JLine object
    *
    * @param index  No description provided
    * @return       The fromPoints value
    */
   public Point getFromPoints (int index)
   {
      if (index == 0)
      {
         return getStartPoint();
      }
      else if (index == 1)
      {
         return getEndPoint();
      }
      else
      {
         throw new ArrayIndexOutOfBoundsException (index);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param start  No description provided
    * @return       No description provided
    */
   public ListIterator iteratorOfPoints (int start)
   {
      return
         new AbstractListIterator (start)
         {
            protected Object get (int index)
            {
               return getFromPoints (index);
            }


            protected int size()
            {
               return sizeOfPoints();
            }


            public void add (Object o)
            {
               throw new UnsupportedOperationException();
            }


            public void remove()
            {
               throw new UnsupportedOperationException();
            }


            public void set (Object o)
            {
               if (o == null)
               {
                  throw new IllegalArgumentException ("Unable to set null-value");
               }
               else if (currentIndex() >= 0 && currentIndex() < size())
               {
                  setPointAt (currentIndex(), (Point) o);
               }
               else
               {
                  throw new NoSuchElementException();
               }
            }
         };
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public final ListIterator iteratorOfPoints()
   {
      return iteratorOfPoints (0);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pt  No description provided
    * @return    No description provided
    */
   public final ListIterator iteratorOfPoints (Point pt)
   {
      return iteratorOfPoints (getIndexFromPoints (pt));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfPoints()
   {
      return 2;
   }


   /**
    * Get the nextFromPoints attribute of the JLine object
    *
    * @param point  No description provided
    * @return       The nextFromPoints value
    */
   public final Point getNextFromPoints (Point point)
   {
      return  (point == null ? null : getNextFromPoints (point.x, point.y));
   }


   /**
    * Get the nextFromPoints attribute of the JLine object
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   The nextFromPoints value
    */
   public Point getNextFromPoints (int x, int y)
   {
      Point result = null;

      int index = getIndexFromPoints (x, y);
      if (index != -1 && index < sizeOfPoints() - 1)
      {
         result = getFromPoints (index + 1);
      }

      return result;
   }


   /**
    * Get the prevFromPoints attribute of the JLine object
    *
    * @param point  No description provided
    * @return       The prevFromPoints value
    */
   public final Point getPrevFromPoints (Point point)
   {
      return  (point == null ? null : getPrevFromPoints (point.x, point.y));
   }


   /**
    * Get the prevFromPoints attribute of the JLine object
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   The prevFromPoints value
    */
   public Point getPrevFromPoints (int x, int y)
   {
      Point result = null;

      int index = getIndexFromPoints (x, y);
      if (index > 0)
      {
         result = getFromPoints (index - 1);
      }

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param point  No description provided
    * @return       No description provided
    */
   public final boolean hasInPoints (Point point)
   {
      if (point == null)
      {
         return false;
      }

      return hasInPoints (point.x, point.y);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   No description provided
    */
   public boolean hasInPoints (int x, int y)
   {
      return  (getIndexFromPoints (x, y) != -1);
   }


   /**
    * @param point  No description provided
    * @return       the index of point or -1 if not in this line
    */
   public final int getIndexFromPoints (Point point)
   {
      if (point == null)
      {
         return -1;
      }

      return getIndexFromPoints (point.x, point.y);
   }


   /**
    * Get the indexFromPoints attribute of the JLine object
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   The indexFromPoints value
    */
   public int getIndexFromPoints (int x, int y)
   {
      int index = -1;
      Iterator pointIter = iteratorOfPoints();
      while (pointIter.hasNext())
      {
         index++;
         Point p = (Point) pointIter.next();
         if (p != null && p.x == x && p.y == y)
         {
            return index;
         }
      }
      return -1;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean dashedFlag;


   /**
    * Sets the lineDashed attribute of the JLine object
    *
    * @param dashed  The new lineDashed value
    */
   public void setLineDashed (boolean dashed)
   {
      dashedFlag = dashed;

      if (dashed)
      {
         BasicStroke defaultStroke = LineStyle.DASHED;
         BasicStroke stroke = new BasicStroke (thickness, defaultStroke.getEndCap(), defaultStroke.getLineJoin(), defaultStroke.getMiterLimit(), defaultStroke.getDashArray(), defaultStroke.getDashPhase());
         setStroke (stroke);
      }
      else
      {
         setStroke (new BasicStroke (thickness));
      }
   }


   /**
    * Get the lineDashed attribute of the JLine object
    *
    * @return   The lineDashed value
    */
   public boolean isLineDashed()
   {
      return dashedFlag;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int thickness = 1;


   /**
    * Sets the thickness attribute of the JLine object
    *
    * @param newThickness  The new thickness value
    */
   public void setThickness (int newThickness)
   {
      thickness = newThickness;

      if (thickness < 0)
      {
         thickness = 0;
      }

      BasicStroke stroke;
      if (getStroke() != null)
      {
         BasicStroke defaultStroke = (BasicStroke) getStroke();
         stroke = new BasicStroke (thickness, defaultStroke.getEndCap(), defaultStroke.getLineJoin(), defaultStroke.getMiterLimit(), defaultStroke.getDashArray(), defaultStroke.getDashPhase());
      }
      else
      {
         stroke = new BasicStroke (thickness);
      }
      setStroke (stroke);
   }


   /**
    * Get the thickness attribute of the JLine object
    *
    * @return   The thickness value
    */
   public int getThickness()
   {
      return thickness;
   }


   /**
    * UMLAttribute : 'delta : double '
    */
   private double delta = 3.0;


   /**
    * Get the value of delta. <p>
    *
    * This value declares the "thickness" of the line.<br>
    * Mouse actions that are at most in a distance of delta are considered to be on this line.
    *
    * @return   Value of delta.
    * @see      #setDelta(double)
    * @see      #getDistance(Point)
    * @see      #getDistance(int,int)
    * @see      #contains(Point)
    * @see      #contains(int,int)
    */
   public double getDelta()
   {
      return this.delta;
   }


   /**
    * Set the value of delta. <p>
    *
    * This value declares the "thickness" of the line.<br>
    * Mouse actions, that are at most in a distance of delta are considered to be on this line.
    *
    * @param delta  Value to assign to delta.
    * @see          #getDelta()
    * @see          #getDistance(Point)
    * @see          #getDistance(int,int)
    * @see          #contains(Point)
    * @see          #contains(int,int)
    */
   public void setDelta (double delta)
   {
      delta = Math.abs (delta);
      if (this.delta != delta)
      {
         this.delta = delta;
         linePointsChanged();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private double cx = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private double cy = 0;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private double length = 0; // length of the line

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean coeffsDirty = true; // if true, the coeffs (cx, cy, length) have to be recalculated


   /**
    * compute the coefficients needed to efficiently calculate the distance of points to the
    * line. These values are buffered, because they are used very often.
    *
    * @see   #invalidateCoeffs
    * @see   #getCx
    * @see   #getCy
    * @see   #getLength
    * @see   #contains(int, int)
    * @see   #contains(java.awt.Point)
    * @see   #getDistance(int, int)
    * @see   #getDistance(java.awt.Point)
    */
   void computeCoeffs()
   {
      Point start = getStartPoint();
      Point end = getEndPoint();

      int dx = end.x - start.x;
      int dy = end.y - start.y;
      int div = dx * dx + dy * dy;

      cx =  ((double) dx) /  ((double) div);
      cy =  ((double) dy) /  ((double) div);
      length = Math.sqrt (div);

      coeffsDirty = false;
   }


   /**
    * invalidates the coefficients needed for calculations on the line. So they are recalculated
    * the next time they are used. Called whenever one of the endpoints of the line changes
    *
    * @see   #computeCoeffs
    */
   void invalidateCoeffs()
   {
      coeffsDirty = true;
   }


   /**
    * Get the coeffsDirty attribute of the JLine object
    *
    * @return   The coeffsDirty value
    */
   boolean isCoeffsDirty()
   {
      return coeffsDirty;
   }


   /**
    * x coeff to quickly calculate the distance of a point to a line <p>
    *
    * The value is cached until one of the endpoints changes, and then recalculated along with
    * cy and length, when one of them is requested.<p>
    *
    * <pre>cx=dx/(dx**2+dy**2), where (dx, dy)=endPoint-startPoint</pre><p>
    *
    * @return   The cx value
    * @see      #getCy
    * @see      #getLength
    * @see      #computeCoeffs
    * @see      #invalidateCoeffs
    */
   protected double getCx()
   {
      if (coeffsDirty)
      {
         computeCoeffs();
      }

      return cx;
   }


   /**
    * y coeff to quickly calculate the distance of a point to a line <p>
    *
    * The value is cached until one of the endpoints changes, and then recalculated along with
    * cx and length, when one of them is requested.<p>
    *
    * <pre>cy=dy/(dx**2+dy**2), where (dx, dy)=endPoint-startPoint</pre><p>
    *
    * @return   The cy value
    * @see      #getCy
    * @see      #getLength
    * @see      #computeCoeffs
    * @see      #invalidateCoeffs
    */
   protected double getCy()
   {
      if (coeffsDirty)
      {
         computeCoeffs();
      }

      return cy;
   }


   /**
    * length of the line<p>
    *
    * The value is cached until one of the endpoints changes, and then recalculated along with
    * cx and cy, when one of them is requested.<p>
    *
    * @return   The startToEndDistance value
    * @see      #getCx
    * @see      #getCy
    * @see      #computeCoeffs
    * @see      #invalidateCoeffs
    */
   public double getStartToEndDistance()
   {
      if (coeffsDirty)
      {
         computeCoeffs();
      }

      return length;
   }


   /**
    * Get the length attribute of the JLine object
    *
    * @return   The length value
    */
   public double getLength()
   {
      return getStartToEndDistance();
   }


   /**
    * Get the direction attribute of the JLine object
    *
    * @param d  No description provided
    * @return   The direction value
    */
   public final Point2D getDirection (double d)
   {
      return getDirection (d, null);
   }


   /**
    * Get the direction attribute of the JLine object
    *
    * @param d  No description provided
    * @param p  No description provided
    * @return   The direction value
    */
   public Point2D getDirection (double d, Point2D p)
   {
      return getDirection (p);
   }


   /**
    * Get the direction attribute of the JLine object
    *
    * @return   The direction value
    */
   public Point2D getDirection()
   {
      return getDirection (null);
   }


   /**
    * Get the direction attribute of the JLine object
    *
    * @param p  No description provided
    * @return   The direction value
    */
   public Point2D getDirection (Point2D p)
   {
      if (p == null)
      {
         p = new Point2D.Double();
      }

      Point start = getStartPoint();
      Point end = getEndPoint();

      if (coeffsDirty)
      {
         computeCoeffs();
      }
      double len = getStartToEndDistance();

      if (len != 0)
      {
         p.setLocation ( (end.x - start.x) / len,
             (end.y - start.y) / len);
      }
      else
      {
         p.setLocation (1, 0);
      }
      return p;
   }


   /**
    * Get the normal attribute of the JLine object
    *
    * @param d  No description provided
    * @return   The normal value
    */
   public final Point2D getNormal (double d)
   {
      return getNormal (d, null);
   }


   /**
    * Get the normal attribute of the JLine object
    *
    * @param d  No description provided
    * @param p  No description provided
    * @return   The normal value
    */
   public Point2D getNormal (double d, Point2D p)
   {
      return getNormal (p);
   }


   /**
    * Get the normal attribute of the JLine object
    *
    * @return   The normal value
    */
   public Point2D getNormal()
   {
      return getNormal (null);
   }


   /**
    * Get the normal attribute of the JLine object
    *
    * @param p  No description provided
    * @return   The normal value
    */
   public Point2D getNormal (Point2D p)
   {
      if (p == null)
      {
         p = new Point2D.Double();
      }

      p = getDirection (p);
      double x = p.getX();
      double y = p.getY();
      p.setLocation (-y, x);

      return p;
   }


   /**
    * Get the angle attribute of the JLine object
    *
    * @param d  No description provided
    * @return   The angle value
    */
   public double getAngle (double d)
   {
      return getAngle();
   }


   /**
    * Get the angle attribute of the JLine object
    *
    * @return   The angle value
    */
   public double getAngle()
   {
      double result;
      Point start = getStartPoint();
      Point end = getEndPoint();
      int dx = end.x - start.x;
      int dy = end.y - start.y;

      if (dx == 0)
      {
         if (dy >= 0)
         {
            result = Math.PI / 2;
         }
         else
         {
            result = Math.PI * 3 / 2;
         }
      }
      else
      {
         result = Math.atan ((double) dy / dx);

         if (dx < 0)
         {
            result += Math.PI;
         }
         else if (result < 0)
         {
            result += 2 * Math.PI;
         }
      }
      return result;
   }


   /**
    * Get the position attribute of the JLine object
    *
    * @param p  No description provided
    * @return   The position value
    */
   public final double getPosition (Point p)
   {
      return getPosition (p.x, p.y);
   }


   /**
    * Get the position attribute of the JLine object
    *
    * @param x  No description provided
    * @param y  No description provided
    * @return   The position value
    */
   public double getPosition (int x, int y)
   {
      Point startPoint = getStartPoint();
      Point endPoint = getEndPoint();

      double cx = getCx();
      double cy = getCy();

      // distance p->startPoint
      double dx = x - startPoint.x;
      double dy = y - startPoint.y;

      double s;

      if (startPoint.x != endPoint.x)
      {
         s = dx * cx + dy * cy;
      }
      else if (startPoint.y != endPoint.y)
      {
         s = dx * cx - dy * cy;
      }
      else
      {
         s = 0;
      }

      return s;
   }


   /**
    * Get the pointAt attribute of the JLine object
    *
    * @param d  No description provided
    * @return   The pointAt value
    */
   public Point2D getPointAt (double d)
   {
      Point start = getStartPoint();
      Point end = getEndPoint();

      return new Point2D.Double (start.x + d *  (end.x - start.x), start.y + d *  (end.y - start.y));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean boundsDirty = false;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void linePointsChanged()
   {
      boundsDirty = true;
      revalidate();
   }


   /**
    * set the bounds to the value of <pre>getPreferredBounds</pre> by calling <pre>super.setBounds</pre>
    *
    * @return   No description provided
    * @see      #getPreferredBounds
    * @see      #setBounds(int, int, int, int)
    */
   boolean adjustBounds()
   {
      Rectangle bounds = getPreferredBounds();
      Rectangle oldBounds = getBounds();
      if (!oldBounds.equals (bounds))
      {
         setBounds (bounds.x, bounds.y, bounds.width, bounds.height);
         revalidate();
         if (boundsDirty)
         {
            repaint();
         }
      }
      boolean oldDirty = boundsDirty;
      boundsDirty = false;

      return  (oldDirty != boundsDirty);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void doLayout()
   {
      if (boundsDirty)
      {
         boolean done = adjustBounds();
         if (done)
         {
            super.doLayout();
         }
      }
      else
      {
         super.doLayout();
      }
   }


   /**
    * compute the bounds from the start- and endpoint and delta. Delegated to the UI of this
    * object.
    *
    * @return   The preferredBounds value
    * @see      #getDelta
    * @see      #getStartPoint()
    * @see      #getEndPoint()
    * @see      #adjustBounds
    */

   public Rectangle getPreferredBounds()
   {
      return  ((LineUI) ui).getPreferredBounds (this);
   }


   /**
    * set Bounds to parameters and update startPoint and endPoint accordingly.<br>
    * The setting of the bounds is done in adjustBounds then.
    *
    * @param x  x-position
    * @param y  y-position
    * @param w  width
    * @param h  height
    * @see      #adjustBounds
    */
   public void setBounds (int x, int y, int w, int h)
   {
      Rectangle bounds = getBounds();

      if (bounds.x != x || bounds.y != y || bounds.width != w || bounds.height != h)
      {
         invalidateCoeffs();
         super.setBounds (x, y, w, h);
      }
   }


   /**
    * @return   The preferredSize value
    * @see      #getPreferredBounds
    */
   public Dimension getPreferredSize()
   {
      return getPreferredBounds().getSize();
   }


   /**
    * @param x  x-coordinate
    * @param y  y-coordinate
    * @return   true if getDistance(x,y) < getDelta(), false otherwise
    * @see      #contains(java.awt.Point)
    * @see      #getDistance(java.awt.Point)
    * @see      #getDistance(int, int)
    */
   public boolean contains (int x, int y)
   {
      return super.contains (x, y) && getDistance (x + getX(), y + getY()) < getDelta();
   }


   /**
    * @param p  Point
    * @return   true if getDistance(p) < getDelta(), false otherwise
    * @see      #contains(int, int)
    * @see      #getDistance(java.awt.Point)
    * @see      #getDistance(int, int)
    */
   public final boolean contains (Point p)
   {
      return contains (p.x, p.y);
   }


   /**
    * @param p  Point
    * @return   distance of p to line
    * @see      #contains(java.awt.Point)
    * @see      #contains(int, int)
    * @see      #getDistance(int, int)
    */
   public final double getDistance (Point p)
   {
      return getDistance (p.x, p.y);
   }


   /**
    * @param x  x-coordinate
    * @param y  y-coordinate
    * @return   distance of (x, y) to line
    * @see      #contains(java.awt.Point)
    * @see      #contains(int, int)
    * @see      #getDistance(int, int)
    */
   public double getDistance (int x, int y)
   {
      Point startPoint = getStartPoint();
      Point endPoint = getEndPoint();

      double cx = getCx();
      double cy = getCy();

      // distance p->startPoint
      double dx = x - startPoint.x;
      double dy = y - startPoint.y;

      double s;

      double
         t;

      if (startPoint.x != endPoint.x)
      {
         s = dx * cx + dy * cy;

         if (s < 0)
         { // p is before startPoint

            return startPoint.distance (x, y);
         }
         else if (s > 1)
         { // p is after endPoint

            return endPoint.distance (x, y);
         }
         else
         { // p is between startPoint and endPoint

            t = dy * cx - dx * cy;

            return Math.abs (t * getStartToEndDistance());
         }
      }
      else if (startPoint.y != endPoint.y)
      {
         s = dy * cy - dx * cx;

         if (s < 0)
         { // p is before startPoint

            return startPoint.distance (x, y);
         }
         else if (s > 1)
         { // p is after endPoint

            return endPoint.distance (x, y); // p is between startPoint and endPoint
         }
         else
         {
            t = dx * cy - dy * cx;

            return Math.abs (t * getStartToEndDistance());
         }
      }

      return startPoint.distance (x, y); // startPoint==endPoint!

   }


   /**
    * Sets the UI of this class.
    *
    * @param ui  the UI for this object. Must be a subclass of LineUI.
    */
   public void setUI (ComponentUI ui)
   {
      super.setUI (ui);
   }


   /**
    * Get the uI attribute of the JLine object
    *
    * @return   The uI value
    */
   public LineUI getUI()
   {
      return (LineUI) this.ui;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Stroke stroke;


   /**
    * Sets the stroke attribute of the JLine object
    *
    * @param value  The new stroke value
    */
   public void setStroke (Stroke value)
   {
      if (this.stroke != value)
      {
         this.stroke = value;
         repaint();
      }
   }


   /**
    * Get the stroke attribute of the JLine object
    *
    * @return   The stroke value
    */
   public Stroke getStroke()
   {
      return this.stroke;
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      String result = getClass().getName() + "@" + Integer.toHexString (hashCode()) + "\n";
      result += "\tBounds: " + getBounds() + "\n";
      result += "\tPoints: [" + getStartPoint().x + ":" + getStartPoint().y + "],[" + getEndPoint().x + ":" + getEndPoint().y + "]";

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      Container parent = getParent();

      if (parent != null)
      {
         parent.remove (this);
      }

      invalidateCoeffs();
   }
}

/*
 * $Log: JLine.java,v $
 * Revision 1.22.2.1  2005/04/25 16:18:26  lowende
 * Removed warnings.
 *
 */
