/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing;

import java.awt.*;
import java.util.Iterator;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;

import de.uni_paderborn.fujaba.fsa.swing.border.OvalBorder;


/**
 * THE choice for oval targets. Cannot layout anything else
 *
 * @author    $Author: l3_g5 $
 * @version   $Revision: 1.14.2.1 $
 */
public class OvalGrabLayouter extends GrabLayouter
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static OvalGrabLayouter layouter = null;


   /**
    * Constructor for class OvalGrabLayouter
    */
   private OvalGrabLayouter()
   {
      super();
   }



   /**
    * @return   the singleton instance.
    */
   public static GrabLayouter get()
   {
      if (layouter == null)
      {
         layouter = new OvalGrabLayouter();
      }
      return layouter;
   }


   /**
    * Searches for the first OvalBorder from outside by recursing into CompundBorders.
    *
    * @param border  the border to start search at.
    * @return        the outmost OvalBorder or null if no OvalBorder was found
    */
   private OvalBorder searchForOvalBorder (Border border)
   {
      if (border instanceof OvalBorder)
      {
         return (OvalBorder) border;
      }
      else if (border instanceof CompoundBorder)
      {
         OvalBorder outer = searchForOvalBorder ( ((CompoundBorder) border).getOutsideBorder());

         if (outer != null)
         {
            return outer;
         }

         OvalBorder inner = searchForOvalBorder ( ((CompoundBorder) border).getInsideBorder());

         if (inner != null)
         {
            return inner;
         }
      }

      return null;
   }


   /**
    * Layouts all the grabs. The manager's target is checked for supported objects. These objects
    * must have oval shape and currently are JOval-objects or objects with an outermost border
    * of type OvalBorder.
    *
    * @param manager  No description provided
    * @see            de.uni_paderborn.fujaba.fsa.swing.GrabManager#getTarget
    * @see            #layoutGrab
    */
   public void layout (GrabManager manager)
   {
      // check manager's target
      OvalBorder ovalBorder = null;

      if (! (manager.getTarget() instanceof JOval))
      {
         ovalBorder = searchForOvalBorder (manager.getTarget().getBorder());
      }

      // layout all the JGrabs
      Iterator grabIter = manager.iteratorOfGrabs();
      while (grabIter.hasNext())
      {
         JGrab grab = (JGrab) grabIter.next();
         layoutGrab (manager, grab, ovalBorder);
      }
   }



   /**
    * Layout the Grab around the oval target.
    *
    * @param manager  No description provided
    * @param grab     No description provided
    * @param border   No description provided
    * @see            #getPreferredLayoutInformation
    */
   protected void layoutGrab (GrabManager manager, JGrab grab, OvalBorder border)
   {
      Rectangle bounds = manager.getTarget().getBounds();
      double[] info = getPreferredLayoutInformation (grab);
      Direction orientation = Direction.fromInt ((int) info[0]);
      double align = info[1];

      align =  (1 - 2 * align) * Math.PI / 4;

      Insets insets = OvalLayout.getInsetsTilBorder (manager.getTarget(), border);
      Rectangle grabBounds = grab.getBounds();
      double grabWidth = grabBounds.getWidth();
      double grabHeight = grabBounds.getHeight();
      insets.left -= grabWidth / 2;
      insets.top -= grabHeight / 2;
      insets.right -= grabWidth / 2;
      insets.bottom -= grabHeight / 2;
      double xradius =  (bounds.getWidth() - insets.left - insets.right) / 2.0;
      double yradius =  (bounds.getHeight() - insets.top - insets.bottom) / 2.0;
      double centerX =  (bounds.getX() + insets.left) + xradius;
      double centerY =  (bounds.getY() + insets.top) + yradius;

      Point grabPoint = new Point();
      if (orientation == Direction.TOP)
      {
         grabPoint.x = (int)  (centerX - Math.sin (align) * xradius);
         grabPoint.y = (int)  (centerY - Math.cos (align) * yradius);
      }
      else if (orientation == Direction.BOTTOM)
      {
         grabPoint.x = (int)  (centerX - Math.sin (align) * xradius);
         grabPoint.y = (int)  (centerY + Math.cos (align) * yradius);
      }
      else if (orientation == Direction.LEFT)
      {
         grabPoint.x = (int)  (centerX - Math.cos (align) * xradius);
         grabPoint.y = (int)  (centerY - Math.sin (align) * yradius);
      }
      else
      {
         grabPoint.x = (int)  (centerX + Math.cos (align) * xradius);
         grabPoint.y = (int)  (centerY - Math.sin (align) * yradius);
      }

      grabPoint.x -= grabWidth / 2;
      grabPoint.y -= grabHeight / 2;

      Container targetParent = manager.getTarget().getParent();
      Container grabParent = grab.getParent();
      if (targetParent != null && grabParent != null)
      {
         grabPoint
            = SwingUtilities.convertPoint (targetParent, grabPoint, grabParent);
      }
      grab.setLocation (grabPoint);
   }



   /**
    * the preferred orientation and alignment are determined by the angle of the first line
    * of the grab to the center of the circle
    *
    * @param grab  No description provided
    * @return      The preferredLayoutInformation value
    */
   protected double[] getPreferredLayoutInformation (JGrab grab)
   {
      Direction orientation = grab.getOrientation();
      double align = grab.getAlignment();
      if (grab.sizeOfLines() == 0 || grab.getTarget() == null)
      {
         if (grab.isAutoOrientation())
         {
            orientation = Direction.LEFT;
            grab.setLayoutOrientation (orientation);
         }
         if (grab.isAutoAlignment())
         {
            align = 0.5;
            grab.setLayoutAlignment (align);
         }
         return new double[]
            {
            orientation.asInt(), align
            };
      }

      Point mid = grab.getTarget().getLocation();
      mid.x += grab.getTarget().getWidth() / 2 - 1;
      mid.y += grab.getTarget().getHeight() / 2 - 1;

      Iterator lineIter = grab.iteratorOfLines();
      JBendLine line = (JBendLine) lineIter.next();
      JBend bend;
      if (line instanceof JPolyLine)
      {
         JPolyLine poly = (JPolyLine) line;
         bend = poly.getPrevFromBends (grab);
         if (bend == null)
         {
            bend = poly.getNextFromBends (grab);
         }
      }
      else
      {
         bend = line.getStartBend();

         if (bend == grab)
         {
            bend = line.getEndBend();
         }
      }

      Point p = bend.getPoint();
      if (bend.getParent() != null && grab.getTarget().getParent() != null)
      {
         p = SwingUtilities.convertPoint (bend.getParent(), p, grab.getTarget().getParent());
      }

      Direction tmpOrient = Direction.LEFT;
      double tmpAlign = 0.5;

      if (!p.equals (mid))
      {
         int angle = (int)  (Math.acos ( (p.x - mid.x) / Math.sqrt ( (p.x - mid.x) *  (p.x - mid.x) +
             (p.y - mid.y) *  (p.y - mid.y))) * 180 / Math.PI);

         if (p.y > mid.y)
         {
            angle = 360 - angle;
         }

         if (angle <= 45)
         {
            tmpOrient = Direction.RIGHT;
            tmpAlign = 1 -  (angle + 45.0) / 90;
         }
         else if (angle > 315)
         {
            tmpOrient = Direction.RIGHT;
            tmpAlign = 1 -  ((double)  (angle - 360 + 45)) / 90;
         }
         else if (45 < angle && angle <= 135)
         {
            tmpOrient = Direction.TOP;
            tmpAlign = 1 -  ((double)  (angle - 45)) / 90;
         }
         else if (135 < angle && angle <= 225)
         {
            tmpOrient = Direction.LEFT;
            tmpAlign =  ((double)  (angle - 135)) / 90;
         }
         else
         {
            tmpOrient = Direction.BOTTOM;
            tmpAlign =  ((double)  (angle - 225)) / 90;
         }
      }

      if (grab.isAutoOrientation())
      {
         grab.setLayoutOrientation (tmpOrient);
         orientation = tmpOrient;
      }

      if (grab.isAutoAlignment())
      {
         grab.setLayoutAlignment (tmpAlign);
         align = tmpAlign;
      }
      return new double[]
         {
         orientation.asInt(), align
         };
   }
}

/*
 * $Log: OvalGrabLayouter.java,v $
 * Revision 1.14.2.1  2006/02/16 15:17:24  l3_g5
 * bugfix
 *
 */
