/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing.border;

import java.awt.*;

import de.uni_paderborn.fujaba.fsa.swing.LineStyle;


/**
 * A class which implements a line border of arbitrary thickness and of a single color.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.13.2.1 $
 */
public class LineBorder extends AbstractColorBorder
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int thickness;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected boolean roundedCorners;


   /**
    * Creates a line border with the specified color and a thickness = 1.
    *
    * @param color  the color for the border
    */
   public LineBorder (Color color)
   {
      this (color, 1, false);
   }


   /**
    * Creates a line border with the specified color and thickness.
    *
    * @param color      the color of the border
    * @param thickness  the thickness of the border
    */
   public LineBorder (Color color, int thickness)
   {
      this (color, thickness, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Stroke stroke = LineStyle.SOLID;


   /**
    * Get the stroke attribute of the LineBorder object
    *
    * @return   The stroke value
    */
   public Stroke getStroke()
   {
      return stroke;
   }


   /**
    * Sets the stroke attribute of the LineBorder object
    *
    * @param stroke  The new stroke value
    */
   public void setStroke (Stroke stroke)
   {
      this.stroke = stroke;
   }


   /**
    * Creates a line border with the specified color, thickness, and corner shape.
    *
    * @param color           the color of the border
    * @param thickness       the thickness of the border
    * @param roundedCorners  whether or not border corners should be round
    */
   LineBorder (Color color, int thickness, boolean roundedCorners)
   {
      super (color);
      this.thickness = thickness;
      this.roundedCorners = roundedCorners;
   }


   /**
    * Paints the border for the specified component with the specified position and size.
    *
    * @param c       the component for which this border is being painted
    * @param g       the paint graphics
    * @param x       the x position of the painted border
    * @param y       the y position of the painted border
    * @param width   the width of the painted border
    * @param height  the height of the painted border
    */
   public void paintBorder (Component c, Graphics g, int x, int y, int width, int height)
   {
      Color oldColor = g.getColor();

      Graphics2D g2d = (Graphics2D) g;
      Stroke oldStroke = g2d.getStroke();
      g2d.setStroke (getStroke());

      g.setColor (getBorderColor());

      if (!roundedCorners)
      {
         g.drawRect (x +  (thickness / 2), y +  (thickness / 2), width - thickness, height - thickness);
      }
      else
      {
         g.drawRoundRect (x +  (thickness / 2), y +  (thickness / 2), width - thickness, height - thickness, thickness, thickness);
      }

      g.setColor (oldColor);
      g2d.setStroke (oldStroke);
   }


   /**
    * Returns the insets of the border.
    *
    * @param c  the component for which this border insets value applies
    * @return   The borderInsets value
    */
   public Insets getBorderInsets (Component c)
   {
      return new Insets (thickness, thickness, thickness, thickness);
   }


   /**
    * Reinitialize the insets parameter with this Border's current Insets.
    *
    * @param c       the component for which this border insets value applies
    * @param insets  the object to be reinitialized
    * @return        The borderInsets value
    */
   public Insets getBorderInsets (Component c, Insets insets)
   {
      insets.left = insets.top = insets.right = insets.bottom = thickness;
      return insets;
   }


   /**
    * Returns the thickness of the border.
    *
    * @return   The thickness value
    */
   public int getThickness()
   {
      return thickness;
   }


   /**
    * Sets the thickness of the border.
    *
    * @param newThickness  The new thickness value
    */
   public void setThickness (int newThickness)
   {
      thickness = newThickness;

      if (thickness < 0)
      {
         thickness = 0;
      }

      BasicStroke stroke;
      if (getStroke() != null)
      {
         BasicStroke defaultStroke = (BasicStroke) getStroke();
         stroke = new BasicStroke (thickness, defaultStroke.getEndCap(), defaultStroke.getLineJoin(), defaultStroke.getMiterLimit(), defaultStroke.getDashArray(), defaultStroke.getDashPhase());
      }
      else
      {
         stroke = new BasicStroke (thickness);
      }
      setStroke (stroke);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean dashedFlag;


   /**
    * Sets the borderDashed attribute of the LineBorder object
    *
    * @param dashed  The new borderDashed value
    */
   public void setBorderDashed (boolean dashed)
   {
      dashedFlag = dashed;

      if (dashed)
      {
         BasicStroke defaultStroke = LineStyle.DASHED;
         BasicStroke stroke = new BasicStroke (thickness, defaultStroke.getEndCap(), defaultStroke.getLineJoin(), defaultStroke.getMiterLimit(), defaultStroke.getDashArray(), defaultStroke.getDashPhase());
         setStroke (stroke);
      }
      else
      {
         setStroke (new BasicStroke (thickness));
      }
   }


   /**
    * Get the borderDashed attribute of the LineBorder object
    *
    * @return   The borderDashed value
    */
   public boolean isBorderDashed()
   {
      return dashedFlag;
   }


   /**
    * Returns whether or not the border is opaque.
    *
    * @return   The borderOpaque value
    */
   public boolean isBorderOpaque()
   {
      return false;
   }

}

/*
 * $Log: LineBorder.java,v $
 * Revision 1.13.2.1  2005/08/23 08:24:57  lowende
 * Removed compile warnings.
 *
 */
