/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.event.*;
import java.util.Iterator;
import java.util.StringTokenizer;

import javax.swing.*;
import javax.swing.text.JTextComponent;

import de.uni_paderborn.fujaba.basic.Utility;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.sdm.Path;


/**
 * Provide completion support for textual expressions in Fujaba.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.10.2.2 $ $Date: 2005/09/30 18:56:51 $
 */
public class CompletionKeyListener implements KeyListener
{
   /**
    * Constructor for class CompletionKeyListener
    *
    * @param target  No description provided
    */
   public CompletionKeyListener (Object target)
   {
      setContext (target, false);
   }


   /**
    * @param target
    * @param declarationWanted  The new context value
    */
   public void setContext (Object target, boolean declarationWanted)
   {
      prefix = null;
      this.declarationWanted = declarationWanted;
      this.addBasicTypes = false;

      if (target != null && target instanceof UMLClass)
      {
         contextClass = (UMLClass) target;
      }
      else if (target != null && target instanceof UMLObject)
      {
         contextObject = (UMLObject) target;
         contextClass = contextObject.getInstanceOf();
         contextStoryPattern = (UMLStoryPattern) contextObject.getFirstFromDiagrams();
      }
      else if (target != null && target instanceof UMLActivity)
      {
         contextActivity = (UMLActivity) target;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLClass contextClass = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLObject contextObject = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLStoryPattern contextStoryPattern = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLActivity contextActivity = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String prefix = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   boolean declarationWanted = false;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   boolean addBasicTypes = false;


   /*
    *  (non-Javadoc)
    *  @see java.awt.event.KeyListener#keyPressed(java.awt.event.KeyEvent)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void keyPressed (KeyEvent e)
   {
      // Auto-generated method stub

   }


   /*
    *  (non-Javadoc)
    *  @see java.awt.event.KeyListener#keyReleased(java.awt.event.KeyEvent)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void keyReleased (KeyEvent e)
   {
      try
      {
         updateContextClass();
         if (contextClass == null)
         {
            return; // <=============== sudden death
         }

         UMLClass localContext = contextClass;

         // on control blank try completion
         int control =  (e.getModifiers() & InputEvent.CTRL_MASK);
         if (e.getKeyChar() == ' ' && control > 0)
         {
            Object obj = e.getSource();

            if (obj != null && obj instanceof JTextComponent)
            {
               prefix = null;

               JTextComponent jtf = (JTextComponent) obj;

               // now retrieve text and cursor position
               int caretPos = jtf.getCaretPosition();
               String fullText = jtf.getText();
               String txt = jtf.getText (0, caretPos);

               localContext = computeLocalContext (localContext, txt);

               computeAlternatives (localContext, jtf, caretPos, fullText);

            }

         }
      }
      catch (Exception ee)
      {

      }
   }


   /**
    */
   private void updateContextClass()
   {
      // initialize the context
      if (contextClass == null && contextActivity != null)
      {
         UMLActivityDiagram activityDiag = (UMLActivityDiagram) contextActivity.getFirstFromDiagrams();
         UMLMethod myMethod = activityDiag.getStoryMethod();
         contextClass = myMethod.getParent();
      }
   }


   /**
    * @param localContext
    * @param txt
    * @return              No description provided
    */
   public UMLClass computeLocalContext (UMLClass localContext, String txt)
   {
      // first look for special characters
      int pos = -1;
      String specialChars = ";+*-,{}=";
      for (int i = 0; i < specialChars.length(); i++)
      {
         pos = txt.lastIndexOf (specialChars.charAt (i));
         if (pos >= 0)
         {
            // cut preceding part
            txt = txt.substring (pos + 1);
         }
      }

      pos = txt.lastIndexOf (':');
      if (declarationWanted && pos >= 0)
      {
         // complete just by basic type names
         txt = txt.substring (pos + 1);
         addBasicTypes = true;
      }

      // now try to complete this.
      StringTokenizer tokens = new StringTokenizer (txt, ".");
      String myToken = "";
      UMLClass oldLocalContext = null;
      while (localContext != null && tokens.hasMoreTokens())
      {
         oldLocalContext = localContext;
         myToken = tokens.nextToken();

         if ("this".equals (myToken))
         {
            localContext = oldLocalContext;

         }
         else
         {
            localContext = changeAttrContext (oldLocalContext, myToken);

            if (localContext == null)
            {
               // try methods
               localContext = changeMethodContext (oldLocalContext, myToken);
            }

            if (localContext == null)
            {
               // it could be the name of a local variable
               localContext = changeLocalVarContext (myToken);
            }
         }
      }

      if (localContext == null)
      {
         // user has already typed some chars of the desired completion
         prefix = myToken.trim();
         localContext = oldLocalContext;
      }
      return localContext;
   }


   /*
    *  (non-Javadoc)
    *  @see java.awt.event.KeyListener#keyTyped(java.awt.event.KeyEvent)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void keyTyped (KeyEvent e)
   {
   }


   /**
    * @param myToken
    * @return         No description provided
    */
   private UMLClass changeLocalVarContext (String myToken)
   {
      if (myToken == null || myToken.equals (""))
      {
         return null;
      }

      updateContextActivity();
      if (contextActivity != null)
      {
         // get all local vars and compare them with myToken
         UMLActivityDiagram myActDiag = (UMLActivityDiagram) contextActivity.getFirstFromDiagrams();

         UMLMethod myMethod = myActDiag.getStartActivity().getSpec();
         if (myMethod != null)
         {
            if ("this".equals (myToken))
            {
               return myMethod.getParent();
            }
            Iterator params = myMethod.iteratorOfParam();
            while (params.hasNext())
            {
               UMLParam param = (UMLParam) params.next();
               if (param.getName().equals (myToken) && param.getParamType() instanceof UMLClass)
               {
                  return (UMLClass) param.getParamType();
               }
            }
         }

         Path objPath = new Path (myActDiag, "elements.storyPattern.elements");
         Object element;
         UMLObject anObject;
         String objName;
         boolean found;
         while (objPath.hasNext())
         {
            element = objPath.next();
            if (element != null && element instanceof UMLObject)
            {
               anObject = (UMLObject) element;
               objName = anObject.getObjectName();
               found = myToken.equals (objName);
               if (found)
               {
                  return anObject.getInstanceOf();
               }
            }

         }
      }

      return null;
   }


   /**
    */
   private void updateContextActivity()
   {
      if (contextStoryPattern != null)
      {
         contextActivity = contextStoryPattern.getRevStoryPattern();
      }
   }


   /**
    * @param localContext
    * @param myToken
    * @return              No description provided
    */
   private UMLClass changeMethodContext (UMLClass localContext, String myToken)
   {
      // try to find a method with this name and return the result type
      Object result = null;
      int bracePos = myToken.indexOf ('(');
      if (bracePos >= 0)
      {
         myToken = myToken.substring (0, bracePos).trim();
      }

      UMLMethod myMethod = localContext.getFromMethodsByShortNameIncludingInherited (myToken);
      if (myMethod != null)
      {
         result = myMethod.getResultType();
         if (result != null && result instanceof UMLClass)
         {
            return (UMLClass) result;
         }
      }
      return null;
   }


   /**
    * @param localContext
    * @param myToken
    * @return              No description provided
    */
   private UMLClass changeAttrContext (UMLClass localContext, String myToken)
   {
      Object obj;
      // from the current context class try to follow attributes or methods
      UMLAttr attr = localContext.getFromAllAttrs (myToken);
      if (attr != null)
      {
         // attr found, switch context and go on
         obj = attr.getAttrType();
         if (obj != null && obj instanceof UMLClass)
         {
            localContext = (UMLClass) obj;
         }
         else
         {
            localContext = null;
         }
      }
      else
      {
         // could not go on
         localContext = null;
      }
      return localContext;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: mksoft $
    * @version   $Revision: 1.10.2.2 $ $Date: 2005/09/30 18:56:51 $
    */
   class CompletionAction extends AbstractAction
   {
      /**
       * Constructor for class CompletionAction
       *
       * @param prefix    No description provided
       * @param fullText  No description provided
       * @param caretPos  No description provided
       * @param jtf       No description provided
       */
      public CompletionAction (String prefix, String fullText, int caretPos, JTextComponent jtf)
      {
         this.prefix = prefix;
         this.fullText = fullText;
         this.caretPos = caretPos;
         this.jtf = jtf;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private String prefix = null;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private String fullText = null;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private int caretPos;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private JTextComponent jtf = null;


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void actionPerformed (ActionEvent e)
      {
         String newText = fullText.substring (0, caretPos);

         // need to kill prefix ?
         if (prefix != null && newText.endsWith (prefix))
         {
            newText = newText.substring (0, newText.length() - prefix.length());
         }

         newText = newText + e.getActionCommand() + fullText.substring (caretPos);

         jtf.setText (newText);

      }

   }


   /**
    * @param localContext
    * @param jtf
    * @param caretPos
    * @param fullText
    */
   private void computeAlternatives (UMLClass localContext, JTextComponent jtf, int caretPos, String fullText)
   {
      // last token has not been found, offer alternatives
      JPopupMenu completionMenu = new JPopupMenu ("choose completions:");

      Action completionAction = new CompletionAction (prefix, fullText, caretPos, jtf);

      if (this.addBasicTypes)
      {
         // just add the names of the basic types.
         Iterator iter = UMLProject.get().getTypeList().iteratorOfTypes();
         while (iter.hasNext())
         {
            UMLType baseType = (UMLType) iter.next();
            String name = baseType.getName();
            if (prefix == null || prefix.trim().equals ("") || name.startsWith (prefix))
            {
               addToCompletionMenu (completionMenu, completionAction, name, name);
            }
         }
         completionMenu.show (jtf, 5 * caretPos, jtf.getHeight());
      }
      else if (localContext != null)
      {
         // add all attributes
         Iterator iter = localContext.iteratorOfAllAttrs();
         String text = "";
         while (iter.hasNext())
         {
            UMLAttr attr = (UMLAttr) iter.next();
            // if (log.isInfoEnabled()) log.info ("alternative: " + attr.getName());
            if (prefix == null || prefix.trim().equals ("") || attr.getName().startsWith (prefix))
            {
               if (declarationWanted)
               {
                  text = attr.getName();
               }
               else
               {
                  text = "get" + Utility.upFirstChar (attr.getName()) + "()";
               }
               addToCompletionMenu (completionMenu, completionAction, attr.getName(), text);
            }
         }

         // add important methods
         iter = localContext.iteratorOfAllAccessibleMethods();
         while (iter.hasNext())
         {
            UMLMethod aMethod = (UMLMethod) iter.next();
            if (aMethod.getAccessedAttribute() == null)
            {
               // not an attribute access method. Add it.
               if (prefix == null || prefix.trim().equals ("") || aMethod.getName().startsWith (prefix))
               {
                  String name = aMethod.getFullMethodName();
                  addToCompletionMenu (completionMenu, completionAction, name, name);
               }
            }

         }

         completionMenu.show (jtf, 5 * caretPos, jtf.getHeight());
      }
   }


   /**
    * @param completionMenu
    * @param completionAction
    * @param name
    * @param command           The object added.
    */
   private void addToCompletionMenu (JPopupMenu completionMenu, Action completionAction, String name, String command)
   {
      JMenuItem methodItem = new JMenuItem();
      methodItem.setAction (completionAction);
      methodItem.setText (name);
      methodItem.setActionCommand (command);
      completionMenu.add (methodItem);
   }

}

/*
 * $Log: CompletionKeyListener.java,v $
 * Revision 1.10.2.2  2005/09/30 18:56:51  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
