/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;


/**
 * The panel displays two rectangles with either containing a combobox and a swap button in
 * the middle under the connecting line. The panel can be used to select connecting elements
 * for different types of lines e.g. links, and transitions. The combo boxes have to be filled
 * with objects of type PEListEntry, which encapsulates an Object and provides a toString method
 * for displaying in the combo box.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.21 $
 */
public class PERolePanel extends PEBaseComponent
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JComboBox leftCombo;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JComboBox rightCombo;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton swapButton;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int gapHorz = 10;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int gapVert = 20;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int gapLine = 40;


   /**
    * Constructor for class PERolePanel
    */
   public PERolePanel()
   {
      super();

      // initialize the buttons and combo boxes
      swapButton = new JButton ("Swap");
      swapButton.addActionListener (new SwapButtonActionAdapter (this));
      this.add (swapButton);

      leftCombo = new JComboBox();
      leftCombo.setEditable (false);
      leftCombo.setMaximumRowCount (5);
      this.add (leftCombo);

      rightCombo = new JComboBox();
      rightCombo.setEditable (false);
      rightCombo.setMaximumRowCount (5);
      this.add (rightCombo);

      setInsets (new Insets (1, 1, 10, 1));
   }


   /**
    * Read access method for horizontal gap.
    *
    * @return   The gapHorz value
    */
   public int getGapHorz()
   {
      return gapHorz;
   }


   /**
    * Write access method for horizontal gap.
    *
    * @param newGapHorz  The new gapHorz value
    */
   public void setGapHorz (int newGapHorz)
   {
      if (newGapHorz > 0 && newGapHorz != gapHorz)
      {
         gapHorz = newGapHorz;
      }
   }


   /**
    * Read access method for vertical gap.
    *
    * @return   The gapVert value
    */
   public int getGapVert()
   {
      return gapVert;
   }


   /**
    * Write access method for vertical gap.
    *
    * @param newGapVert  The new gapVert value
    */
   public void setGapVert (int newGapVert)
   {
      if (newGapVert > 0 && newGapVert != gapVert)
      {
         gapVert = newGapVert;
      }
   }


   /**
    * Read access method for line gap.
    *
    * @return   The gapLine value
    */
   public int getGapLine()
   {
      return gapLine;
   }


   /**
    * Write access method for line gap.
    *
    * @param newGapLine  The new gapLine value
    */
   public void setGapLine (int newGapLine)
   {
      if (newGapLine > 0 && newGapLine != gapLine)
      {
         gapLine = newGapLine;
      }
   }


   /**
    * Calculates the preferred size of the component, which includes also an inset.
    *
    * @return   The preferredSize value
    */
   public Dimension getPreferredSize()
   {
      Dimension dim = new Dimension (0, 0);

      // Calculate the size of the boxes
      Dimension leftSize = leftCombo.getPreferredSize();
      Dimension rightSize = rightCombo.getPreferredSize();
      Dimension buttonSize = swapButton.getPreferredSize();

      // take combo boxes into accunt
      dim.width = leftSize.width + rightSize.width + 4 * getGapHorz();
      dim.height = Math.max (leftSize.height, rightSize.height) + 2 * getGapVert();

      // add the buttons width
      dim.width = dim.width + buttonSize.width + 2 * getGapLine();
      dim.height = Math.max (dim.height, 2 * buttonSize.height);

      // Add the inset rectangle
      Insets inset = getInsets();
      dim.width = dim.width + inset.left + inset.right;
      dim.height = dim.height + inset.top + inset.bottom;

      return dim;
   }


   /**
    * Calculates the minimal size of the component, which is the preferred size.
    *
    * @return   The minimumSize value
    */
   public Dimension getMinimumSize()
   {
      return getPreferredSize();
   }


   /**
    * Delegates the call to the left combo box
    *
    * @param item  The object added.
    */
   public void addItemToLeftComboBox (PEListEntry item)
   {
      leftCombo.addItem (item);
   }


   /**
    * Delegates the call to the right combo box.
    *
    * @param item  The object added.
    */
   public void addItemToRightComboBox (PEListEntry item)
   {
      rightCombo.addItem (item);
   }


   /**
    * Returns the selected item in the left combo box.
    *
    * @return   The leftComboSelectedItem value
    */
   public PEListEntry getLeftComboSelectedItem()
   {
      return  ((PEListEntry) leftCombo.getSelectedItem());
   }


   /**
    * Sets the selected item in the left combo box.
    *
    * @param item  The new leftComboSelectedItem value
    */
   public void setLeftComboSelectedItem (PEListEntry item)
   {
      leftCombo.setSelectedItem (item);
   }


   /**
    * Returns the selected item in the right combo box.
    *
    * @return   The rightComboSelectedItem value
    */
   public PEListEntry getRightComboSelectedItem()
   {
      return  ((PEListEntry) rightCombo.getSelectedItem());
   }


   /**
    * Sets the selected item in the right combo box.
    *
    * @param item  The new rightComboSelectedItem value
    */
   public void setRightComboSelectedItem (PEListEntry item)
   {
      rightCombo.setSelectedItem (item);
   }


   /**
    * Swaps the Entries in the left and right combo box if possible. If not, does nothing.
    */
   public void swapEntries()
   {
      Object oldLeftEntry = leftCombo.getSelectedItem();
      Object oldRightEntry = rightCombo.getSelectedItem();

      leftCombo.setSelectedItem (oldRightEntry);
      rightCombo.setSelectedItem (oldLeftEntry);

      if (leftCombo.getSelectedItem() != oldRightEntry || rightCombo.getSelectedItem() != oldLeftEntry)
      {
         leftCombo.setSelectedItem (oldLeftEntry);
         rightCombo.setSelectedItem (oldRightEntry);
      }
      repaint();
   }


   /**
    * Adds a ItemListener to the left and right combo box.
    *
    * @param listener  The object added.
    */
   public void addItemListener (ItemListener listener)
   {
      leftCombo.addItemListener (listener);
      rightCombo.addItemListener (listener);
   }


   /**
    * Removes a ItemListener from the left and right combo box.
    *
    * @param listener  No description provided
    */
   public void removeItemListener (ItemListener listener)
   {
      leftCombo.removeItemListener (listener);
      rightCombo.removeItemListener (listener);
   }


   /**
    * Repaints the whole component.
    *
    * @param g  No description provided
    */
   public void paint (Graphics g)
   {
      super.paint (g);

      Insets inset = getInsets();
      Dimension origSize = getSize();
      int height = getPreferredSize().height - inset.top - inset.bottom;

      // Note, no vertical adaption, horizontally expand the line.
      // Calculate the width only.
      g.setColor (Color.black);

      // draw left rectange two times, because no line features are available
      int leftWidth = leftCombo.getSize().width + 2 * getGapHorz();
      g.drawRect (inset.left, inset.top, leftWidth, height);
      g.drawRect (inset.left + 1, inset.top + 1, leftWidth - 2, height - 2);

      // draw right rectangle two times, because no line features are available
      int rightWidth = rightCombo.getSize().width + 2 * getGapHorz();
      g.drawRect (origSize.width - inset.right - rightWidth - 1, inset.top, rightWidth, height);
      g.drawRect (origSize.width - inset.right - rightWidth, inset.top + 1, rightWidth - 2, height - 2);

      // draw the connected line three times, because no line features are available
      g.drawLine (inset.left + leftWidth, inset.top + height / 2, origSize.width - inset.right - rightWidth - 1, inset.top + height / 2);
      g.drawLine (inset.left + leftWidth, inset.top + 1 + height / 2, origSize.width - inset.right - rightWidth - 1, inset.top + 1 + height / 2);
      g.drawLine (inset.left + leftWidth, inset.top - 1 + height / 2, origSize.width - inset.right - rightWidth - 1, inset.top - 1 + height / 2);

      // draw text "source" and "target"
      g.drawString ("source", inset.left + leftWidth + 5, inset.top + height / 2 - 5);
      g.drawString ("target", origSize.width - inset.right - rightWidth - 5 - g.getFontMetrics().stringWidth ("target"), inset.top + height / 2 - 5);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void doLayout()
   {
      Insets inset = getInsets();
      Dimension origSize = getSize();
      int height = getPreferredSize().height - inset.top - inset.bottom;

      // layout the left combo box
      Dimension dim = leftCombo.getPreferredSize();
      leftCombo.setSize (dim);

      int leftWidth = dim.width + 2 * getGapHorz();
      Point loc = new Point (inset.left +  (leftWidth - dim.width) / 2, inset.top +  (height - dim.height) / 2);
      leftCombo.setLocation (loc);

      // layout the right combo box
      dim = rightCombo.getPreferredSize();
      rightCombo.setSize (dim);

      int rightWidth = dim.width + 2 * getGapHorz();
      loc = new Point (origSize.width - rightWidth - inset.right - 1 +  (rightWidth - dim.width) / 2, inset.top +  (height - dim.height) / 2);
      rightCombo.setLocation (loc);

      // layout the swap button
      dim = swapButton.getPreferredSize();
      swapButton.setSize (dim);

      loc = new Point (inset.left + leftWidth +  (origSize.width - inset.left - inset.right - leftWidth - rightWidth - dim.width) / 2, height + inset.top - dim.height);
      swapButton.setLocation (loc);
   }


   /**
    * Access method for an one to n association.
    */
   protected void addComponents()
   {
   }


   /**
    * Sets the readOnly attribute of the PERolePanel object
    *
    * @param b  The new readOnly value
    */
   public void setReadOnly (boolean b)
   {
   }


   /**
    * Get the horzResizable attribute of the PERolePanel object
    *
    * @return   The horzResizable value
    */
   public boolean isHorzResizable()
   {
      return true;
   }


   /**
    * Get the vertResizable attribute of the PERolePanel object
    *
    * @return   The vertResizable value
    */
   public boolean isVertResizable()
   {
      return true;
   }


   /**
    * Sets the combosEnabled attribute of the PERolePanel object
    *
    * @param value  The new combosEnabled value
    */
   public void setCombosEnabled (boolean value)
   {
      rightCombo.setEnabled (value);
      leftCombo.setEnabled (value);
   }


   /**
    * Get the combosEnabled attribute of the PERolePanel object
    *
    * @return   The combosEnabled value
    */
   public boolean isCombosEnabled()
   {
      if (rightCombo.isEnabled() && leftCombo.isEnabled())
      {
         return true;
      }
      else
      {
         return false;
      }
   }

}


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.21 $
 */
class SwapButtonActionAdapter implements ActionListener
{


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   PERolePanel adaptee;


   /**
    * Constructor for class SwapButtonActionAdapter
    *
    * @param adaptee  No description provided
    */
   SwapButtonActionAdapter (PERolePanel adaptee)
   {
      this.adaptee = adaptee;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      adaptee.swapEntries();
   }
}

/*
 * $Log: PERolePanel.java,v $
 * Revision 1.21  2004/10/22 16:41:38  lowende
 * Deprecated warnings removed. Other compile warnings removed.
 *
 */
