/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.List;

import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import de.uni_kassel.prop.CustomEditor;
import de.uni_kassel.prop.ObjectInspector;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FEmptyIterator;


/**
 * Dialog for adding and removing stereotypes to classes.
 *
 * @author    $Author: trinet $
 * @version   $Revision: 1.25 $
 */
public class StereotypeDialog extends JDialog
{
   // --------------------------------------------------------------------------
   // FIELDS
   // --------------------------------------------------------------------------

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static String dialogTitle = "Edit Stereotypes";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton buttonOk;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton buttonCancel;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JComboBox comboboxStereotype;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   StereotypePanel panelStereotype;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   DefaultListModel stereotypeListModel;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLIncrement increment;

   // --------------------------------------------------------------------------
   // CONSTRUCTORS
   // --------------------------------------------------------------------------

   /**
    * Constructor for class StereotypeDialog
    *
    * @param owner      dialog owner gui
    * @param increment  whose stereotypes are edited
    */
   public StereotypeDialog (Dialog owner, UMLIncrement increment)
   {
      super (owner, dialogTitle + " of " + increment, false);
      init (increment);
   } // StereotypeDialog


   /**
    * Constructor for class StereotypeDialog
    *
    * @param owner      dialog owner gui
    * @param increment  whose stereotypes are edited
    */
   public StereotypeDialog (JFrame owner, UMLIncrement increment)
   {
      super (owner, dialogTitle + " of " + increment, false);
      init (increment);
   } // StereotypeDialog


   // --------------------------------------------------------------------------
   // SETUP
   // --------------------------------------------------------------------------

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param increment  No description provided
    */
   protected void init (UMLIncrement increment)
   {
      this.increment = increment;

      JPanel container = new JPanel();
      container.setLayout (new BorderLayout());
      container.add (setupMainPanel(), BorderLayout.NORTH);
      container.add (setupButtonPanel(), BorderLayout.SOUTH);

      getContentPane().add (container);

      // Fill the dialog components with the class settings.
      readStereotypes();

      // Final stuff for dialog.
      addWindowListener (new DialogClose());
      pack();
      centerDialog (this);
   } // init


   /**
    * Creates the main panel.
    *
    * @return   No description provided
    */
   protected JPanel setupMainPanel()
   {
      JPanel panel = new JPanel();

      GridBagLayout gridBag = new GridBagLayout();
      panel.setLayout (gridBag);

      // Add components to the panel using constraints.
      GridBagConstraints constraints = new GridBagConstraints();

      //=========================================================
      // STEREOTYPE
      //=========================================================

      panelStereotype = new StereotypePanel();

      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.weightx = 1.0;
      constraints.weighty = 1.0;
      constraints.insets = new Insets (8, 0, 0, 0);
      gridBag.setConstraints (panelStereotype, constraints);
      panel.add (panelStereotype);

      return panel;
   } // setupMainPanel


   /**
    * Creates a panel with the buttons 'Ok' and 'Cancel'.
    *
    * @return   No description provided
    */
   protected JPanel setupButtonPanel()
   {
      JPanel buttonPanel = new JPanel();
      buttonPanel.setLayout (new FlowLayout (FlowLayout.RIGHT));
      // *INDENT-OFF*
      buttonOk = new JButton ("Ok");
      buttonOk.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               okPressed();
            }
         });
      buttonPanel.add (buttonOk);

      buttonCancel = new JButton ("Cancel");
      buttonCancel.addActionListener (
         new ActionListener()
         {
            public void actionPerformed (ActionEvent e)
            {
               cancelPressed();
            }
         });
      buttonPanel.add (buttonCancel);
      // *INDENT-ON*

      getRootPane().setDefaultButton (buttonOk);

      return buttonPanel;
   } // createButtonPanel


   /**
    * Read properties from the dialog. <p>
    *
    * This method is called every time the user chooses the Ok-button from the dialog.</p>
    */
   public void okPressed()
   {
      setVisible (false);
      writeStereotypes();
   } // okPressed


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void cancelPressed()
   {
      setVisible (false);
   } // cancelPressed


   // --------------------------------------------------------------------------
   // COMFORT
   // --------------------------------------------------------------------------

   /**
    * Calculate and set the position of the dialog to the center of the display screen.
    *
    * @param diag  No description provided
    */
   public static void centerDialog (JDialog diag)
   {
      Dimension screenSize = diag.getToolkit().getScreenSize();
      Dimension size = diag.getSize();

      int y =  (screenSize.height - size.height) / 2;
      int x =  (screenSize.width - size.width) / 2;

      diag.setLocation (x, y);
   } // centerDialog


   /**
    * Read the settings from the class and setup the dialog components.
    */
   protected void readStereotypes()
   {
      if (increment != null)
      {
         Iterator it = UMLStereotypeManager.get().iteratorOfStereotypes();
         while (it.hasNext())
         {
            comboboxStereotype.addItem ( ((UMLStereotype) it.next()).getText());
         }

         it = increment.iteratorOfStereotypes();
         while (it.hasNext())
         {
            UMLStereotype type = (UMLStereotype) it.next();
            stereotypeListModel.addElement (type.getText());
         }

         panelStereotype.verifyAddButton();
      }
   } // readStereotypes


   /**
    * Read the settings from the dialog components and update the class with the new information.
    */
   protected void writeStereotypes()
   {
      if (increment != null)
      {
         increment.removeAllFromStereotypes();

         Enumeration enumeration = stereotypeListModel.elements();
         while (enumeration.hasMoreElements())
         {
            String type = (String) enumeration.nextElement();
            increment.addToStereotypes (UMLStereotypeManager.get().getFromStereotypes (type));
         }
      }
   } // writeStereotypes


   // --------------------------------------------------------------------------
   // INNER CLASSES
   // --------------------------------------------------------------------------

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: trinet $
    * @version   $Revision: 1.25 $
    */
   class DialogClose extends WindowAdapter
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void windowClosing (WindowEvent e)
      {
         cancelPressed();
      } // windowClosing

   } // DialogClose



   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: trinet $
    * @version   $Revision: 1.25 $
    */
   class StereotypePanel extends JPanel implements ListSelectionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      JList stereotypeList;

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final static String strAdd = "Add";
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final static String strRemove = "Remove";

      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      JButton buttonAdd;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      JButton buttonRemove;

      /**
       * button for editing stereotype information
       */
      JButton buttonEdit;


      /**
       * Constructor for class StereotypePanel
       */
      public StereotypePanel()
      {
         super();

         super.setBorder (new TitledBorder ("Stereotypes"));

         stereotypeListModel = new DefaultListModel();

         // Create the list and put it in a scroll pane
         stereotypeList = new JList (stereotypeListModel);
         stereotypeList.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
         stereotypeList.setSelectedIndex (-1);
         stereotypeList.addListSelectionListener (this);
         JScrollPane listScrollPanel = new JScrollPane (stereotypeList);

         buttonAdd = new JButton (strAdd);
         buttonAdd.setActionCommand (strAdd);
         buttonAdd.addActionListener (new AddButtonListener());

         buttonRemove = new JButton (strRemove);
         buttonRemove.setActionCommand (strRemove);
         buttonRemove.addActionListener (new RemoveButtonListener());

         buttonEdit = new JButton ("Edit");
         buttonEdit.addActionListener (new EditButtonListener());

         comboboxStereotype = new JComboBox();
         comboboxStereotype.setEditable (true);
         comboboxStereotype.addKeyListener (new ComboBoxKeyListener());
         comboboxStereotype.addActionListener (new ComboBoxActionListener());

         GridBagLayout gridBag = new GridBagLayout();
         super.setLayout (gridBag);

         // Add components to the panel using constraints.
         GridBagConstraints constraints = new GridBagConstraints();

         constraints.gridwidth = 3; // GridBagConstraints.REMAINDER;

         constraints.fill = GridBagConstraints.HORIZONTAL;
         constraints.weightx = 3.0;
         constraints.weighty = 1.0;
         constraints.insets = new Insets (8, 0, 0, 0);
         gridBag.setConstraints (comboboxStereotype, constraints);
         super.add (comboboxStereotype);

         constraints.gridwidth = 1;
         constraints.fill = GridBagConstraints.HORIZONTAL;
         constraints.weightx = 0.0;
         constraints.weighty = 1.0;
         constraints.insets = new Insets (8, 0, 0, 0);
         gridBag.setConstraints (buttonAdd, constraints);
         super.add (buttonAdd);

         constraints.gridwidth = 1;
         constraints.fill = GridBagConstraints.HORIZONTAL;
         constraints.weightx = 0.0;
         constraints.weighty = 1.0;
         constraints.insets = new Insets (8, 0, 0, 0);
         gridBag.setConstraints (buttonRemove, constraints);
         super.add (buttonRemove);

         constraints.gridwidth = GridBagConstraints.REMAINDER;
         constraints.fill = GridBagConstraints.HORIZONTAL;
         constraints.weightx = 0.0;
         constraints.weighty = 1.0;
         constraints.insets = new Insets (8, 0, 0, 0);
         gridBag.setConstraints (buttonEdit, constraints);
         super.add (buttonEdit);

         constraints.gridwidth = GridBagConstraints.REMAINDER;
         constraints.fill = GridBagConstraints.HORIZONTAL;
         constraints.weightx = 1.0;
         constraints.weighty = 1.0;
         constraints.insets = new Insets (8, 0, 0, 0);
         gridBag.setConstraints (listScrollPanel, constraints);
         super.add (listScrollPanel);
      }


      /**
       * Get the listModel attribute of the StereotypePanel object
       *
       * @return   The listModel value
       */
      public ListModel getListModel()
      {
         return stereotypeListModel;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void valueChanged (ListSelectionEvent e)
      {
         if (e.getValueIsAdjusting() == false)
         {
            if (stereotypeList.getSelectedIndex() == -1)
            {
               // No selection, disable remove button.
               buttonRemove.setEnabled (false);
            }
            else
            {
               // Selection, update text field.
               buttonRemove.setEnabled (true);
               comboboxStereotype.setSelectedItem (stereotypeList.getSelectedValue());
            }
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      void verifyAddButton()
      {
         String selectedItem = (String) comboboxStereotype.getSelectedItem();

         if (selectedItem != null)
         {
            stereotypeList.setSelectedValue (selectedItem, true);
            if (stereotypeListModel.contains (selectedItem))
            {
               buttonAdd.setEnabled (false);
            }
            else
            {
               buttonAdd.setEnabled (true);
            }
            UMLStereotype stereotype = UMLStereotypeManager.get().hasKeyInStereotypes (selectedItem) ?
               UMLStereotypeManager.get().getFromStereotypes (selectedItem) : null;
            buttonEdit.setEnabled (stereotype != null && ObjectInspector.get().getConfig().getCustomEditor (stereotype.getClass()) != null);
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @author    $Author: trinet $
       * @version   $Revision: 1.25 $ $Date: 2004/11/08 19:16:15 $
       */
      class RemoveButtonListener implements ActionListener
      {
         /**
          * No comment provided by developer, please add a comment to improve documentation.
          *
          * @param e  No description provided
          */
         public void actionPerformed (ActionEvent e)
         {
            int index = stereotypeList.getSelectedIndex();

            if (index != -1)
            {
               stereotypeListModel.remove (index);
            }

            int size = stereotypeListModel.getSize();

            if (size == 0)
            {
               //Nobody's left, disable firing.
               buttonRemove.setEnabled (false);
            }
            else
            {
               //Adjust the selection.
               if (index == stereotypeListModel.getSize())
               { //removed item in last position

                  index--;
               }
               stereotypeList.setSelectedIndex (index); //otherwise select same index

            }

            verifyAddButton();
         }
      } // RemoveButtonListener


      /**
       * used for edit button only.
       *
       * @author    $Author: trinet $
       * @version   $Revision: 1.25 $ $Date: 2004/11/08 19:16:15 $
       */
      class EditButtonListener implements ActionListener
      {
         /**
          * Called when the button is pressed
          *
          * @param e  event
          */
         public void actionPerformed (ActionEvent e)
         {
            final String selectedItem = (String) comboboxStereotype.getSelectedItem();
            if (createStereotype (selectedItem))
            {
               UMLStereotype stereotype = UMLStereotypeManager.get().getFromStereotypes (selectedItem);
               final CustomEditor customEditor = ObjectInspector.get().getConfig().getCustomEditor (stereotype.getClass());
               if (customEditor != null)
               {
                  final UMLStereotype result = (UMLStereotype) customEditor.edit (stereotype);
                  if (result != stereotype)
                  {
                     comboboxStereotype.removeItem (selectedItem);
                     comboboxStereotype.addItem (result.getText());
                     comboboxStereotype.setSelectedItem (result.getText());
                  }
                  comboboxStereotype.repaint();
               }
            }
         }
      } // RemoveButtonListener


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @author    $Author: trinet $
       * @version   $Revision: 1.25 $
       */
      class AddButtonListener implements ActionListener
      {
         /**
          * No comment provided by developer, please add a comment to improve documentation.
          *
          * @param e  No description provided
          */
         public void actionPerformed (ActionEvent e)
         {
            int listindex = stereotypeList.getSelectedIndex();
            int size = stereotypeListModel.getSize();
            int comboindex = comboboxStereotype.getSelectedIndex();

            if (comboboxStereotype.getSelectedItem() != null)
            {
               // If no selection or if item in last position is selected,
               // add the new context to end of list, and select new context.
               if (listindex == -1 ||  (listindex + 1 == size))
               {
                  stereotypeListModel.addElement (comboboxStereotype.getSelectedItem());
                  stereotypeList.setSelectedIndex (size);

               }
               else
               {
                  // Otherwise insert the new context after the current selection,
                  // and select new context.
                  stereotypeListModel.insertElementAt (comboboxStereotype.getSelectedItem(), listindex + 1);
                  stereotypeList.setSelectedIndex (listindex + 1);
               }
               if (comboindex == -1)
               {
                  comboboxStereotype.addItem (comboboxStereotype.getSelectedItem());
               }

               verifyAddButton();
            }
            else
            {
               // Nothing to insert.
               Toolkit.getDefaultToolkit().beep();
            }
         }
      } // AddButtonListener


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @author    $Author: trinet $
       * @version   $Revision: 1.25 $
       */
      class ComboBoxActionListener implements ActionListener
      {
         /**
          * No comment provided by developer, please add a comment to improve documentation.
          *
          * @param e  No description provided
          */
         public void actionPerformed (ActionEvent e)
         {
            if (!createStereotype ((String) comboboxStereotype.getSelectedItem()))
            {
               comboboxStereotype.setSelectedIndex (0);
            }
            verifyAddButton();
         }
      } // ComboBoxListener


      /**
       * Checks the stereotype combobox for keyboard events. If the entered stereotype is not
       * in the list of existing stereotypes, the add button is enabled, otherwise disabaled.
       *
       * @author    $Author: trinet $
       * @version   $Revision: 1.25 $
       */
      class ComboBoxKeyListener extends KeyAdapter
      {
         /**
          * Invoked when a key has been typed. This event occurs when a key press is followed
          * by a key release.
          *
          * @param e  No description provided
          */
         public void keyTyped (KeyEvent e)
         {
            getRootPane().setDefaultButton (panelStereotype.buttonAdd);
            String text =  ((String) comboboxStereotype.getEditor().getItem()).trim();
            if (text != null)
            {
               if (stereotypeListModel.contains (text))
               {
                  buttonAdd.setEnabled (false);
               }
               else
               {
                  buttonAdd.setEnabled (true);
               }
            }
         }
      } // ComboBoxKeyListener

   } // StereotypePanel


   /**
    * all classes that will be shown to the user for creating stereotypes.
    */
   private static List stereotypeClasses;


   /**
    * Add a class that will be shown to the user for creating stereotypes.
    *
    * @param value  class added, must be subclass of UMLStereotype
    * @return       true if added
    */
   public static boolean addToStereotypeClasses (Class value)
   {
      boolean changed = false;
      if (value != null && UMLStereotype.class.isAssignableFrom (value))
      {
         initStereotypeClasses();
         if (!stereotypeClasses.contains (value))
         {
            changed = stereotypeClasses.add (value);
         }
      }
      return changed;
   }


   /**
    * init classes list.
    */
   private static void initStereotypeClasses()
   {
      if (stereotypeClasses == null)
      {
         stereotypeClasses = new LinkedList();
         stereotypeClasses.add (UMLStereotype.class);
      }
   }


   /**
    * @return   iterator through classes that will be shown to the user for creating stereotypes.
    */
   public static Iterator iteratorOfStereotypeClasses()
   {
      if (stereotypeClasses == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return stereotypeClasses.iterator();
      }
   }


   /**
    * remove a class from classes shown to the user for creating stereotypes.
    *
    * @param value  class that will no longer be shown to the user for creating stereotypes.
    * @return       true if removed
    */
   public static boolean removeFromStereotypeClasses (Class value)
   {
      boolean changed = false;
      if ( (stereotypeClasses != null) &&  (value != null))
      {
         changed = stereotypeClasses.remove (value);
      }
      return changed;
   }


   /**
    * Number of classes that will be shown to the user for creating stereotypes.
    *
    * @return   count of classes
    */
   public static int sizeOfStereotypeClasses()
   {
      return stereotypeClasses != null ? stereotypeClasses.size() : 0;
   }


   /**
    * Create a stereotype with given name if it does not exist, ask user for class.
    *
    * @param name  name of the new stereotype
    * @return      true if exists or successfully created
    */
   boolean createStereotype (String name)
   {
      if (!UMLStereotypeManager.get().hasKeyInStereotypes (name))
      {
         initStereotypeClasses();
         String[] classNames = new String[sizeOfStereotypeClasses()];
         int index = 0;
         for (Iterator it = iteratorOfStereotypeClasses(); it.hasNext(); )
         {
            Class cls = (Class) it.next();
            String clsName = cls.getName();
            if (clsName.indexOf ('.') >= 0)
            {
               clsName = clsName.substring (clsName.lastIndexOf ('.') + 1);
            }
            classNames[index++] = clsName;
         }
         String chosenClassName = (String) JOptionPane.showInputDialog (this, "A stereotype with the specified name does not exist. If you want to introduce " +
            "a new stereotype choose a class of stereotypes and confirm.", "Create stereotype",
            JOptionPane.OK_CANCEL_OPTION, null, classNames, classNames[0]);

         if (chosenClassName != null)
         {
            try
            {
               for (Iterator it = iteratorOfStereotypeClasses(); it.hasNext(); )
               {
                  Class cls = (Class) it.next();
                  if (cls.getName().endsWith ("." + chosenClassName))
                  {
                     final UMLStereotype newStereotype;
                     newStereotype = (UMLStereotype) cls.newInstance();
                     newStereotype.setText (name);
                     UMLStereotypeManager.get().addToStereotypes (newStereotype);
                     comboboxStereotype.addItem (name);
                     comboboxStereotype.setSelectedItem (name);
                     return true;
                  }
               }
            }
            catch (Exception e)
            {
               e.printStackTrace();
               JOptionPane.showMessageDialog (this, "Creation failed see console for further details: " + e.getMessage());
               return false;
            }
         }
         else
         {
            return false;
         }
      }
      return true;
   }
}

/*
 * $Log: StereotypeDialog.java,v $
 * Revision 1.25  2004/11/08 19:16:15  trinet
 * changed saving of FSAProperties, added String support for Maps in BasicIncrement, some cleanup
 *
 */
