/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui.mdi;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.net.URL;
import java.util.Iterator;

import javax.swing.*;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FujabaApp;
import de.upb.tools.fca.FLinkedList;


/**
 * Provide a taskbar alike pane for internal frames in a JDesktopPane.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.14.2.2 $ $Date: 2005/09/30 18:57:28 $
 */
public class DesktopTaskBar extends JPanel implements ActionListener
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (DesktopTaskBar.class);

   /**
    * desktop for frames
    */
   JDesktopPane desktop;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int PREFERRED_BUTTON_HEIGHT = 25;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int MAX_TITLE_LENGTH = 15;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   ButttonPanel taskButtonPanel;


   /**
    * Creates a new task bar
    *
    * @param desktop           desktop this taskbar is attached to
    * @param backForthButtons  if true a 'back' and a 'forth' button is added ate the beginning
    *                         of the taskbar
    */
   public DesktopTaskBar (JDesktopPane desktop, boolean backForthButtons)
   {
      super (new BorderLayout());
      this.desktop = desktop;

      desktop.addContainerListener (
         new ContainerListener()
         {
            public void componentAdded (ContainerEvent e)
            {
               if (e.getChild() instanceof JInternalFrame)
               {
                  addFrame ((JInternalFrame) e.getChild());
               }
            }


            public void componentRemoved (ContainerEvent e)
            {

            }
         });

      if (backForthButtons)
      {
         setupBackForth();
      }

      taskButtonPanel = new ButttonPanel();
      JScrollPane scrollPane =
         new JScrollPane (taskButtonPanel,
            ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED, ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER)
         {
            /**
             * @return   the value of the <code>preferredSize</code> property
             * @see      javax.swing.plaf.ComponentUI
             */
            public Dimension getPreferredSize()
            {
               Dimension preferredSize = super.getPreferredSize();
               getVerticalScrollBar().setUnitIncrement (taskButtonPanel.getSingleHeight());
               return new Dimension ((int) preferredSize.getWidth(), taskButtonPanel.getSingleHeight());
            }
         };
      scrollPane.setBorder (null);
      this.add (scrollPane, BorderLayout.CENTER);
   }


   /**
    * add buttons for a new frame.
    *
    * @param associatedFrame  added frame
    */
   void addFrame (final JInternalFrame associatedFrame)
   {
      //check if we already have it
      final Component[] menuComponents = taskButtonPanel.getComponents();
      for (int i = 0; i < menuComponents.length; i++)
      {
         Component menuComponent = menuComponents[i];
         if (menuComponent instanceof JComponent)
         {
            if ( ((JComponent) menuComponent).getClientProperty ("associatedFrame") == associatedFrame)
            {
               return;
            }
         }
      }

      final JToggleButton button =
         new JToggleButton (getFrameShortTitle (associatedFrame), associatedFrame.getFrameIcon(), true)
         {
            /**
             * If the <code>preferredSize</code> has been set to a non-<code>null</code> value
             * just returns it. If the UI delegate's <code>getPreferredSize</code> method returns
             * a non <code>null</code> value then return that; otherwise defer to the component's
             * layout manager.
             *
             * @return   the value of the <code>preferredSize</code> property
             * @see      #setPreferredSize(Dimension)
             * @see      javax.swing.plaf.ComponentUI
             */
            public Dimension getPreferredSize()
            {
               final Dimension preferredSize = super.getPreferredSize();
               if (preferredSize != null)
               {
                  preferredSize.setSize (preferredSize.getWidth(), PREFERRED_BUTTON_HEIGHT);
               }
               return preferredSize;
            }
         };
      button.setHorizontalAlignment (SwingConstants.LEFT);
      button.addActionListener (this);
      button.putClientProperty ("associatedFrame", associatedFrame);
      taskButtonPanel.add (button);

      associatedFrame.addInternalFrameListener (
         new InternalFrameAdapter()
         {
            /**
             * Invoked when an internal frame is activated.
             *
             * @param e  event
             */
            public void internalFrameActivated (InternalFrameEvent e)
            {
               button.removeActionListener (DesktopTaskBar.this);
               button.setSelected (true);

               if (!inBackForth && forthList != null)
               {
                  forthList.clear();
                  updateBackAndForth();
               }

               button.addActionListener (DesktopTaskBar.this);
            }


            /**
             * Invoked when an internal frame is de-activated.
             *
             * @param e  event
             */
            public void internalFrameDeactivated (InternalFrameEvent e)
            {
               if (!inBackForth && backList != null)
               {
                  // allow only one window change event every 50ms to avoid
                  // having the user to click back multiple times for one change
                  if (lastEventTime + 50 < System.currentTimeMillis())
                  {
                     if (backList.isEmpty() || backList.getLast() != e.getInternalFrame())
                     {
                        backList.add (e.getInternalFrame());
                        updateBackAndForth();
                     }

                     lastEventTime = System.currentTimeMillis();
                  }
               }
               button.setSelected (false);
            }


            /**
             * Invoked when an internal frame has been closed.
             *
             * @param e  event
             */
            public void internalFrameClosed (InternalFrameEvent e)
            {
               taskButtonPanel.remove (button);
               updateBackAndForth();
               revalidate();
               repaint();
            }


            /**
             * Invoked when an internal frame has been opened.
             *
             * @param e  event
             */
            public void internalFrameOpened (InternalFrameEvent e)
            {
               taskButtonPanel.add (button);
               revalidate();
               repaint();
            }


            /**
             * Invoked when an internal frame is iconified.
             *
             * @param e  No description provided
             */
            public void internalFrameIconified (InternalFrameEvent e)
            {
               e.getInternalFrame().setVisible (false);
            }
         });

      associatedFrame.addPropertyChangeListener ("title",
         new PropertyChangeListener()
         {
            public void propertyChange (PropertyChangeEvent evt)
            {
               button.setText (getFrameShortTitle (associatedFrame));
            }
         });

      revalidate();
      repaint();
   }


   /**
    * Get the frameShortTitle attribute of the DesktopTaskBar object
    *
    * @param associatedFrame  No description provided
    * @return                 The frameShortTitle value
    */
   String getFrameShortTitle (final JInternalFrame associatedFrame)
   {
      String title = associatedFrame.getTitle();
      if (title != null && title.length() > MAX_TITLE_LENGTH)
      {
         title = title.substring (0, MAX_TITLE_LENGTH - 3) + "...";
      }
      return title;
   }


   /**
    * Invoked when a button in the desktop taskbar was pressed.
    *
    * @param e  event
    */
   public void actionPerformed (ActionEvent e)
   {
      if (e.getSource() instanceof JToggleButton)
      {
         // then select the associated frame (if any)
         JToggleButton button = (JToggleButton) e.getSource();

         if (button.isSelected())
         {
            JInternalFrame associatedFrame = (JInternalFrame)  ((JComponent) e.getSource()).getClientProperty ("associatedFrame");

            if (associatedFrame != null)
            {
               activate (associatedFrame);
            }
         }
      }
   }


   /**
    * list of frames that were selected (most recently selected is the last one in the list)
    */
   FLinkedList backList;
   /**
    * list of frames that were visited and left by the 'back' button (most recently left is
    * the first one in the list)
    */
   FLinkedList forthList;
   /**
    * Filename of the 'back' icon, relative to class' package
    */
   public final static String BACK_ICON_NAME = "images/back.png";
   /**
    * Filename of the 'forth' icon, relative to class' package
    */
   public final static String FORTH_ICON_NAME = "images/forth.png";
   /**
    * 'back' button (select previously selected internal frame)
    */
   private JButton backButton;
   /**
    * 'forth' button (select internal frame recently left by the 'back' button)
    */
   private JButton forthButton;
   /**
    * true while processing 'back' or 'forth' request
    */
   boolean inBackForth;
   /**
    * last value of {@link System#currentTimeMillis()} upon occurence of last frame event
    */
   long lastEventTime;


   /**
    * enable/disable 'back' and 'forth' buttons according to lists, tidy up lists before
    */
   void updateBackAndForth()
   {
      for (Iterator iterator = backList.iterator(); iterator.hasNext(); )
      {
         JInternalFrame frame = (JInternalFrame) iterator.next();
         if (!frame.isVisible())
         {
            backList.remove (frame);
         }
      }
      for (Iterator iterator = forthList.iterator(); iterator.hasNext(); )
      {
         JInternalFrame frame = (JInternalFrame) iterator.next();
         if (!frame.isVisible())
         {
            forthList.remove (frame);
         }
      }
      forthButton.setEnabled (!forthList.isEmpty());
      backButton.setEnabled (!backList.isEmpty());
   }


   /**
    * create the 'back' and 'forth' buttons
    */
   private void setupBackForth()
   {
      JPanel backForthPanel = new JPanel (new FlowLayout (FlowLayout.LEADING));
      this.add (backForthPanel, BorderLayout.LINE_START);
      backList = new FLinkedList();
      forthList = new FLinkedList();
      backButton = new JButton();
      forthButton = new JButton();
      backButton.setEnabled (false);
      forthButton.setEnabled (false);
      URL backIconURL = FujabaApp.class.getResource (BACK_ICON_NAME);
      URL forthIconURL = FujabaApp.class.getResource (FORTH_ICON_NAME);
      if (backIconURL != null)
      {
         backButton.setIcon (new ImageIcon (Toolkit.getDefaultToolkit().createImage (backIconURL)));
      }
      else
      {
         log.error ("Icon " + BACK_ICON_NAME + " not found!");
         backButton.setText ("back");
      }
      if (forthIconURL != null)
      {
         forthButton.setIcon (new ImageIcon (Toolkit.getDefaultToolkit().createImage (forthIconURL)));
      }
      else
      {
         log.error ("Icon " + BACK_ICON_NAME + " not found!");
         forthButton.setText ("forth");
      }
      backButton.setToolTipText ("Go back to last viewed window");
      backButton.setMnemonic (KeyEvent.VK_LEFT);
      forthButton.setToolTipText ("Go forward to next viewed window");
      forthButton.setMnemonic (KeyEvent.VK_RIGHT);
      backButton.setPreferredSize (new Dimension (Math.max ((int) backButton.getMinimumSize().getWidth(), PREFERRED_BUTTON_HEIGHT),
         Math.max ((int) backButton.getMinimumSize().getHeight(), PREFERRED_BUTTON_HEIGHT)));
      forthButton.setPreferredSize (new Dimension (Math.max ((int) forthButton.getMinimumSize().getWidth(), PREFERRED_BUTTON_HEIGHT),
         Math.max ((int) forthButton.getMinimumSize().getHeight(), PREFERRED_BUTTON_HEIGHT)));
      backForthPanel.add (backButton);
      backForthPanel.add (forthButton);

      backButton.addActionListener (
         new ActionListener()
         {
            /**
             * Invoked when an action occurs.
             *
             * @param e  event
             */
            public void actionPerformed (ActionEvent e)
            {
               if (!backList.isEmpty())
               {
                  JInternalFrame frame = (JInternalFrame) backList.removeLast();
                  final JInternalFrame selectedFrame = desktop.getSelectedFrame();
                  if (selectedFrame != null)
                  {
                     forthList.addFirst (selectedFrame);
                  }
                  updateBackAndForth();
                  if (frame.getParent() == desktop)
                  {
                     inBackForth = true;
                     try
                     {
                        activate (frame);
                     }
                     finally
                     {
                        inBackForth = false;
                     }
                  }
               }
            }
         });

      forthButton.addActionListener (
         new ActionListener()
         {
            /**
             * Invoked when an action occurs.
             *
             * @param e  event
             */
            public void actionPerformed (ActionEvent e)
            {
               if (!forthList.isEmpty())
               {
                  JInternalFrame frame = (JInternalFrame) forthList.removeFirst();
                  final JInternalFrame selectedFrame = desktop.getSelectedFrame();
                  if (selectedFrame != null)
                  {
                     backList.addLast (selectedFrame);
                  }
                  updateBackAndForth();
                  if (frame.getParent() == desktop)
                  {
                     inBackForth = true;
                     try
                     {
                        activate (frame);
                     }
                     finally
                     {
                        inBackForth = false;
                     }
                  }
               }
            }
         });
   }


   /**
    * make the frame the selected frame.
    *
    * @param frame  what to activate
    */
   void activate (JInternalFrame frame)
   {
      try
      {
         frame.setSelected (true);
         frame.setVisible (true);
         frame.setIcon (false);
      }
      catch (PropertyVetoException e1)
      {
         //leave it
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: mksoft $
    * @version   $Revision: 1.14.2.2 $ $Date: 2005/09/30 18:57:28 $
    */
   private class ButttonPanel extends JPanel
   {
      /**
       *Constructor for class ButttonPanel
       */
      public ButttonPanel()
      {
         super (new FlowLayout (FlowLayout.LEFT));
      }


      /**
       * @return   the value of the <code>preferredSize</code> property
       * @see      javax.swing.plaf.ComponentUI
       */
      public Dimension getPreferredSize()
      {
         Dimension preferredSize = super.getPreferredSize();
         int newWidth = (int) preferredSize.getWidth();
         int newHeight = (int) preferredSize.getHeight();
         int width = getWidth();
         int vgap =  ((FlowLayout) getLayout()).getVgap();
         while (newWidth > width && width > 0)
         {
            newHeight += preferredSize.getHeight() - vgap;
            newWidth -= width;
         }
         return new Dimension (newWidth, newHeight);
      }


      /**
       * Get the singleHeight attribute of the ButttonPanel object
       *
       * @return   The singleHeight value
       */
      public int getSingleHeight()
      {
         return (int) super.getPreferredSize().getHeight() -  ((FlowLayout) getLayout()).getVgap();
      }
   }
}

/*
 * $Log: DesktopTaskBar.java,v $
 * Revision 1.14.2.2  2005/09/30 18:57:28  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
