/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.metamodel;

import java.util.Enumeration;
import java.util.Iterator;

import de.upb.tools.sdm.Path;


/**
 * Represents a class (e.g. UML class).
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.4.2.1 $ $Date: 2006/01/23 15:56:42 $
 */
public interface FClass extends FDeclaration, FType
{

   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String FULL_CLASS_NAME_PROPERTY = "fullClassName";


   /**
    * This function computes the full qualified classname of this class e. g. if the class
    * <code>FooBar</code> is defined in the package <code>upb</code> and <code>upb</code> is
    * in <code>de</code> then this function will return <code>"de.upb.FooBar"</code>.
    *
    * @return   the full qualified class name
    */
   public abstract String getFullClassName();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String ABSTRACT_PROPERTY = "abstract";


   /**
    * Change the 'abstract' flag of a class.
    *
    * @param value  true to make a class abstract
    * @return       true if value was changed, false if it already had the specified value
    */
   public abstract boolean setAbstract (boolean value);


   /**
    * Read the 'abstract' flag of the class.
    *
    * @return   true if this is an abstract class
    */
   public abstract boolean isAbstract();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String FILE_PROPERTY = "file";


   /**
    * Sets the file this class is defined in.
    *
    * @param file  file this class is defined in
    * @return      true if value was changed, false if it already had the specified value
    */
   public abstract boolean setFile (FFile file);


   /**
    * Get the file this class is defined in.
    *
    * @return   the file
    */
   public abstract FFile getFFile();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String REV_IMPORTED_CLASSES_PROPERTY = "revImportedClasses";


   /**
    * @param file  file that possibly includes this class
    * @return      true if this class is imported in the specified file
    * @see         #iteratorOfRevImportedClasses()
    */
   public abstract boolean hasInRevImportedClasses (FFile file);


   /**
    * @return   enumeration of files that import this class
    * @see      #iteratorOfRevImportedClasses()
    */
   public abstract Enumeration elementsOfRevImportedClasses();


   /**
    * The revImportedClasses property is the conterpart of FFile.importedClasses. It lists
    * all files that import this class (e.g. in their Java codes import clause).
    *
    * @return   iterator through files that import this class
    * @see      FFile#iteratorOfImportedClasses()
    */
   public abstract Iterator iteratorOfRevImportedClasses();


   /**
    * @param file  file that imports this class now
    * @see         #iteratorOfRevImportedClasses()
    */
   public abstract void addToRevImportedClasses (FFile file);


   /**
    * @param file  that does no longer import this class
    * @see         #iteratorOfRevImportedClasses()
    */
   public abstract void removeFromRevImportedClasses (FFile file);


   /**
    * Access method for a To N-association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToParsedMembers (FDeclaration value);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInParsedMembers (FDeclaration value);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfParsedMembers();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromParsedMembers();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromParsedMembers (FDeclaration value);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfParsedMembers();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String ATTRS_PROPERTY = "attrs";


   /**
    * @param attr  attribute that is member of this class now (is declared by this class)
    * @return      true if value was added, false is already linked
    * @see         #iteratorOfAttrs()
    */
   public abstract boolean hasInAttrs (FAttr attr);


   /**
    * @param key  name of an attribute
    * @return     true if this class declares an attribute with the specified name
    * @see        #iteratorOfAttrs()
    */
   public abstract boolean hasKeyInAttrs (String key);


   /**
    * The attrs property of FClass contains all attributes ({@link FAttr}s) that are declared
    * in this class. The reverse property is FAttr.parent.
    *
    * @return   iterator through declared {@link FAttr}s
    * @see      FAttr#getFParent()
    */
   public abstract Iterator iteratorOfAttrs();


   /**
    * @return   iterator through attribute names (Strings)
    * @see      #iteratorOfAttrs()
    */
   public abstract Iterator keysOfAttrs();


   /**
    * @return   iterator through attribute entries (String name, FAttr attribute)
    * @see      #iteratorOfAttrs()
    */
   public abstract Iterator entriesOfAttrs();


   /**
    * @return   number of attributes this class declares
    * @see      #iteratorOfAttrs()
    */
   public abstract int sizeOfAttrs();


   /**
    * @param attrName  name of the attribute to return
    * @return          the attribute with the specified name, declared in this class, null
    *      if no attribute of this name is declared by this class
    * @see             #iteratorOfAttrs()
    */
   public abstract FAttr getFromFAttrs (String attrName);


   /**
    * @param attr  attribute that is declared by this class now
    * @return      true if added, false if was already added
    * @see         #iteratorOfAttrs()
    */
   public abstract boolean addToAttrs (FAttr attr);


   /**
    * @param attr  attribute that is no longer declared by this class
    * @return      true if removed, false if was already removed (not contained)
    * @see         #iteratorOfAttrs()
    */
   public abstract boolean removeFromAttrs (FAttr attr);


   /**
    * @param attrName  name of the attribute to be removed
    * @return          true if removed, false if was already removed (not contained)
    * @see             #iteratorOfAttrs()
    */
   public abstract boolean removeKeyFromAttrs (String attrName);


   /**
    * Remove all declared attributes from this class.
    *
    * @see   #iteratorOfAttrs()
    */
   public abstract void removeAllFromAttrs();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String ROLES_PROPERTY = "roles";


   /**
    * @return   number of roles this class has
    * @see      #iteratorOfRoles()
    */
   public abstract int sizeOfRoles();


   /**
    * @param role  potential role this class has
    * @return      true if class has the specified role
    * @see         #iteratorOfRoles()
    */
   public abstract boolean hasInRoles (FRole role);


   /**
    * @return   enumeration of all roles this class has in associations
    * @see      #iteratorOfRoles()
    */
   public abstract Enumeration elementsOfRoles();


   /**
    * The roles property lists all roles this class has in associations. A role is part of
    * an association.
    *
    * @return   iterator through all roles this class has in associations
    * @see      FRole
    * @see      FRole#getFRevRoles()
    */
   public abstract Iterator iteratorOfRoles();


   /**
    * @param roleName  name of a role
    * @return          the role of this class with the specified name, null if no role with
    *      this name
    * @see             #iteratorOfRoles()
    */
   public abstract FRole getFromFRoles (String roleName);


   /**
    * @param role  new role this class has now
    * @see         #iteratorOfRoles()
    */
   public abstract void addToRoles (FRole role);


   /**
    * @param elem  role this class this class has no more
    * @see         #iteratorOfRoles()
    */
   public abstract void removeFromRoles (FRole elem);


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String REV_ROLES_PROPERTY = "revRoles";


   /**
    * Find a partner role of any role this class has: E.g. Class 'Man' associates class 'Boy'
    * where Man has the role "parent" and Boy has the role "child". You can now ask Man for
    * getFromRevRoles("child") to find the described role at Boy. todo: The behaviour of this
    * method is not soundly defined if mutiple partner roles exist with the specified name
    *
    * @param roleName  name of a partner role
    * @return          a partner role
    */
   public abstract FRole getFromFRevRoles (String roleName);


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String METHODS_PROPERTY = "methods";


   /**
    * @param method  potential method in this class
    * @return        true if method is declared in this class
    * @see           #iteratorOfMethods()
    */
   public abstract boolean hasInMethods (FMethod method);


   /**
    * @param signature  signature of a method
    * @return           true if this class declares a method with this signature
    * @see              #iteratorOfMethods()
    */
   public abstract boolean hasKeyInMethods (String signature);


   /**
    * The methods property contains all methods this class declares.
    *
    * @return   iterator through declared methods
    * @see      FMethod#getFParent()
    */
   public abstract Iterator iteratorOfMethods();


   /**
    * @return   number of mthods in this class
    * @see      #iteratorOfMethods()
    */
   public abstract int sizeOfMethods();


   /**
    * @param signature  possible signature
    * @return           the method declared in this class with the specified signature, null
    *      if no such method
    * @see              #iteratorOfMethods()
    */
   public abstract FMethod getFromFMethods (String signature);


   /**
    * @param method  method that is declared in this class now
    * @return        true if added, false if was already added
    * @see           #iteratorOfMethods()
    */
   public abstract boolean addToMethods (FMethod method);


   /**
    * @param method  method that is no longer declared in this class
    * @return        true if removed, false if was already removed (not contained)
    * @see           #iteratorOfMethods()
    */
   public abstract boolean removeFromMethods (FMethod method);


   /**
    * @param signature  signature of the method to be removed from declared methods
    * @return           true if removed, false if was already removed (not contained)
    * @see              #iteratorOfMethods()
    */
   public abstract boolean removeKeyFromMethods (String signature);


   /**
    * Remove all methods declared in this class.
    *
    * @see   #iteratorOfMethods()
    */
   public abstract void removeAllFromMethods();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String DECLARED_IN_PACKAGE_PROPERTY = "declaredInPackage";


   /**
    * @param pkg  the package this class is declared in now
    * @return     true if the class was declared in another package before
    * @see        #getFDeclaredInPackage()
    */
   public abstract boolean setDeclaredInPackage (FPackage pkg);


   /**
    * The declaredInPackage property denotes the package this class is declared in, null if
    * this is an inner or anonymous class.
    *
    * @return   the package this class is declared in, null if not applicable
    * @see      FPackage#iteratorOfDeclares()
    * @see      #getFDeclaredInClass()
    * @see      #getFDeclaredInMethod()
    */
   public abstract FPackage getFDeclaredInPackage();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String DECLARED_IN_METHOD_PROPERTY = "declaredInMethod";


   /**
    * @param method  method where this class is declared in now
    * @return        true if value was changed
    * @see           #getFDeclaredInMethod()
    */
   public abstract boolean setDeclaredInMethod (FMethod method);


   /**
    * The declaredInMethod property denotes the method this class is declared in if it is an
    * anonymous class, null if this is a top level or inner class.
    *
    * @return   the method this anonymous class is declared in, null if not applicable
    * @see      FMethod#iteratorOfDeclares()
    * @see      #getFDeclaredInClass()
    * @see      #getFDeclaredInMethod()
    */
   public abstract FMethod getFDeclaredInMethod();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String DECLARED_IN_CLASS_PROPERTY = "declaredInClass";


   /**
    * @param cls  class where this class is declared in now
    * @return     true if value was changed
    * @see        #getFDeclaredInClass()
    */
   public abstract boolean setDeclaredInClass (FClass cls);


   /**
    * The declaredInClass property denotes the class this inner class is declared in, null
    * if this is a top level or anonymous class.
    *
    * @return   the class this inner class is declared in, null if not applicable
    * @see      FClass#iteratorOfDeclares()
    * @see      #getFDeclaredInPackage()
    * @see      #getFDeclaredInMethod()
    */
   public abstract FClass getFDeclaredInClass();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String DECLARES_PROPERTY = "declares";


   /**
    * @param cls  possible inner class
    * @return     true if cls is an inner class of this class
    * @see        #iteratorOfDeclares()
    */
   public abstract boolean hasInDeclares (FClass cls);


   /**
    * @param classname  unqualified classname
    * @return           true if this class has an inner class with specified name
    * @see              #iteratorOfDeclares()
    */
   public abstract boolean hasKeyInDeclares (String classname);


   /**
    * The declares property holds all inner classes of this class.
    *
    * @return   an iterator through inner classes
    */
   public abstract Iterator iteratorOfDeclares();


   /**
    * @return   iterator through the unqualified names of inner classes (Strings)
    * @see      #iteratorOfDeclares()
    */
   public abstract Iterator keysOfDeclares();


   /**
    * @return   iterator through inner class entries (String name, FClass class)
    * @see      #iteratorOfDeclares()
    */
   public abstract Iterator entriesOfDeclares();


   /**
    * @return   number of inner classes
    * @see      #iteratorOfDeclares()
    */
   public abstract int sizeOfDeclares();


   /**
    * @param classname  unqualified classname
    * @return           the inner class declared in this class with the specified name, null
    *      if no inner class of that name
    * @see              #iteratorOfDeclares()
    */
   public abstract FClass getFromFDeclares (String classname);


   /**
    * @param cls  the class that is inner class of this class now
    * @return     true if the value was added, false if it was already an inner class of this
    *      class
    * @see        #iteratorOfDeclares()
    */
   public abstract boolean addToDeclares (FClass cls);


   /**
    * @param cls  class that is no longer an inner class of this class
    * @return     true if class was removed, false if it was no inner class of this class
    * @see        #iteratorOfDeclares()
    */
   public abstract boolean removeFromDeclares (FClass cls);


   /**
    * @param classname  unqualified classname of an inner class
    * @return           true if the class with specified name was removed, false if no inner
    *      class with this name
    * @see              #iteratorOfDeclares()
    */
   public abstract boolean removeKeyFromDeclares (String classname);


   /**
    * Remove all inner classes (delete link, classes remain intact).
    *
    * @see   #iteratorOfDeclares()
    */
   public abstract void removeAllFromDeclares();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String REV_SUBCLASS_PROPERTY = "revSubclass";


   /**
    * @param generalization  possible generalization that has this class as subclass
    * @return                true if this is the subclass in the generalization
    * @see                   #iteratorOfRevSubclass()
    */
   public abstract boolean hasInRevSubclass (FGeneralization generalization);


   /**
    * @return   number of superclasses/interfaces
    * @see      #iteratorOfRevSubclass()
    */
   public abstract int sizeOfRevSubclass();


   /**
    * The revSubclass property contains alls generalizations where this class is the subclass.
    * This means all superclasses can be found this way.
    *
    * @return   iterator through generalizations
    */
   public abstract Iterator iteratorOfRevSubclass();


   /**
    * @param elem  generalization where this class is the subclass now
    * @see         #iteratorOfRevSubclass()
    */
   public abstract void addToRevSubclass (FGeneralization elem);


   /**
    * @param elem  generalization where this class is no longer the subclass
    * @see         #iteratorOfRevSubclass()
    */
   public abstract void removeFromRevSubclass (FGeneralization elem);


   /**
    * Remove all generalizations where this class is the subclass.
    *
    * @see   #iteratorOfRevSubclass()
    */
   public abstract void removeAllFromRevSubclass();


   /**
    * @return   enumeration of generalizations
    * @see      #iteratorOfRevSubclass()
    */
   public abstract Enumeration elementsOfRevSubclass();


   /**
    * @return       No description provided
    * @deprecated   please use {@link #iteratorOfSuperClasses()}
    */
   public abstract Path elementsOfSuperClasses();


   /**
    * @return   iterator through all superclasses and interfaces
    */
   public abstract Path iteratorOfSuperClasses();


   /**
    * Property name used for change events, unparsing and persistency - do not change.
    */
   public final static String REV_SUPERCLASS_PROPERTY = "revSuperclass";


   /**
    * @param elem  generalization
    * @return      true if this class is the superclass in the generalization
    * @see         #iteratorOfRevSuperclass()
    */
   public abstract boolean hasInRevSuperclass (FGeneralization elem);


   /**
    * @return   enumeration of generalizations where this class is the superclass
    * @see      #iteratorOfRevSuperclass()
    */
   public abstract Enumeration elementsOfRevSuperclass();


   /**
    * The revSuperclass property contains all generalization where this class is the superclass.
    * You can find subclasses on this way.
    *
    * @return   iterator through generalizations where this class is the superclass
    */
   public abstract Iterator iteratorOfRevSuperclass();


   /**
    * @param elem  generalization where this class is the superclass now
    * @see         #iteratorOfRevSuperclass()
    */
   public abstract void addToRevSuperclass (FGeneralization elem);


   /**
    * @param elem  generalization where this class is no longer the superclass
    * @see         #iteratorOfRevSuperclass()
    */
   public abstract void removeFromRevSuperclass (FGeneralization elem);


   /**
    * Remove all generalizations where this class is the superclass.
    *
    * @see   #iteratorOfRevSuperclass()
    */
   public abstract void removeAllFromRevSuperclass();

   // --- Deprecated methods ---
   /**
    * @return       The static value
    * @deprecated   Use <<static>> stereotype instead
    */
   public abstract boolean isStatic();


   /**
    * @param value  The new static value
    * @deprecated   Use <<static>> stereotype instead
    */
   public abstract void setStatic (boolean value);


   /**
    * @return       The final value
    * @deprecated   Use <<final>> stereotype instead
    */
   public abstract boolean isFinal();


   /**
    * @return       The reference value
    * @deprecated   Use <<reference>> stereotype instead
    */
   public abstract boolean isReference();


   /**
    * @return       The fSuperClass value
    * @deprecated   will be renamed when using JDK 1.5 (without F)
    */
   public abstract FClass getFSuperClass();


   /**
    * Search for an attribute in the class hierarchy. An attribute with the specified name
    * is searched in this class first and then in all superclasses, recursively.
    *
    * @param attributeName  name of an attribute to be found
    * @return               the first matching attribute
    * @deprecated           will be renamed when using JDK 1.5 (without F)
    */
   public FAttr getFromAllFAttrs (String attributeName);


   /**
    * Search for a method in the class hierarchy. A method with the specified signature is
    * searched in this class first and then in all superclasses, recursively.
    *
    * @param methodSignature  name of an attribute to be found
    * @return                 the first matching method
    * @deprecated             will be renamed when using JDK 1.5 (without F)
    */
   public FMethod getFromAllFMethods (String methodSignature);


   /**
    * Find a method with specified name. todo: The behaviour of this method is not soundly
    * defined if more than one method with the given name exists
    *
    * @param methodname  name (not signature) of a method
    * @return            a method, declared in this class, with specified name
    * @deprecated        will be renamed when using JDK 1.5 (without F)
    */
   public abstract FMethod getFromFMethodsByShortName (String methodname);


   /**
    * Find a method with specified name in this class or superclasses. The first method that
    * is found searching from this class to the superclasses is returned. todo: The behaviour
    * of this method is not soundly defined if more than one method with the given name exists
    * in a single class or in different superclasses.
    *
    * @param methodname  name (not signature) of a method
    * @return            a method, declared in this class or inherited from another class,
    *      with specified name
    * @deprecated        will be renamed when using JDK 1.5 (without F)
    */
   public abstract FMethod getFromFMethodsByShortNameIncludingInherited (String methodname);


   /**
    * @return       the statechart for this class, null if not applicable
    * @deprecated   will be renamed when using JDK 1.5 (without F)
    */
   public FDiagram getFStatechart();


   /**
    * todo: is this used by anyone?!
    *
    * @return       the reactive diagram associated with this class
    * @deprecated   will be renamed when using JDK 1.5 (without F) or removed?
    */
   public abstract FDiagram getFReactiveDiagram();


   /**
    * @return       package this class is enclosed in - even for inner and anonymous classes
    * @deprecated   will be renamed when using JDK 1.5 (without F)
    */
   public abstract FPackage findFPackage();
}

/*
 * $Log: FClass.java,v $
 * Revision 1.4.2.1  2006/01/23 15:56:42  cschneid
 * UMLClass--UMLMethod assoc no longer qualified
 *
 */
