/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.preferences;

import java.awt.*;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Field;
import java.util.*;

import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FTreeMap;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.7 $ $Date: 2004/10/20 17:50:07 $
 */
public class ColorsPreferences extends AbstractPreferences
{
   /**
    * Prefix for color preferences
    */
   public final static String PROPERTY_PREFIX = "Colors.";

   /**
    * Property for a list of color names
    */
   public final static String PROPERTY_NAMES = "names";

   /**
    * Added for getVectorSetting call
    */
   private final static Vector DEF_CODE_GEN_TARGET_NAME = new Vector();

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color DEFAULT_FOREGROUND = Color.black;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color DEFAULT_BACKGROUND = Color.white;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color DIAGRAM_BACKGROUND = DEFAULT_BACKGROUND;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color DEFAULT_FOCUSED = Color.decode ("2987775");

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color DEFAULT_SELECTED = Color.blue;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color DEFAULT_BORDER = DEFAULT_FOREGROUND;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color SELECTED_BORDER = DEFAULT_SELECTED;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color CREATION = Color.GREEN.darker();

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color ASSIGNMENT = CREATION;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color DELETION = Color.red;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color OPTIONAL = Color.gray;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color ERROR = Color.red;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color ACTIVITY_BACKGROUND = new Color (255, 255, 240);

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color CLASS_BACKGROUND = DEFAULT_BACKGROUND;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color STORY_BACKGROUND = new Color (235, 235, 255);

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color SELECTION_RECT = Color.lightGray;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color COLLAPSE_ANCHOR = Color.gray;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color EDITOR_BORDER = Color.gray;

   /**
    * Constant for default drawing of Fujaba GUI (do not make static to allow default value
    * reading)
    */
   public Color MULTILINK = Color.lightGray;


   /**
    * Default constructor
    *
    * @param init  No description provided
    */
   private ColorsPreferences (boolean init)
   {
      if (init)
      {
         setDefaults (true);
         load();
      }
   }


   /**
    * Stores color (String -> Color)
    */
   private Map colors;

   /**
    * The color options for this VM
    */
   private static ColorsPreferences options;


   /**
    * Sets the defaults attribute of the AbstractPreferences object
    */
   public void setDefaults()
   {
      setDefaults (false);
   }


   /**
    * Sets the defaults attribute of the AbstractPreferences object
    *
    * @param temporarily  when true save() is not called
    */
   private void setDefaults (boolean temporarily)
   {
      ColorsPreferences defaultOptions = new ColorsPreferences (false);
      Field[] constants = getClass().getFields();
      for (int i = 0; i < constants.length; ++i)
      {
         if (constants[i].getType().equals (Color.class))
         {
            try
            {
               setColor (constants[i].getName(), (Color) constants[i].get (defaultOptions));
            }
            catch (Exception e)
            {
               //ignore
               e.printStackTrace();
            }
         }
      }

      if (!temporarily)
      {
         save();
      }
   }


   // ----------------------------------------------------------------------------
   // property change support
   // ----------------------------------------------------------------------------

   /**
    * Access method for an one to n association.
    *
    * @param propertyName  The object added.
    * @param listener      The object added.
    */
   public void addPropertyChangeListener (String propertyName, PropertyChangeListener listener)
   {
      // use default properties for the Fujaba core
      PreferencesProperties.get().addPropertyChangeListener (propertyName, listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param propertyName  No description provided
    * @param listener      No description provided
    */
   public void removePropertyChangeListener (String propertyName, PropertyChangeListener listener)
   {
      // use default properties for the Fujaba core
      PreferencesProperties.get().removePropertyChangeListener (propertyName, listener);
   }


   // ----------------------------------------------------------------------------
   // getter and setter for the properties
   // ----------------------------------------------------------------------------

   /**
    * Change a named color
    *
    * @param name   name of the color or coloured item to be altered
    * @param color  new value for color
    */
   public void setColor (String name, Color color)
   {
      if (colors == null)
      {
         colors = new FTreeMap();
      }
      setModified (true);
      colors.put (name, color);

      try
      {
         Field constant = getClass().getField (name);
         if (constant != null && constant.getType().equals (Color.class))
         {
            constant.set (this, color);
         }
      }
      catch (Exception e)
      {
         //ignore silently
      }
   }


   /**
    * Load colors from PropertyOptions
    */
   public void load()
   {
      Vector names = PreferencesProperties.get().getVectorSetting (PROPERTY_PREFIX + PROPERTY_NAMES,
         DEF_CODE_GEN_TARGET_NAME);
      Iterator it = names.iterator();
      while (it.hasNext())
      {
         String name = (String) it.next();
         Color color = getColor (name);
         if (color == null)
         {
            color = getDefaultColor (name);
         }
         color = PreferencesProperties.get().getColorSetting (PROPERTY_PREFIX + name, color);
         setColor (name, color);
      }
   }


   /**
    * @param name  name of a color
    * @return      the default value for this color
    */
   private Color getDefaultColor (String name)
   {
      return name == null ? null : Color.black;
   }


   /**
    * Save colors to PropertyOptions
    */
   public void save()
   {
      Vector names = new Vector();
      Iterator it = getColors();
      while (it.hasNext())
      {
         String name = (String) it.next();
         PreferencesProperties.get().putSetting (PROPERTY_PREFIX + name, getColor (name));
         names.add (name);
      }
      PreferencesProperties.get().putSetting (PROPERTY_PREFIX + PROPERTY_NAMES, names);
   }


   /**
    * Obtain the color for a name
    *
    * @param name  name of the color or coloured item
    * @return      the associated Color
    */
   public Color getColor (String name)
   {
      if (colors == null)
      {
         colors = new FTreeMap();
      }
      return (Color) colors.get (name);
   }


   /**
    * @return   An Iterator through all color names (Strings)
    */
   public Iterator getColors()
   {
      if (colors == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return colors.keySet().iterator();
      }
   }


   /**
    * @return   color Options for this VM
    */
   public static ColorsPreferences get()
   {
      if (options == null)
      {
         options = new ColorsPreferences (true);
      }

      return options;
   }
}

/*
 * $Log: ColorsPreferences.java,v $
 * Revision 1.7  2004/10/20 17:50:07  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
