/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.preferences;

import java.io.*;
import java.net.URL;
import java.util.*;

import javax.xml.parsers.*;

import org.apache.log4j.*;
import org.apache.log4j.helpers.OptionConverter;
import org.apache.log4j.spi.LoggerRepository;
import org.apache.log4j.varia.NullAppender;
import org.apache.log4j.xml.DOMConfigurator;
import org.apache.xerces.utils.XMLCharacterProperties;
import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;
import org.w3c.dom.*;
import org.xml.sax.*;

import de.uni_paderborn.fujaba.logging.LoggerInfo;
import de.uni_paderborn.fujaba.logging.LoggingConfigException;


/**
 * <P>
 *
 * Singleton class containing options for log4j</P> <P>
 *
 * To use the logging mechanism, create a static variable in your class following this pattern:
 * </P> <P>
 *
 * <CODE>private static final Logger L = Logger.getLogger (OptionsLog4J.class);</CODE><P>
 *
 * <P>
 *
 * You now have a log4j Logger (org.apache.log4j.Logger) for your class. Detailed instructions
 * for use can be found on <A href="http://jakarta.apache.org/log4j"> the log4j web site</A>
 * but in general, logging statements will look something like this:</P> <P>
 *
 * <CODE>
 * if (L.isDebugEnabled())
 * {
 *    L.debug ("This is a log statement at debug level");
 * }
 * </CODE> or <CODE>
 * if (L.isInfoEnabled())
 * {
 *    L.info ("This is a log statement at info level");
 * }
 * </CODE> <P>
 *
 * <I>NB The use of the <CODE>isLEVELEnabled</CODE> methods is not required, but prevents log
 * statements being unnecessarily created if they will not be output as they are at too low
 * a level. Please refer to the log4j documentation for further information. </I> </P>
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.6.2.3 $
 */
public class LoggingPreferences extends AbstractPreferences
{

   // NB: "Log" messages in this class are simply output to stdout or stderr as they
   // are either printed before the logging mechanism has been configured or are error
   // messages to say there was a problem configuring the logging mechanism (the user
   // should probably know this and it cannot be predicted where the message will end up if
   // it is sent to the logging mechanism regardless) Other opinions on
   // this approach are however very welcome! Suzy (suzy@upb.de)

   /**
    * The instance of this singleton class. Use OptionsLog4J.get() to get an instance of this
    * class.
    *
    * @see   #get
    */
   private static LoggingPreferences options;

   /**
    * The DOM Document containing the options for log4j.
    */
   private Document log4JConfig;

   /**
    * The file which the log4j options are stored in NB: Changed to follow .fujabarc
    */
   private String configFile = PreferencesProperties.getPropertyDir() + "core" + File.separatorChar + "log4j-config.xml";
   // was:
   // private String configFile = System.getProperty ("user.home") + File.separatorChar + "log4j-config.xml";

   /**
    * The file to load default options from if the user's config file is not available
    */
   private String defaultConfigFile = "/de/uni_paderborn/fujaba/logging/log4j-default-config.xml";
   // nb: extra / added to the beginning of this string for resource location purposes

   /**
    * Cache of appender nodes for appender names
    */
   private Hashtable appenderCache = new Hashtable();


   /**
    * Cache of logger nodes for logger names
    */
   //private Hashtable loggerCache = new Hashtable();


   /**
    * The <B>private</B> constructor for this class. Use OptionsLog4J.get() to get an instance
    * of this class.
    *
    * @see   #get
    */
   protected LoggingPreferences()
   {
      // load in options, creating new file if necessary
      // if file seems to exist but there is a problem with it, then the defaults are loaded
      // note that this is different to setting the defaults from the GUI as this would
      // cause the configFile to be overwritten with the defaults (as per what seems to be
      // Fujaba standard with other property screens...) whereas here the defaults are simply temporarily loaded

      // check whether file exists
      if (new File (configFile).exists())
      {
         // try to load in options from file
         try
         {
            loadFile (new File (configFile));
            configureLog4J();
         }
         catch (LoggingConfigException logEx)
         {
            // OK, well I'd say this should go to a log but at
            // this point in time that doesn't look like an option ;)
            // Consider sys property to check to see whether messages
            // about logging such as these should be output
            System.err.println ("Error whilst loading log4j settings from " + configFile);
            System.err.println ("Error was:");
            System.err.println (logEx.getMessage());
            System.err.println ("Reverting to default settings.");

            initWithDefaults();

            if (configLoaded())
            {
               configureLog4J();
            }
         }
         /*
          *  catch (SAXException saxEx)
          *  {
          *  / See above for why this is going to System.err.println
          *  System.err.println ("Parsing error whilst loading log4j settings from " + configFile);
          *  System.err.println ("Error was:");
          *  System.err.println (saxEx.toString());
          *  System.err.println ("Reverting to default settings.");
          *  loadDefaults();
          *  }
          */
      }
      else
      {
         System.err.println ("No log4j configuration file found - loading default settings from " + defaultConfigFile);

         initWithDefaults();

         if (configLoaded())
         {
            // write to file
            saveSettings();
            configureLog4J();
         }
      }

   } // OptionsLog4J


   /**
    * Tries to initialise the mechanism using the default settings, and takes appropriate action
    * if this is not possible.
    */
   private void initWithDefaults()
   {
      try
      {
         loadDefaults();
      }
      catch (LoggingConfigException e)
      {
         System.err.println (e.getMessage());

         // This really really shouldn't happen
         // It could only happen if the Fujaba installation is dodgy, or if someone
         // has screwed up the default logging file
         // Assume best course of action is to turn everything off for the time being

         System.err.println ("Could not initialise logging mechanism.  All logging stopped until mechanism is configured.");

         LoggerRepository repository = LogManager.getLoggerRepository();

         // reset the configuration to kill any dodgy default configuration which has snuck in
         repository.resetConfiguration();

         // stick a null appender on the root logger to stop log4j constantly complaining it has no
         // appenders
         Logger rootLogger = repository.getRootLogger();

         rootLogger.addAppender (new NullAppender());

         // set threshold to null to stop any messages getting through (and to not waste time
         // creating messages which aren't going to go anywhere)
         rootLogger.setLevel (Level.OFF);

         // NB: Have taken this approach rather than changing the threshold of the repository as
         // it can be overridden by later configurations... (having said that seeing as we reset the
         // repository in any case this would probably be fine anyway?)
      }
   }


   /**
    * This method provides the user the singleton instance of this class.
    *
    * @return   the singleton instance of this class
    */
   public static synchronized LoggingPreferences get()
   {
      if (options == null)
      {
         options = new LoggingPreferences();
      }
      return options;
   } // get



   /**
    * Set the default options. Loads default options and then uses them to configure log4j.
    */
   public void setDefaults()
   {
      appenderCache = new Hashtable();
      //loggerCache = new Hashtable();
      try
      {
         loadDefaults();
      }
      catch (LoggingConfigException e)
      {
         System.err.println (e.getMessage());
      }

      if (log4JConfig != null)
      {
         configureLog4J();
      }
   } // setDefaults


   /**
    * Loads a file of configuration options.
    *
    * @param file                     the configuration file
    * @throws LoggingConfigException  if there are problems loading the file
    */
   public void loadFile (File file) throws LoggingConfigException
   {
      appenderCache = new Hashtable();

      FileInputStream configStream = null;

      try
      {
         configStream = new FileInputStream (file);
      }
      catch (FileNotFoundException e)
      {
         throw new LoggingConfigException ("Cannot load " + file.getName() + " - file not found.");
      }

      ParseConfigErrorHandler handler = new ParseConfigErrorHandler();

      Document tempConfig = null;

      try
      {
         tempConfig = parse (configStream, handler);
         configStream.close();
      }
      catch (IOException ioEx)
      {
         throw new LoggingConfigException ("I/O problem encountered whilst parsing " + file.getName() + ".\n" + ioEx.getMessage());
      }
      catch (SAXException saxEx)
      {
         throw new LoggingConfigException ("Parsing error in " + file.getName() + ".\n" + saxEx.getMessage());
      }

      if (handler.getErrorCount() > 0)
      {
         throw new LoggingConfigException ("Parsing error in " + file.getName() + ".\n" + handler.getMessage());
      }

      if (tempConfig == null)
      {
         throw new LoggingConfigException ("Unidentified error whilst parsing " + file.getName());
      }

      log4JConfig = tempConfig;
   }


   /**
    * Configure log4j according to options stored in this object.
    */
   public void configureLog4J()
   {
      LogManager.resetConfiguration();
      // otherwise loggers and appenders
      // which previously had settings but
      // are now simply not mentioned
      // will carry on as before (think this
      // would be confusing to user)

      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - log4j not configured.");
         return;
      }

      DOMConfigurator.configure (log4JConfig.getDocumentElement());
   } // configureLog4J


   /**
    * Save the current options to user's log4j config file.
    */
   public void saveSettings()
   {
      // avoid NPE probs
      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - settings have not been saved");
         return;
      }

      // use DOM tree to determine output format
      OutputFormat format = new OutputFormat (log4JConfig);

      // turn indenting on for us human beens
      format.setIndenting (true);

      // create directory for file if necessary
      File parentDir = new File (configFile).getParentFile();

      if (parentDir != null)
      {
         parentDir.mkdirs();
      }

      // open config file for writing
      FileWriter configWriter;

      try
      {
         configWriter = new FileWriter (configFile);
      }
      catch (IOException ioEx)
      {
         System.err.println ("Could not open " + configFile + " for writing.");
         System.err.println ("log4j settings not saved.");
         return;
      }

      // serialize DOM tree, output to config file
      XMLSerializer serializer = new XMLSerializer (configWriter, format);

      try
      {
         serializer.serialize (log4JConfig);
         // if (log.isInfoEnabled()) log.info ("log4j settings saved to " + configFile);
      }
      catch (IOException ioEx)
      {
         System.err.println ("Could not save log4j settings to " + configFile + "!");
      }

      // close file
      try
      {
         configWriter.close();
      }
      catch (IOException ioEx)
      {
         System.err.println ("Could not close connection to " + configFile);
      }
   } // saveSettings


   /**
    * Generate a message of every level to each logger to test configuration. This works on
    * the currently loaded config - if changes were made in the GUI, the user needs to have
    * applied them before the test will use the settings displayed there.
    */
   public void generateTestMsgs()
   {
      // Get all loggers currently registered with log4j
      Enumeration loggers = LogManager.getCurrentLoggers();

      while (loggers.hasMoreElements())
      {
         // Have checked log4j source code, and even if a Category is declared
         // in the config file, DOMConfigurator sets it up as a Logger.
         // Therefore we can safely cast to Logger
         Logger logger = (Logger) loggers.nextElement();
         testLogger (logger);
      }

      // Root logger does not appear in above enumeration so test that too
      Logger rootLogger = LogManager.getRootLogger();
      testLogger (rootLogger);
   }


   /**
    * Make a test call at each level to a specified logger
    *
    * @param logger  the logger to test
    */
   private void testLogger (Logger logger)
   {
      String name = logger.getName();

      logger.debug ("Testing a DEBUG call to logger " + name);
      logger.info ("Testing an INFO call to logger " + name);
      logger.warn ("Testing a WARN call to logger " + name);
      logger.error ("Testing an ERROR call to logger " + name);
      logger.fatal ("Testing a FATAL call to logger " + name);
   }


   /**
    * Checks a String to see if it is a valid logger name
    *
    * @param name  the name to check
    * @return      <CODE>true</CODE> if the name is valid, <CODE>false</CODE> if not
    */
   public boolean isValidLoggerName (String name)
   {
      // return XMLChar.isValidName (name);
      return XMLCharacterProperties.validName (name);
   }


   /**
    * Load the default log4j settings.
    *
    * @throws LoggingConfigException  if there is a problem whilst loading the default options
    */
   private void loadDefaults() throws LoggingConfigException
   {
      // create new DOM based on default settings
      // try to load in options from file

      // boolean successful = false;

      ParseConfigErrorHandler handler = new ParseConfigErrorHandler();
      Document tempConfig = null;

      try
      {
         Class clazz = this.getClass();
         URL defaultsURL = clazz.getResource (defaultConfigFile);
         if (defaultsURL == null)
         {
            // This really shouldn't happen if Fujaba is set up correctly
            throw new LoggingConfigException ("Could not find [" + defaultConfigFile + "]. Used [" + clazz.getClassLoader() +
               "] class loader in the search.");
         }
         else
         {
            // System.err.println("Default config file found under " + defaultsURL.toString());
            tempConfig = parse (defaultsURL.openStream(), handler);
            // successful = true;
         }
      }
      catch (IOException ioEx)
      {
         throw new LoggingConfigException ("I/O exception encountered whilst trying to load default log4j configuration from " + defaultConfigFile +
            "\nError was:" + ioEx.toString());
      }
      catch (SAXException saxEx)
      {
         throw new LoggingConfigException ("Parsing error whilst loading default log4j configuration from " + defaultConfigFile +
            "\nError was:" + saxEx.toString());
      }

      if (handler.getErrorCount() > 0)
      {
         throw new LoggingConfigException ("Parsing error whilst loading default log4j configuration from " + defaultConfigFile +
            "\nError was:" + handler.getMessage());
         // successful = false;
      }

      // catches any situation where no exception was thrown but we still didn't get
      // a DOM tree
      // if (successful && tempConfig == null)
      if (tempConfig == null)
      {
         throw new LoggingConfigException ("Unidentified error whilst parsing default log4j configuration file " + defaultConfigFile);
         // successful = false;
      }

      // set config up with loaded DOM tree
      log4JConfig = tempConfig;

   } // loadDefaults


   /**
    * Parse <CODE>xmlFile</CODE> to retrieve a <CODE>Document</CODE>
    *
    * @param xmlStream      an open stream to XML data to parse
    * @param handler        the handler the parser should use
    * @return               the DOM of the XML file
    * @throws IOException   if there is an I/O problem whilst opening or reading from the file
    * @throws SAXException  if there is a parsing error
    */
   private Document parse (InputStream xmlStream, ErrorHandler handler) throws IOException, SAXException
   {
      // Using an InputSource so that can set the log4j.dtd location to avoid
      // copying to user dir (which seems a bit silly)

      InputSource source = new InputSource (xmlStream);

      // nicked following code from DOMConfigurator in log4j package
      Class clazz = this.getClass();
      URL dtdURL = clazz.getResource ("/org/apache/log4j/xml/log4j.dtd");
      if (dtdURL == null)
      {
         System.err.println ("Could not find [log4j.dtd]. Used [" + clazz.getClassLoader() +
            "] class loader in the search.");
      }
      else
      {
         // System.err.println("URL to log4j.dtd is [" + dtdURL.toString()+"].");
         source.setSystemId (dtdURL.toString());
      }
      // end of nicked code

      try
      {
         DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
         factory.setValidating (true);
         DocumentBuilder parser = factory.newDocumentBuilder();

         // set handler for parser
         parser.setErrorHandler (handler);

         // Want the parser to validate so that we can check against log4j.dtd
         //parser.setFeature ("http://xml.org/sax/features/validation", true);

         return parser.parse (source);
      }
      catch (ParserConfigurationException pce)
      {
         throw new RuntimeException (pce);
      }
   } // parse


   /**
    * Determines whether the logging configuration has been successfully loaded
    *
    * @return   <CODE>true</CODE> if the configuration has been loaded, <CODE>false</CODE>
    *      otherwise
    */
   public boolean configLoaded()
   {
      return ! (log4JConfig == null);
   }


   // ========= methods for manipulating the DOM ========

   // ========= methods for loggers =========

   /**
    * Update the logger represented by <CODE>info</CODE> according to the information in <CODE>info</CODE>
    *
    * @param info  information about the logger
    */
   public void updateLogger (LoggerInfo info)
   {
      // avoid NPE probs
      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - cannot update logger");
         return;
      }

      String name = info.getName();

      Node loggerNode = getLoggerNode (name);

      // Have also decided to get rid of this as it means that loggers for classes
      // which have not yet been loaded will not show up which could be undesirable for
      // the user if they cannot find the logging setting they wish to change?

      /*
       *  / delete if level is inherited but check no other subnodes first
       *  if (info.getLevel() == LoggerInfo.INHERITED)
       *  {
       *  / if it's not there, nothing to bother about
       *  if (loggerNode == null)
       *  {
       *  return;
       *  }
       *  boolean otherSubnodes = false;
       *  NodeList loggerChildren = loggerNode.getChildNodes();
       *  for (int i = 0; i < loggerChildren.getLength(); i++)
       *  {
       *  Node loggerChild = loggerChildren.item (i);
       *  if (loggerChild.getNodeType() == Node.ELEMENT_NODE
       *  && !loggerChild.getNodeName().equals ("level"))
       *  {
       *  otherSubnodes = true;
       *  break;
       *  }
       *  }
       *  / if there are no other subnodes, delete and we're finished
       *  if (!otherSubnodes)
       *  {
       *  loggerNode.getParentNode().removeChild(loggerNode);
       *  return;
       *  }
       *  / otherwise we need to update the node
       *  }
       */
      if (loggerNode == null)
      {
         try
         {
            List successors = new Vector();
            successors.add ("root");
            successors.add ("categoryFactory");

            Map atts = new Hashtable();
            atts.put ("name", name);

            loggerNode = insertElement (log4JConfig.getDocumentElement(), "logger", atts, successors);
         }
         catch (DOMException domEx)
         {
            System.err.println ("Error thrown whilst trying to create element for logger " + name + ": " + domEx);
            return;
         }
      }

      if (loggerNode != null)
      {
         updateLoggerNode (loggerNode, info);
      }
      else
      {
         System.err.println ("Unidentified error whilst trying to create element for logger " + name);
      }
   }


   /**
    * Delete a logger from the DOM
    *
    * @param name  the name of the logger
    */
   public void deleteLogger (String name)
   {
      // remove from loggers collection
      // LoggerInfo.removeFromLoggers(name);

      // remove from DOM
      Node loggerNode = getLoggerNode (name);

      if (loggerNode == null)
      {
         return;
      }
      else
      {
         // this cannot be root so we know it has a parent
         Node loggerParent = loggerNode.getParentNode();
         loggerParent.removeChild (loggerNode);
      }
   }


   /**
    * Updates the root logger to correspond to <CODE>info</CODE>
    *
    * @param info  information about the root logger
    */
   public void updateRootLogger (LoggerInfo info)
   {
      // avoid NPE probs
      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - cannot update root logger");
         return;
      }

      Element docEl = log4JConfig.getDocumentElement();

      Node rootNode = null;

      NodeList configChildren = docEl.getChildNodes();
      for (int i = 0; i < configChildren.getLength(); i++)
      {
         Node configChild = configChildren.item (i);
         if (configChild.getNodeType() == Node.ELEMENT_NODE
            && configChild.getNodeName().equals ("root"))
         {
            rootNode = configChild;
            // updateLoggerNode (configChild, info);
         }
      }

      if (rootNode == null)
      { // need to think about adding appenders!!

         try
         {
            List successors = new Vector();
            successors.add ("categoryFactory");

            Map atts = new Hashtable();

            rootNode = insertElement (docEl, "root", atts, successors);
         }
         catch (DOMException domEx)
         {
            System.err.println ("Error thrown whilst trying to create element for root logger: " + domEx);
            return;
         }
      }

      if (rootNode != null)
      {
         updateLoggerNode (rootNode, info);
      }
      else
      {
         System.err.println ("Unidentified error whilst trying to create element for root logger");
      }
   }


   /**
    * Gets the node representing a logger of a certain name
    *
    * @param name  the name of the logger
    * @return      the Node representing the logger or null if it is not found
    */
   private Node getLoggerNode (String name)
   {
      // avoid NPE probs
      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - cannot get logger node");
         return null;
      }

      /*
       *  Node loggerNode = (Node) loggerCache.get (name);
       *  if (loggerNode != null)
       *  {
       *  return loggerNode;
       *  }
       *  else
       *  {
       */
      Node loggerNode = null;

      Element docEl = log4JConfig.getDocumentElement();

      NodeList configChildren = docEl.getChildNodes();

      for (int i = 0; i < configChildren.getLength(); i++)
      {
         Node configChild = configChildren.item (i);
         if ( (configChild.getNodeType() == Node.ELEMENT_NODE)
            &&  (configChild.getNodeName().equals ("logger"))
            &&  (subst ( ((Element) configChild).getAttribute ("name")).equals (name)))
         {
            loggerNode = configChild;
            // loggerCache.put(name, loggerNode);
            // found logger so break
            break;
         }
      }

      // if the node wasn't found, this will return null
      return loggerNode;
   }


   /**
    * Updates the level element in <CODE>node</CODE> to correspond to <CODE>info</CODE>
    *
    * @param node  the node to update
    * @param info  corresponding logger info
    */
   private void updateLoggerNode (Node node, LoggerInfo info)
   {
      // avoid NPE probs
      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - cannot update logger node");
         return;
      }

      boolean foundLevel = false;
      NodeList children = node.getChildNodes();
      for (int j = 0; j < children.getLength(); j++)
      {
         Node child = children.item (j);
         if ( (child.getNodeType() == Node.ELEMENT_NODE)
            &&  (child.getNodeName().equals ("level")))
         {
            foundLevel = true;
            // try to update the value attribute to value in logger tree
            try
            {
                ((Element) child).setAttribute ("value", info.getLevel().toLowerCase());
            }
            catch (DOMException domEx)
            {
               System.err.println ("Error thrown whilst trying to update level element for " + info.getName() + ": " + domEx);
            }
         }
      }

      if (!foundLevel)
      {
         Element loggerEl = (Element) node;
         try
         {
            List successors = new Vector();
            successors.add ("appender-ref");

            Map atts = new Hashtable();
            atts.put ("value", info.getLevel().toLowerCase());

            insertElement (loggerEl, "level", atts, successors);
         }
         catch (DOMException domEx)
         {
            System.err.println ("Error thrown whilst trying to create level element for logger " + info.getName() + ": " + domEx);
         }
      }
   }


   /**
    * Insert an element into the DOM tree
    *
    * @param parent         the parent of the new element
    * @param name           the name of the new element
    * @param atts           any attributes the new element should have
    * @param successors     a list of names of elements which may be found after the new element
    *      (to guide positioning)
    * @return               the new element
    * @throws DOMException  if something goes wrong
    */
   private Element insertElement (Element parent, String name, Map atts, List successors) throws DOMException
   {
      // create new element
      Element newEl = log4JConfig.createElement (name);

      // add any required atts
      Iterator attsIter = atts.keySet().iterator();

      while (attsIter.hasNext())
      {
         String attName = (String) attsIter.next();
         String attValue = (String) atts.get (attName);
         newEl.setAttribute (attName, attValue);
      }

      // insert the element before the first element with a name in the
      // successor list, otherwise at the end
      Node followingNode = null;
      NodeList children = parent.getChildNodes();
      for (int i = 0; i < children.getLength(); i++)
      {
         Node child = children.item (i);
         if (child.getNodeType() == Node.ELEMENT_NODE
            && successors.contains (child.getNodeName()))
         {
            followingNode = child;
            break;
         }
      }

      // if followingNode is null, element will be inserted as
      // last child
      parent.insertBefore (newEl, followingNode);

      return newEl;
   }


   /**
    * Gets information about the root logger
    *
    * @return   information about the root logger
    */
   public LoggerInfo getRootLogger()
   {
      // avoid NPE probs
      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - cannot retrieve root logger");
         return null;
      }

      // Have checked and this works ok even when no root specified in
      // log4j-config.xml

      String rootLevel = LoggerInfo.DEBUG; // in case no level specified

      // look for root logger (immediate descendant if it exists)
      NodeList configChildren = log4JConfig.getDocumentElement().getChildNodes();
      for (int i = 0; i < configChildren.getLength(); i++)
      {
         Node configChild = configChildren.item (i);

         if ( (configChild.getNodeType() == Node.ELEMENT_NODE)
            &&  (configChild.getNodeName().equals ("root")))
         {
            // now search the children of the <root> element for
            // a level declaration
            NodeList rootChildren = configChild.getChildNodes();
            for (int j = 0; j < rootChildren.getLength(); j++)
            {
               Node rootChild = rootChildren.item (j);
               // checking if element for same reasons as above
               if (rootChild.getNodeType() == Node.ELEMENT_NODE
                  && rootChild.getNodeName().equals ("level"))
               {

                  // get the value attribute of the level node
                  Element levelEl = (Element) rootChild;
                  rootLevel = subst (levelEl.getAttribute ("value"));

                  break;
               }
            }
            // in case no level specified (which is not necessary according
            // to DTD), still no point in going through rest of doc
            break;
         } // end of [this is a <root>] if-statement

      } // end of [kids of config file document element] for-loop

      return new LoggerInfo ("root", rootLevel);
   }


   /**
    * Get a collection of objects with information on all the loggers contained within the
    * XML file
    *
    * @return   a collection of LoggerInfo objects representing the defined loggers
    */
   public Collection getLoggers()
   {
      // avoid NPE probs
      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - cannot retrieve loggers");
         return null;
      }

      List loggers = new Vector();

      // go through DOM looking for loggers
      NodeList configChildren = log4JConfig.getDocumentElement().getChildNodes();
      for (int i = 0; i < configChildren.getLength(); i++)
      {
         Node configChild = configChildren.item (i);

         if ( (configChild.getNodeType() == Node.ELEMENT_NODE)
            &&  (configChild.getNodeName().equals ("logger")))
         {

            String loggerName = subst ( ((Element) configChild).getAttribute ("name"));
            String loggerLevel = LoggerInfo.INHERITED;
            // in case no level specified
            // NB: We are resetting log4j each
            // time so anything not in the DOM
            // can be assumed to be at default
            // value

            // go through kids of <logger> to find <level> declaration
            NodeList loggerChildren = configChild.getChildNodes();
            for (int j = 0; j < loggerChildren.getLength(); j++)
            {
               Node loggerChild = loggerChildren.item (j);
               if (loggerChild.getNodeType() == Node.ELEMENT_NODE
                  && loggerChild.getNodeName().equals ("level"))
               {
                  Element levelEl = (Element) loggerChild;
                  loggerLevel = subst (levelEl.getAttribute ("value"));

                  // stop iterating through [kids of <logger>] as element
                  // being searched for has been found
                  break;
               }
            } // end of [kids of <logger>] for-loop

            // add logger info object
            loggers.add (new LoggerInfo (loggerName, loggerLevel));

         } // end of it's-a-logger if statement
      } // end of config kids for statement
      return loggers;
   }

   // ============= methods for appenders ==============

   /**
    * Gets the value of the parameter <CODE>param</CODE> for appender <CODE>appender</CODE>
    *
    * @param param     the name of the parameter
    * @param appender  the name of the appender
    * @return          the value of the parameter
    */
   public String getAppenderParam (String param, String appender)
   {
      Node appenderNode = getAppenderNode (appender);
      return getParam (param, appenderNode);
   }


   /**
    * Get the conversion pattern specified for the appender whose settings are contained under
    * <CODE>appenderNode</CODE>
    *
    * @param appenderName  the name of the appender
    * @return              the conversion pattern
    */
   public String getAppenderPattern (String appenderName)
   {
      Node appenderNode = getAppenderNode (appenderName);
      NodeList appenderChildren = appenderNode.getChildNodes();
      for (int i = 0; i < appenderChildren.getLength(); i++)
      {
         Node appenderChild = appenderChildren.item (i);
         if ( (appenderChild.getNodeType() == Node.ELEMENT_NODE)
            && appenderChild.getNodeName().equals ("layout"))
         {
            return getParam ("ConversionPattern", appenderChild);
         }
      }
      return null;
   }


   /**
    * Sets the value of the parameter <CODE>param</CODE> for appender <CODE>appender</CODE>
    * to value <CODE>value</CODE>
    *
    * @param param     the name of the parameter
    * @param value     the value to set the parameter to
    * @param appender  the name of the appender which the parameter is for
    */
   public void setAppenderParam (String param, String value, String appender)
   {
      Node appenderNode = getAppenderNode (appender);
      setParam (param, value, appenderNode);
   }


   /**
    * Set the conversion pattern to be used for an appender
    *
    * @param pattern       the pattern to be used
    * @param appenderName  the name of the appender
    */
   public void setAppenderPattern (String pattern, String appenderName)
   {
      Node appenderNode = getAppenderNode (appenderName);
      NodeList appenderChildren = appenderNode.getChildNodes();
      for (int i = 0; i < appenderChildren.getLength(); i++)
      {
         Node appenderChild = appenderChildren.item (i);
         if ( (appenderChild.getNodeType() == Node.ELEMENT_NODE)
            && appenderChild.getNodeName().equals ("layout"))
         {
            setParam ("ConversionPattern", pattern, appenderChild);
         }
      }
   }


   /**
    * Gets the node in the log4j DOM representing the appender with name <CODE>name</CODE>
    *
    * @param name  the name of the appender
    * @return      the node representing the appender
    */
   private Node getAppenderNode (String name)
   {
      // avoid NPE probs
      if (!configLoaded())
      {
         System.err.println ("log4j configuration settings not loaded correctly - cannot get appender node");
         return null;
      }

      Node appenderNode = (Node) appenderCache.get (name);
      if (appenderNode != null)
      {
         return appenderNode;
      }
      else
      {
         Element docEl = log4JConfig.getDocumentElement();
         NodeList docChildren = docEl.getChildNodes();

         for (int i = 0; i < docChildren.getLength(); i++)
         {
            Node docChild = docChildren.item (i);
            if (docChild.getNodeType() == Node.ELEMENT_NODE
               && docChild.getNodeName().equals ("appender"))
            {
               String appenderName = subst ( ((Element) docChild).getAttribute ("name"));
               if (appenderName.equals (name))
               {
                  appenderCache.put (name, docChild);
                  return docChild;
               }
            }
         }
         System.err.println ("About to return null for an appender - code to deal with this probably hasn't been written yet");
         System.err.println ("If this has happened without manual alteration of the log4j config file and still persists after loading");
         System.err.println ("the default values for logging please mail suzy@upb.de and tell me what you did!");
         System.err.println ("Problems caused by unexpected stuff in log4j config files will be solved soon, code still under construction");
         return null;
      }
   }


   /**
    * Gets the value of the parameter <CODE>name</CODE> under <CODE>node</CODE>
    *
    * @param name  the name of the parameter
    * @param node  the node under which the parameter is to be found
    * @return      the value of the parameter
    */
   private String getParam (String name, Node node)
   {
      NodeList children = node.getChildNodes();
      for (int i = 0; i < children.getLength(); i++)
      {
         Node child = children.item (i);
         if ( (child.getNodeType() == Node.ELEMENT_NODE)
            && child.getNodeName().equals ("param"))
         {
            Element paramEl = (Element) child;
            if (subst (paramEl.getAttribute ("name")).equalsIgnoreCase (name))
            {
               // return paramEl.getAttribute ("value");
               return subst (paramEl.getAttribute ("value"));
            }
         }
      }
      return null;
   }


   /**
    * Set the value of a parameter.
    *
    * @param name   the name of the parameter
    * @param value  the value to set the parameter to
    * @param node   the node under which the parameter is to be found
    */
   private void setParam (String name, String value, Node node)
   {
      NodeList children = node.getChildNodes();
      for (int i = 0; i < children.getLength(); i++)
      {
         Node child = children.item (i);
         if ( (child.getNodeType() == Node.ELEMENT_NODE)
            && child.getNodeName().equals ("param"))
         {
            Element paramEl = (Element) child;
            if (subst (paramEl.getAttribute ("name")).equalsIgnoreCase (name))
            {
               try
               {
                  paramEl.setAttribute ("value", value);
               }
               catch (DOMException domEx)
               {
                  System.err.println ("Error thrown whilst trying to update a parameter element:" + domEx);
               }
            }
         }
      }
   }


   /**
    * Use the log4j method for replacing system properties found in the string
    *
    * @param val  the String object to examine
    * @return     No description provided
    */
   private String subst (String val)
   {
      return OptionConverter.substVars (val, null);
   }
}


/**
 * An error handler for parsing of the log4j config file
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.6.2.3 $
 */
class ParseConfigErrorHandler implements ErrorHandler
{


   /**
    * Number of errors encountered
    */
   private int count = 0;

   /**
    * Error messages concatenated together
    */
   private String msg = "";


   /**
    * Returns the number of errors and fatal errors encountered
    *
    * @return   number of errors
    */
   public int getErrorCount()
   {
      return count;
   }


   /**
    * Returns all the error messages received concatenated together
    *
    * @return   the error messages
    */
   public String getMessage()
   {
      return msg;
   }


   /**
    * Called by the parser when an error is encountered
    *
    * @param e  the exception caused by the error
    */
   public void error (SAXParseException e)
   {
      msg = msg + "[Error on line " + e.getLineNumber() + "]\n";
      msg = msg + e.toString() + "\n\n";
      count++;
      // e.printStackTrace();
   }


   /**
    * Called by the parser when an fatal error is encountered
    *
    * @param e  the exception caused by the fatal error
    */
   public void fatalError (SAXParseException e)
   {
      msg = msg + "[Error on line " + e.getLineNumber() + "]\n";
      msg = msg + e.toString() + "\n\n";
      count++;
      // e.printStackTrace();
   }


   /**
    * Called by the parser when a warning is generated
    *
    * @param e  the exception caused by the warning
    */
   public void warning (SAXParseException e)
   {
      // take warning as ok to ignore
   }
}

/*
 * $Log: LoggingPreferences.java,v $
 * Revision 1.6.2.3  2005/09/30 18:57:11  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
