/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.gui.PEVariable;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.preferences.IconsPreferences;
import de.upb.tools.fca.*;
import de.upb.tools.pcs.CollectionChangeEvent;


/**
 * <h2>Associations</h2> <pre>
 *          +-----------+ 1              1
 * UMLClass | getName() +------------------ UMLAttr
 *          +-----------+ parent     attrs
 *
 *          0..1                                      0..1
 * UMLRole ------------------------------------------------ UMLAttr
 *          implementingAssocRole      associatedAttribute
 *
 *            0..*                            0..1
 * UMLMethod -------------------------------------- UMLAttr
 *            accessMethods      accessedAttribute
 *
 * </pre>
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.243 $
 */
public class UMLAttr extends UMLDeclaration implements FAttr
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLAttr.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String PROPERTY_NAME = "name";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String PROPERTY_UML_STATIC = "umlStatic";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String PROPERTY_ATTR_TYPE = "attrType";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int UNUSED = FDeclaration.PRIVATE - 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int CREATE_ACCESS_METHODS_YES = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int CREATE_ACCESS_METHODS_NO = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int CREATE_ACCESS_METHODS_DEFAULT = 2;


   /**
    * Constructor for class UMLAttr
    */
   public UMLAttr()
   {
      super();
   }


   /**
    * Constructor for class UMLAttr
    *
    * @param coobraPersistent  No description provided
    */
   public UMLAttr (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Constructor for class UMLAttr
    *
    * @param visibility    No description provided
    * @param type          No description provided
    * @param name          No description provided
    * @param initialValue  No description provided
    * @param umlStatic     No description provided
    * @param javaFinal     No description provided
    */
   public UMLAttr (int visibility,
                   String type,
                   String name,
                   String initialValue,
                   boolean umlStatic,
                   boolean javaFinal)
   {
      this (name, umlStatic, javaFinal, visibility, initialValue,
         UMLProject.get().getTypeList().getFromTypes (type),
         null, null);
   }


   /**
    * Constructor for class UMLAttr
    *
    * @param visibility    No description provided
    * @param type          No description provided
    * @param name          No description provided
    * @param initialValue  No description provided
    * @param umlStatic     No description provided
    * @param javaFinal     No description provided
    * @param parent        No description provided
    */
   public UMLAttr (int visibility,
                   String type,
                   String name,
                   String initialValue,
                   boolean umlStatic,
                   boolean javaFinal,
                   UMLClass parent)
   {
      this (name, umlStatic, javaFinal, visibility, initialValue,
         UMLProject.get().getTypeList().getFromTypes (type),
         parent, null);
      if (UMLProject.get().getTypeList().getFromTypes (type) == null)
      {
         setAttrType (type);
      }
   }


   /**
    * Constructor for class UMLAttr
    *
    * @param name          No description provided
    * @param umlStatic     No description provided
    * @param javaFinal     No description provided
    * @param visibility    No description provided
    * @param initialValue  No description provided
    * @param attrType      No description provided
    */
   public UMLAttr (String name,
                   boolean umlStatic,
                   boolean javaFinal,
                   int visibility,
                   String initialValue,
                   UMLType attrType)
   {
      this (name, umlStatic, javaFinal, visibility, initialValue, attrType, null, null);
   }


   /**
    * Constructor for class UMLAttr
    *
    * @param name          No description provided
    * @param umlStatic     No description provided
    * @param javaFinal     No description provided
    * @param visibility    No description provided
    * @param initialValue  No description provided
    * @param attrType      No description provided
    * @param parent        No description provided
    * @param constraint    No description provided
    */
   public UMLAttr (String name,
                   boolean umlStatic,
                   boolean javaFinal,
                   int visibility,
                   String initialValue,
                   UMLType attrType,
                   UMLClass parent,
                   UMLConstraint constraint)
   {
      super();
      setName (name);
      setUmlStatic (umlStatic);
      setFinal (javaFinal);
      setVisibility (visibility);
      setInitialValue (initialValue);
      setAttrType (attrType);
      setParent (parent);
      addToConstraints (constraint);
   }


   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name = null;


   /**
    * Get the name attribute of the UMLAttr object
    *
    * @return   The name value
    */
   public String getName()
   {
      // @@@FIX ME: there should be no need to do this
      if (this.name == null)
      {
         this.name = "_@" + getID().toString();
         if (log.isDebugEnabled())
         {
            log.debug ("create a temp name for a attr");
         }
      }
      return this.name;
   }


   /**
    * Sets the name attribute of the UMLAttr object
    *
    * @param newName  The new name value
    */
   public void setName (String newName)
   {
      if (this.name == null ||
         !this.name.equals (newName))
      {
         UMLClass clazz = this.getParent();
         this.setParent (null);

         String oldName = this.name;
         this.name = newName;

         this.setParent (clazz);

         //----- update all qualified assocs

         firePropertyChange (PROPERTY_NAME, oldName, newName);
      }
   }


   /**
    * Get the nameUpdate attribute of the UMLAttr object
    *
    * @return   The nameUpdate value
    */
   public String getNameUpdate()
   {
      return this.getName();
   }


   /**
    * Sets the nameUpdate attribute of the UMLAttr object
    *
    * @param value  The new nameUpdate value
    */
   public void setNameUpdate (String value)
   {
      this.setName (value);
      PEVariable.handleAccessMethods (null, true, this.getVisibility(), this);
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean pointer;


   /**
    * Get the value of pointer.
    *
    * @return       Value of pointer.
    * @deprecated
    */
   public boolean isPointer()
   {
      return this.pointer;
   }


   /**
    * Set the value of pointer.
    *
    * @param pointer  Value to assign to pointer.
    * @deprecated
    */
   public void setPointer (boolean pointer)
   {
      if (this.pointer != pointer)
      {
         this.pointer = pointer;
         firePropertyChange ("pointer", !pointer, pointer);
      }
   }


   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlStatic = false;


   /**
    * Get the umlStatic attribute of the UMLAttr object
    *
    * @return   The umlStatic value
    */
   public boolean isUmlStatic()
   {
      return umlStatic;
   }


   /**
    * Get the static attribute of the UMLAttr object
    *
    * @return   The static value
    */
   public boolean isStatic()
   {
      return isUmlStatic();
   }


   /**
    * Sets the umlStatic attribute of the UMLAttr object
    *
    * @param umlStatic  The new umlStatic value
    */
   public void setUmlStatic (boolean umlStatic)
   {
      boolean oldValue = this.umlStatic;
      this.umlStatic = umlStatic;

      firePropertyChange (PROPERTY_UML_STATIC, oldValue, umlStatic);
   }


   /**
    * Sets the static attribute of the UMLAttr object
    *
    * @param umlStatic  The new static value
    */
   public void setStatic (boolean umlStatic)
   {
      setUmlStatic (umlStatic);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient boolean syncVisibilities = true;


   /**
    * Sync umlVisibility and visibility. Only needed for old project support
    *
    * @param sync  The new syncVisibilities value
    */
   public static void setSyncVisibilities (boolean sync)
   {
      if (syncVisibilities != sync)
      {
         if (sync)
         {
            fixVisibilities();
         }
         syncVisibilities = sync;
      }
   }


   /**
    * Get the syncVisibilities attribute of the UMLAttr class
    *
    * @return       The syncVisibilities value
    * @deprecated   Move to GUI?
    */
   public static boolean isSyncVisibilities()
   {
      return syncVisibilities;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @deprecated   use {@link #setVisibility} instead
    */
   private transient int umlVisibility = UNUSED;


   /**
    * Get the umlVisibility attribute of the UMLAttr object
    *
    * @return       The umlVisibility value
    * @deprecated   use {@link #getVisibility()} instead. Having visibility and umlVisibility
    *      in parallel was only necessary to handle attrs without access methods
    */
   public int getUmlVisibility()
   {
      if (umlVisibility == UNUSED)
      {
         setUmlVisibility (getVisibility());
      }
      return this.umlVisibility;
   } // getUmlVisibility


   /**
    * Sets the umlVisibility attribute of the UMLAttr object
    *
    * @param newUmlVisibility  The new umlVisibility value
    * @deprecated              use {@link #setVisibility(int)} instead. Having visibility and
    *      umlVisibility in parallel was only necessary to handle attrs without access methods
    */
   public void setUmlVisibility (int newUmlVisibility)
   {
      if (this.umlVisibility != newUmlVisibility)
      {
         int oldValue = this.umlVisibility;
         this.umlVisibility = newUmlVisibility;

         firePropertyChange ("umlVisibility", oldValue, newUmlVisibility);
      }
      if (syncVisibilities && getVisibility() != newUmlVisibility)
      {
         setVisibility (newUmlVisibility);
      }
   } // setUmlVisibility


   /**
    * Sets the visibility attribute of the UMLAttr object
    *
    * @param visibility  The new visibility value
    */
   public void setVisibility (int visibility)
   {
      super.setVisibility (visibility);
      if (syncVisibilities && getUmlVisibility() != visibility)
      {
         setUmlVisibility (visibility);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String initialValue;


   /**
    * Get the initialValue attribute of the UMLAttr object
    *
    * @return   The initialValue value
    */
   public String getInitialValue()
   {
      return initialValue;
   }


   /**
    * Sets the initialValue attribute of the UMLAttr object
    *
    * @param initialValue  The new initialValue value
    */
   public void setInitialValue (String initialValue)
   {
      String oldValue = this.initialValue;
      this.initialValue = initialValue;
      firePropertyChange (INITIAL_VALUE_PROPERTY, oldValue, initialValue);
   }



   /**
    * <pre>
    *          +-----------+ 1              1
    * UMLClass | getName() +------------------ UMLAttr
    *          +-----------+ parent     attrs
    * </pre>
    */
   private transient UMLClass parent;


   /**
    * Sets the parent attribute of the UMLAttr object
    *
    * @param obj  The new parent value
    * @return     No description provided
    */
   public boolean setParent (FClass obj)
   {
      boolean changed = false;

      if (this.parent != obj)
      {
         UMLClass oldValue = this.parent;
         if (this.parent != null)
         {
            this.parent = null;
            oldValue.removeFromAttrs (this);
         }
         this.parent = (UMLClass) obj;
         if (obj != null)
         {
            obj.addToAttrs (this);
         }
         changed = true;

         // side effects

         firePropertyChange (PARENT_PROPERTY, oldValue, obj);
      }

      return changed;
   }


   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the parent attribute of the UMLAttr object
    *
    * @return   The parent value
    */
   public UMLClass getParent()
   {
      return parent;
   }


   /**
    * Get the fParent attribute of the UMLAttr object
    *
    * @return   The fParent value
    */
   public FClass getFParent()
   {
      return getParent();
   }
   //TODO-END

   // ######################################################################

   /**
    * <pre>
    *               n        qualifiedAttr         0..1
    * UMLQualifier ------------------------------------- UMLAttr
    *               revQualifiedAttr      qualifiedAttr
    * </pre>
    */
   private transient FHashSet revQualifiedAttr;


   /**
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInRevQualifiedAttr (FQualifier value)
   {
      return  ( (this.revQualifiedAttr != null) &&
          (value != null) &&
         this.revQualifiedAttr.contains (value));
   }


   /**
    * @return   No description provided
    */
   public Iterator iteratorOfRevQualifiedAttr()
   {
      return  ( (this.revQualifiedAttr == null)
         ? FEmptyIterator.get()
         : this.revQualifiedAttr.iterator());
   }


   /**
    * @return   No description provided
    */
   public int sizeOfRevQualifiedAttr()
   {
      return  ( (this.revQualifiedAttr == null)
         ? 0
         : this.revQualifiedAttr.size());
   }


   /**
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToRevQualifiedAttr (FQualifier value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.revQualifiedAttr == null)
         {
            this.revQualifiedAttr = new FHashSet();

         }
         changed = this.revQualifiedAttr.add (value);
         if (changed)
         {
            value.setQualifiedAttr (this);
         }
      }
      return changed;
   }


   /**
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromRevQualifiedAttr (FQualifier value)
   {
      boolean changed = false;
      if ( (this.revQualifiedAttr != null) &&  (value != null))
      {
         changed = this.revQualifiedAttr.remove (value);
         if (changed)
         {
            value.setQualifiedAttr (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevQualifiedAttr()
   {
      UMLQualifier tmpValue;
      Iterator iter = this.iteratorOfRevQualifiedAttr();
      while (iter.hasNext())
      {
         tmpValue = (UMLQualifier) iter.next();
         this.removeFromRevQualifiedAttr (tmpValue);
      }

   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLType attrType;

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the attrType attribute of the UMLAttr object
    *
    * @return   The attrType value
    */
   public UMLType getAttrType()
   {
      return attrType;
   }


   /**
    * Get the fAttrType attribute of the UMLAttr object
    *
    * @return   The fAttrType value
    */
   public FType getFAttrType()
   {
      return getAttrType();
   }
   // TODO-END

   /**
    * Get the uMLType attribute of the UMLAttr object
    *
    * @return   The uMLType value
    */
   public UMLType getUMLType()
   {
      return getAttrType();
   }


   /**
    * Sets the attrType attribute of the UMLAttr object
    *
    * @param attrType  The new attrType value
    */
   public void setAttrType (FType attrType)
   {
      if (this.attrType != attrType)
      { // new partner

         UMLType oldAttrType = this.attrType;
         if (this.attrType != null)
         { // inform old partner

            this.attrType = null;
            oldAttrType.removeFromRevAttrType (this);
         }
         this.attrType = (UMLType) attrType;
         if (attrType != null)
         { // inform new partner

             ((UMLType) attrType).addToRevAttrType (this);
         }

         firePropertyChange (PROPERTY_ATTR_TYPE, oldAttrType, attrType);
      }
   }


   /**
    * Sets the attribute type by its name. Searches first in the type list and then in the
    * reference list. If no type with this name is found, a new type will be created.
    *
    * @param tmpType  The (full qualified) attribute type as string.
    */
   public void setAttrType (String tmpType)
   {
      UMLType umlType = null;

      String type = UMLProject.getTypeOfFullQualifiedType (tmpType);

      if (UMLProject.get().getTypeList().hasKeyInTypes (type))
      {
         // Get the type from the projects type list.
         umlType = UMLProject.get().getTypeList().getFromTypes (type);
      }
      else if (UMLProject.get().hasKeyInReferences (tmpType))
      {
         // Get the type from the projects reference list.
         umlType = UMLProject.get().getFromReferences (tmpType);
      }
      else
      {
         // Create a new reference type.
         umlType = UMLProject.get().getOrNewType (this.getParent(), tmpType);
          ((UMLClass) umlType).setDisplayLevel (FDeclaration.CODE_DISPLAY_LEVEL);
      }

      setAttrType (umlType);
   } // setAttrType



   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet instances = new FPropTreeSet (this, "instances");


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInInstances (UMLAttrExprPair elem)
   {
      return this.instances.contains (elem);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfInstances()
   {
      return new EnumerationForAnIterator (iteratorOfInstances());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfInstances()
   {
      return instances.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToInstances (UMLAttrExprPair elem)
   {
      if (elem != null && !this.hasInInstances (elem))
      {
         this.instances.add (elem);
         elem.setInstanceOf (this);
         firePropertyChange (CollectionChangeEvent.get (this, "instances", this.instances, null, elem, CollectionChangeEvent.ADDED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromInstances (UMLAttrExprPair elem)
   {
      if (this.hasInInstances (elem))
      {
         this.instances.remove (elem);
         elem.setInstanceOf (null);
         firePropertyChange (CollectionChangeEvent.get (this, "instances", this.instances, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromInstances()
   {
      UMLAttrExprPair item;
      Iterator iter = iteratorOfInstances();

      while (iter.hasNext())
      {
         item = (UMLAttrExprPair) iter.next();
         item.setInstanceOf (null);
         firePropertyChange (CollectionChangeEvent.get (this, "instances", this.instances, item,
            null, CollectionChangeEvent.REMOVED));
      }
   }


   /**
    * Get the visibilityType attribute of the UMLAttr class
    *
    * @param visibility  No description provided
    * @return            The visibilityType value
    */
   public static String getVisibilityType (int visibility)
   {
      String visibilityType = null;

      switch (visibility)
      {
         case FDeclaration.PRIVATE:
            visibilityType = IconsPreferences.PRIV_FIELD;
            break;
         case FDeclaration.PROTECTED:
            visibilityType = IconsPreferences.PROT_FIELD;
            break;
         case FDeclaration.PACKAGE:
            visibilityType = IconsPreferences.PACK_FIELD;
            break;
         case FDeclaration.PUBLIC:
            visibilityType = IconsPreferences.PUB_FIELD;
            break;
         case FDeclaration.USERDEFINED:
            visibilityType = IconsPreferences.USER_FIELD;
            break;
      }

      return visibilityType;
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      setParent (null);
      removeAllFromRevQualifiedAttr();
      removeAllFromInstances();

      setAttrType ((UMLType) null);

      UMLRole tmpImplementingAssocRole = getImplementingAssocRole();
      if (tmpImplementingAssocRole != null)
      {
         setImplementingAssocRole (null);
      } // if
      removeAllFromAccessMethods();

      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return getParent();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param data             No description provided
    * @param setOfNeighbours  No description provided
    */
   public void writeAttributes (StringBuffer data, FTreeSet setOfNeighbours)
   {
      super.writeAttributes (data, setOfNeighbours);
      // This is a hack: attrType is not be saved by standard save mechanism
      // therefore it is marked transient.
      if (attrType instanceof UMLBaseTypes)
      {
         writeToStringBuffer (ATTR_TYPE_PROPERTY, (UMLBaseTypes) attrType, setOfNeighbours);
      }
      else if (attrType instanceof UMLClass)
      {
         writeToStringBuffer (ATTR_TYPE_PROPERTY, (UMLClass) attrType, setOfNeighbours);
      }
      else
      {
         writeToStringBuffer (ATTR_TYPE_PROPERTY, (UMLArray) attrType, setOfNeighbours);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param objects     No description provided
    * @param attributes  No description provided
    */
   public void readAttributes (Hashtable objects, FDuplicatedTreeMap attributes)
   {
      super.readAttributes (objects, attributes);
      // this is a hack: see above
      if (attrType instanceof UMLBaseTypes)
      {
         setAttrType ((UMLType) readFromStringTokenizer
             (ATTR_TYPE_PROPERTY, (UMLBaseTypes) attrType, objects, attributes));
      }
      else if (attrType instanceof UMLClass)
      {
         setAttrType ((UMLType) readFromStringTokenizer
             (ATTR_TYPE_PROPERTY, (UMLClass) attrType, objects, attributes));
      }
      else
      {
         setAttrType ((UMLType) readFromStringTokenizer
             (ATTR_TYPE_PROPERTY, (UMLArray) attrType, objects, attributes));
      }
   }


   /**
    * Get the text attribute of the UMLAttr object
    *
    * @return   The text value
    */
   public String getText()
   {
      StringBuffer buffer = new StringBuffer();

      buffer.append (FDeclaration.VISIBILITY_CHAR[getVisibility()]);
      buffer.append (" ");

      buffer.append (getName());
      buffer.append (" : ");
      buffer.append ( (getAttrType() == null)
         ? "<unknown>"
         : getAttrType().getName());

      if ( (getInitialValue() != null) && !getInitialValue().equals (""))
      {
         buffer.append (" = ");
         buffer.append (getInitialValue());
      }

      return buffer.toString();
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      StringBuffer result = new StringBuffer();

      result.append ("UMLAttr[");
      result.append (getText());
      result.append ("]");

      return result.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int createAccessMethods = CREATE_ACCESS_METHODS_DEFAULT;


   /**
    * Get the createAccessMethods attribute of the UMLAttr object
    *
    * @return   The createAccessMethods value
    */
   public int getCreateAccessMethods()
   {
      return this.createAccessMethods;
   }


   /**
    * Sets the createAccessMethods attribute of the UMLAttr object
    *
    * @param createAccessMethods  The new createAccessMethods value
    */
   public void setCreateAccessMethods (int createAccessMethods)
   {
      if (this.createAccessMethods != createAccessMethods)
      {
         int oldValue = this.createAccessMethods;
         boolean oldNoAccessMethods = isNoAccessMethods();

         this.createAccessMethods = createAccessMethods;
         firePropertyChange ("createAccessMethods", oldValue, createAccessMethods);

         if (isNoAccessMethods() != oldNoAccessMethods)
         {
            firePropertyChange ("noAccessMethods", oldNoAccessMethods, !oldNoAccessMethods);
         }
      }
   }


   /**
    * Sets the noAccessMethods attribute of the UMLAttr object
    *
    * @param noAccessMethods  The new noAccessMethods value
    * @deprecated             use {@link #getCreateAccessMethods()} instead
    */
   public void setNoAccessMethods (boolean noAccessMethods)
   {
      setCreateAccessMethods ( (noAccessMethods ? CREATE_ACCESS_METHODS_NO : CREATE_ACCESS_METHODS_DEFAULT));
   }


   /**
    * Get the noAccessMethods attribute of the UMLAttr object
    *
    * @return       The noAccessMethods value
    * @deprecated   use {@link #getCreateAccessMethods()} instead
    */
   public boolean isNoAccessMethods()
   {
      return  (getCreateAccessMethods() == CREATE_ACCESS_METHODS_NO);
   }


   /**
    * Get the needsAccessMethods attribute of the UMLAttr object
    *
    * @return   The needsAccessMethods value
    */
   public boolean isNeedsAccessMethods()
   {
      switch (getCreateAccessMethods())
      {
         case CREATE_ACCESS_METHODS_NO:
            return false;
         case CREATE_ACCESS_METHODS_DEFAULT:
            return !isConstant();
         default:
            return true;
      }
   }

   // Java specific properties
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean javaFinal = false;


   /**
    * Get the final attribute of the UMLAttr object
    *
    * @return   The final value
    */
   public boolean isFinal()
   {
      return this.javaFinal;
   } // isUmlFinal


   /**
    * Sets the final attribute of the UMLAttr object
    *
    * @param javaFinal  The new final value
    */
   public void setFinal (boolean javaFinal)
   {
      boolean oldValue = this.javaFinal;
      this.javaFinal = javaFinal;

      firePropertyChange ("final", oldValue, javaFinal);
   } // setFinal


   /**
    * Get the constant attribute of the UMLAttr object
    *
    * @return   The constant value
    */
   public boolean isConstant()
   {
      return isStatic() && isFinal();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean javaTransient = false;


   /**
    * Sets the transient attribute of the UMLAttr object
    *
    * @param javaTransient  The new transient value
    */
   public void setTransient (boolean javaTransient)
   {
      boolean oldValue = this.javaTransient;
      this.javaTransient = javaTransient;

      firePropertyChange ("transient", oldValue, javaTransient);
   }


   /**
    * Get the transient attribute of the UMLAttr object
    *
    * @return   The transient value
    */
   public boolean isTransient()
   {
      return javaTransient;
   }


   /**
    * Hangs the current ASGElement into the ASG-tree. Needed for cut'n'paste.
    *
    * @param parent  Where to put the current ASGElement.
    */
   public void setCutCopyPasteParent (FElement parent)
   {
      if (parent instanceof UMLClass)
      {
          ((UMLClass) parent).addToAttrs (this);
      }
      else if (parent instanceof UMLAttr)
      {
         UMLAttr attr = (UMLAttr) parent;
         if (attr.getParent() != null)
         {
            attr.getParent().addToAttrs (this);
         }
      }
   }


   /**
    * Search for an associated access method of the specified type.
    *
    * @param type  The type of the access method.
    * @return      If existent, the associated access method of the specified type, otherwise
    *      null.
    */
   public UMLMethod findAccessMethod (int type)
   {
      return null;
   }


   /**
    * Check whether this attribute has associated access methods or not.
    *
    * @return   True, if this attribute has at least one access method, otherwise false.
    */
   public boolean hasAccessMethods()
   {
      return sizeOfAccessMethods() != 0;
   }


   /**
    * Check whether this attribute has an associated access method of the specified type.
    *
    * @param type  No description provided
    * @return      True, if this attribute has an access method of the specified type.
    */
   public boolean hasAccessMethod (int type)
   {
      return findAccessMethod (type) != null;
   }


   /**
    * <pre>
    *          0..1                Assoc                 0..1
    * UMLRole ------------------------------------------------ UMLAttr
    *          implementingAssocRole      associatedAttribute
    * </pre>
    */
   private transient UMLRole implementingAssocRole;


   /**
    * UMLMethod: '+ setImplementingAssocRole (value: UMLRole): Boolean'.
    *
    * @param value  The new implementingAssocRole value
    * @return       No description provided
    */
   public boolean setImplementingAssocRole (FRole value)
   {
      boolean changed = false;

      if (this.implementingAssocRole != value)
      {
         UMLRole oldValue = this.implementingAssocRole;
         if (this.implementingAssocRole != null)
         {
            this.implementingAssocRole = null;
            oldValue.setAssociatedAttribute (null);
         }

         this.implementingAssocRole = (UMLRole) value;
         firePropertyChange ("implementingAssocRole", oldValue, value);

         if (value != null)
         {
             ((UMLRole) value).setAssociatedAttribute (this);
         }
         changed = true;
      }

      return changed;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * UMLMethod: '+ getImplementingAssocRole (): UMLRole'.
    *
    * @return   The implementingAssocRole value
    */
   public UMLRole getImplementingAssocRole()
   {
      return this.implementingAssocRole;
   }


   /**
    * Get the fImplementingAssocRole attribute of the UMLAttr object
    *
    * @return   The fImplementingAssocRole value
    */
   public FRole getFImplementingAssocRole()
   {
      return getImplementingAssocRole();
   }
   // TODO-END

   /**
    * <pre>
    *            0..*           Assoc            0..1
    * UMLMethod -------------------------------------- UMLAttr
    *            accessMethods      accessedAttribute
    * </pre>
    */
   private FHashSet accessMethods;


   /**
    * UMLMethod: '+ hasInAccessMethods (value: UMLMethod): Boolean'.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInAccessMethods (UMLMethod value)
   {
      return  ( (this.accessMethods != null) &&
          (value != null) &&
         this.accessMethods.contains (value));
   }


   /**
    * UMLMethod: '+ iteratorOfAccessMethods (): Iterator'.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAccessMethods()
   {
      return  ( (this.accessMethods == null)
         ? FEmptyIterator.get()
         : this.accessMethods.iterator());
   }


   /**
    * UMLMethod: '+ sizeOfAccessMethods (): Integer'.
    *
    * @return   No description provided
    */
   public int sizeOfAccessMethods()
   {
      return  ( (this.accessMethods == null)
         ? 0
         : this.accessMethods.size());
   }


   /**
    * UMLMethod: '+ addToAccessMethods (value: UMLMethod): Boolean'.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToAccessMethods (UMLMethod value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.accessMethods == null)
         {
            this.accessMethods = new FPropHashSet (this, "accessMethods");
         }

         changed = this.accessMethods.add (value);

         if (changed)
         {
            value.setAccessedAttribute (this);
         }
      }

      return changed;
   }


   /**
    * UMLMethod: '+ removeFromAccessMethods (value: UMLMethod): Boolean'.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromAccessMethods (UMLMethod value)
   {
      boolean changed = false;

      if ( (this.accessMethods != null) &&  (value != null))
      {
         changed = this.accessMethods.remove (value);

         if (changed)
         {
            value.setAccessedAttribute (null);
         }
      }

      return changed;
   }


   /**
    * UMLMethod: '+ removeAllFromAccessMethods (): Void'.
    */
   public void removeAllFromAccessMethods()
   {
      UMLMethod tmpValue;

      Iterator iter = this.iteratorOfAccessMethods();
      while (iter.hasNext())
      {
         tmpValue = (UMLMethod) iter.next();
         this.removeFromAccessMethods (tmpValue);
         tmpValue.removeYou();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public static void fixVisibilities()
   {
      UMLProject project = UMLProject.get();
      fixVisibilities (project);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param project  No description provided
    */
   private static void fixVisibilities (UMLProject project)
   {
      fixVisibilities (project.getRootPackage());
      Iterator references = project.iteratorOfReferences();
      while (references.hasNext())
      {
         UMLClass umlClass = (UMLClass) references.next();
         fixVisibilities (umlClass);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlPackage  No description provided
    */
   private static void fixVisibilities (UMLPackage umlPackage)
   {
      Iterator classes = umlPackage.iteratorOfDeclares();
      while (classes.hasNext())
      {
         UMLClass umlClass = (UMLClass) classes.next();
         fixVisibilities (umlClass);
      }
      Iterator packages = umlPackage.iteratorOfPackages();
      while (packages.hasNext())
      {
         UMLPackage tmpPackage = (UMLPackage) packages.next();
         fixVisibilities (tmpPackage);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlClass  No description provided
    */
   private static void fixVisibilities (UMLClass umlClass)
   {
      Iterator attrs = umlClass.iteratorOfAttrs();
      while (attrs.hasNext())
      {
         UMLAttr umlAttr = (UMLAttr) attrs.next();
         fixVisibility (umlAttr);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlAttr  No description provided
    */
   private static void fixVisibility (UMLAttr umlAttr)
   {
      int umlVisibility = umlAttr.getUmlVisibility();

      if (umlVisibility == FDeclaration.USERDEFINED)
      {
         umlAttr.setUmlVisibility (umlAttr.getVisibility());
      }
      else
      {
         umlAttr.setVisibility (umlAttr.getUmlVisibility());
      }
   }

}

/*
 * $Log: UMLAttr.java,v $
 * Revision 1.243  2005/02/14 14:28:19  cschneid
 * introduced message view, getParentElement() for FElement, code generation and compiler messages in message view
 *
 */
