/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.Iterator;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.metamodel.FClass;
import de.uni_paderborn.fujaba.metamodel.FClassDiagram;
import de.upb.tools.fca.FLinkedList;


/**
 * @author    $Author: fklar $
 * @version   $Revision: 1.80 $
 */
public class UMLClassDiagram extends UMLDiagram implements FClassDiagram
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLClassDiagram.class);


   /**
    * Constructor for class UMLClassDiagram
    */
   public UMLClassDiagram()
   {
      super();
   }


   /**
    * Constructor for class UMLClassDiagram
    *
    * @param name     No description provided
    * @param project  No description provided
    */
   public UMLClassDiagram (String name, UMLProject project)
   {
      super (name, project);
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Returns a class with the full qualified name <code>className</code>.
    * If the class can not be found, a new one is created within
    * the given package. If <code>packageName</code> is empty or null,
    * the default package will be used.
    *
    * @param className    Full qualified class name.
    * @param reuse        If true, method tries to reuse existing classes
    * 						from another class diagram.
    * @param packageName  Name of the package className should be placed into.
    *
    * @return             If successful a UMLClass, null otherwise.
    */
   public UMLClass provideClass (String className, String packageName, boolean reuse)
   {
      if (className == null)
      {
         return null;
      }

      if (packageName == null)
      {
         packageName = "";
      }

      UMLClass providedClass = null;
      UMLProject theProject = UMLProject.get();

      if (reuse)
      {
         if (packageName.length() == 0)
         {
            providedClass = theProject.getFromClasses (className);
         }
         else
         {
            providedClass = theProject.getFromClasses (packageName + "." + className);
         }

         if (providedClass == null)
         {
            // get package this class should be placed into
            // if this function fails the class automatically will be placed in the default-package
            final UMLPackage umlPackage = theProject.getNewFromPackages (packageName);

            // use this constructor to make shure that the full-classname is used
            // as type-identifier in UMLTypeList
            providedClass = new UMLClass (className, null, UMLProject.get().getTypeList(), umlPackage);
         }

         if (!hasInElements (providedClass))
         {
            addToElements (providedClass);
         }
      }
      else
      {
         providedClass = getFromClasses (className);
         if (providedClass == null)
         {
            // get package this class should be placed into
            // if this function fails the class automatically will be placed in the default-package
            final UMLPackage umlPackage = theProject.getNewFromPackages (packageName);

            // use this constructor to make shure that the full-classname is used
            // as type-identifier in UMLTypeList
            providedClass = new UMLClass (className, null, UMLProject.get().getTypeList(), umlPackage);

            addToElements (providedClass);
         }
      }

      return providedClass;
   } // provideClass


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClassDiagram#provideFClass(java.lang.String,
    *  java.lang.String, boolean)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param className    No description provided
    * @param packageName  No description provided
    * @param reuse        No description provided
    * @return             No description provided
    */
   public FClass provideFClass (String className, String packageName, boolean reuse)
   {
      return provideClass (className, packageName, reuse);
   }

   // TODO-END

   /**
    * Does the specified class exists in this diagram?
    *
    * @param className  The name of the class to be searched for.
    * @return           A boolean value specifying whether the class exists in this
    *         diagram.
    */
   public boolean hasInClasses (String className)
   {
      return  (getFromClasses (className) != null);
   }


   /**
    * Does the specified class exists in this diagram?
    *
    * @param className    The name of the class to be searched for.
    * @param packageName  The package in which the class is located.
    * @return             A boolean value specifying whether the class exists in this
    *         diagram.
    */
   public boolean hasInClasses (String className, String packageName)
   {
      return  (getFromClasses (className, packageName) != null);
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the first class in this diagram that matches the given name. NOTE
    * that there could be multiple classes with the same name which are in
    * different packages. If you want to get a class that is located in special
    * package use UMLClassDiagram#getFromClasses(java.lang.String,
    * java.lang.String) instead.
    *
    * @param className  The name of the class to be searched for
    * @return           If the specified class exists in this diagram it is returned,
    *         otherwise null is returned.
    */
   public UMLClass getFromClasses (String className)
   {
      UMLClass result = null;
      ASGElement tmpElement;

      Iterator iter = iteratorOfElements();
      while (iter.hasNext() && result == null)
      {
         tmpElement = (ASGElement) iter.next();

         if (tmpElement instanceof UMLClass && tmpElement.getName().equals (className))
         {
            result = (UMLClass) tmpElement;
         }
      }
      return result;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FClassDiagram#getFromFClasses(java.lang.String)
    */
   /**
    * Get the fromFClasses attribute of the UMLClassDiagram object
    *
    * @param className  No description provided
    * @return           The fromFClasses value
    */
   public FClass getFromFClasses (String className)
   {
      return getFromClasses (className);
   }


   /**
    * Get the specified class from this diagram, including the classes
    * packagename.
    *
    * @param className    Classname to search for.
    * @param packageName  Name of the package in which the class is located.
    * @return             If the specified class exists in this diagram it is returned,
    *         otherwise null is returned.
    */
   public UMLClass getFromClasses (String className, String packageName)
   {
      if (packageName.length() == 0)
      {
         return getFromClasses (className);
      }

      UMLClass result = null;
      ASGElement tmpElement;
      String fullClassNameToBeMatched = packageName + "." + className;
      String fullClassName;

      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         tmpElement = (ASGElement) iter.next();
         if (tmpElement instanceof UMLClass)
         {
            result = (UMLClass) tmpElement;
            fullClassName = result.getFullClassName();
            if (fullClassName.equals (fullClassNameToBeMatched))
            {
               return result;
            }
         }
      }
      return null;
   }

   // TODO-END

   /**
    * Call this function to get all contained classes of the diagram.
    *
    * @return   An array of all classes, which are displayed on the diagram.
    */
   public FLinkedList getAllClassItems()
   {
      FLinkedList classes = new FLinkedList();

      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         ASGElement tmpElement = (ASGElement) iter.next();
         if (tmpElement instanceof UMLClass)
         {
            classes.add (tmpElement);
         }
      }

      return classes;
   } // getAllClassItems


   /**
    * Deletes all unbound classes which are declared as referenced but aren't
    * referenced by any other class in the diagram.
    */
   public void deleteUnboundReferencedClasses()
   {
      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         ASGElement tmpElement = (ASGElement) iter.next();
         if (tmpElement instanceof UMLClass)
         {
            UMLClass clazz = (UMLClass) tmpElement;

            if (clazz.isReference())
            {
               // check all elementsOf..., but not attrs and methods
               if ( (!clazz.elementsOfInstances().hasMoreElements()) &&  (clazz.sizeOfRevAttrType() == 0)
                  &&  (!clazz.elementsOfRevImportedClasses().hasMoreElements())
                  &&  (clazz.sizeOfRevParamType() == 0) &&  (clazz.sizeOfRevResultType() == 0)
                  &&  (!clazz.elementsOfRevSubclass().hasMoreElements())
                  &&  (!clazz.elementsOfRevSuperclass().hasMoreElements())
                  &&  (!clazz.elementsOfRoles().hasMoreElements()))
               {
                  if (log.isDebugEnabled())
                  {
                     log.debug ("Unbound referenced class deleted: " + clazz.getText());
                  }
                  clazz.removeYou();
               }
            }
         }
      }
   } // deleteUnboundReferencedClasses


   /**
    * Returns a correct name for the tree.
    *
    * @return   No description provided
    */
   public String toString()
   {
      return getName();
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      super.removeYou();
   }
}

/*
 * $Log: UMLClassDiagram.java,v $
 * Revision 1.80  2005/03/11 17:41:18  fklar
 * added null-pointer-check to provideClass-method
 *
 */
