/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.lang.ref.SoftReference;
import java.util.Iterator;
import java.util.Set;

import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.FujabaDebug;
import de.uni_paderborn.fujaba.basic.KeyValuePair;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.parser.Parser;
import de.uni_paderborn.fujaba.parser.ParserManager;
import de.uni_paderborn.fujaba.preferences.IconsPreferences;
import de.upb.tools.fca.*;


/**
 * <h2>Associatons</h2>
 *
 * <pre>
 *            1                    1 +---------------------+
 * UMLMethod ------------------------+ getFullMethodName() | UMLClass
 *            methods         parent +---------------------+
 *
 *          +-----------+ 1                          1
 * UMLClass | getName() +------------------------------ UMLMethod
 *          +-----------+ declares    declaredInMethod
 *
 *           0..1                           0..*
 * UMLAttr -------------------------------------- UMLMethod
 *          accessedAttribute      accessMethods
 *
 *            0..1    astRootNode     0..1
 * UMLMethod ------------------------------ ASTRootNode
 *            uMLMethod        aSTRootNode
 *
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.380.2.5 $
 */
public class UMLMethod extends UMLDeclaration implements FMethod
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLMethod.class);


   /**
    * Constructor for class UMLMethod
    */
   public UMLMethod()
   {
      super();
   }


   /**
    * Constructor for class UMLMethod
    *
    * @param coobraPersistent  No description provided
    */
   public UMLMethod (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Constructor for class UMLMethod
    *
    * @param name  No description provided
    */
   public UMLMethod (String name)
   {
      this (name, false, FDeclaration.PUBLIC, null, null, null);
   }


   /**
    * Constructor for class UMLMethod
    *
    * @param name        No description provided
    * @param umlStatic   No description provided
    * @param visibility  No description provided
    * @param resultType  No description provided
    */
   public UMLMethod (String name, boolean umlStatic,
                     int visibility, UMLType resultType)
   {
      this (name, umlStatic, visibility, null, resultType, null);
   }


   /**
    * Constructor for class UMLMethod
    *
    * @param name        No description provided
    * @param umlStatic   No description provided
    * @param visibility  No description provided
    * @param constraint  No description provided
    * @param resultType  No description provided
    * @param revSpec     No description provided
    */
   public UMLMethod (String name, boolean umlStatic,
                     int visibility, UMLConstraint constraint,
                     UMLType resultType, UMLStartActivity revSpec)
   {
      super();
      setName (name);
      setStatic (umlStatic);
      setFinal (false);
      setAbstract (false);
      setSynchronized (false);
      setVisibility (visibility);
      addToConstraints (constraint);
      setResultType (resultType);
      setRevSpec (revSpec);
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      StringBuffer result = new StringBuffer();

      result.append ("UMLMethod[");
      result.append (getFullMethodName());
      result.append ("]");

      return result.toString();
   }


   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name = null;


   /**
    * Get the name attribute of the UMLMethod object
    *
    * @return   The name value
    */
   public String getName()
   {
      // To be added to a class an attr must have a valid name.
      // Thus, I just create an internal one.
      // This one should not become visible at the GUI. AZ.
      if (this.name == null)
      {
         this.name = "_@" + getID().toString();
         if (log.isDebugEnabled())
         {
            log.debug ("create a temp name for a method");
         }
      }
      return name;
   }


   /**
    * Sets the name attribute of the UMLMethod object
    *
    * @param name  The new name value
    */
   public void setName (String name)
   {
      if (this.name == null ||
         this.name.equals (name) == false)
      {
         String oldValue = this.name;
         this.name = name;

         firePropertyChange (NAME_PROPERTY, oldValue, name);

         //----- update all qualified assoccs
         this.updateAssocOverFullMethodName();
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient String cacheForFullMethodName = null;


   /**
    * this method is used in UMLClass.addToMethod( Map.Entry ) to set the key under which this
    * mehtod is stored in the assoc
    *
    * @param key  The new fullMethodName value
    */
   /*
    *  package
    */
   void setFullMethodName (String key)
   {
      if (key == null || !key.equals (this.cacheForFullMethodName))
      {
         this.cacheForFullMethodName = key;
      }
   }


   /**
    * This function assembles the full name of the method out of the name and the signature.
    * e. g.: getFullMethodName(), setName(String),
    *
    * @return   The fullMethodName value
    */
   public String getFullMethodName()
   {
      if (cacheForFullMethodName == null || !cacheForFullMethodName.startsWith (this.name))
      {
         updateFullMethodName();
      }

      return cacheForFullMethodName;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name             No description provided
    * @param iteratorOfParam  No description provided
    * @return                 No description provided
    */
   public static String constructFullMethodName (String name, Iterator iteratorOfParam)
   {
      StringBuffer buffer = new StringBuffer();
      UMLParam param;
      UMLType type;

      buffer.append (name);
      buffer.append ("(");

      boolean setComma = false;
      while (iteratorOfParam.hasNext())
      {
         param = (UMLParam) iteratorOfParam.next();
         type = param.getParamType();
         if (type != null && type.getName() != null)
         {
            if (setComma == true)
            {
               buffer.append (',');
            }
            setComma = true;
            buffer.append (type.getName());
         }
         else
         {
            Logger log = Logger.getLogger (UMLMethod.class);
            if (log.isEnabledFor (Priority.WARN))
            {
               log.warn ("null type in method parameter (" + name + ")");
            }
         }
      }
      buffer.append (")");
      return buffer.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void updateFullMethodName()
   {
      cacheForFullMethodName = constructFullMethodName (getName(), iteratorOfParam());
   }


   /*
    *  package
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void updateAssocOverFullMethodName()
   {
      if (!UMLProject.isLoading())
      {
         UMLClass myClass = this.getParent();

         this.setParent (null);

         cacheForFullMethodName = null;

         this.setParent (myClass);
      }
   }


   /**
    * Get the text attribute of the UMLMethod object
    *
    * @return   The text value
    */
   public String getText()
   {
      String result = "";
      if (getResultType() != null && UMLBaseTypes.INITIALIZER.equals (getResultType().getName()))
      {
         result = FDeclaration.VISIBILITY_CHAR[getVisibility()] + " Initializer";
      }
      else
         if (getResultType() != null && UMLBaseTypes.CONSTRUCTOR.equals (getResultType().getName()))
      {
         result = FDeclaration.VISIBILITY_CHAR[getVisibility()] + " " + getMethodDeclWithoutResultType();
      }
      else
      {
         result = FDeclaration.VISIBILITY_CHAR[getVisibility()] + " " + getMethodDecl();
      }
      return result;
   }


   /**
    * Get the qualifiedMethodDecl attribute of the UMLMethod object
    *
    * @return   The qualifiedMethodDecl value
    */
   public String getQualifiedMethodDecl()
   {
      return this.getParent() + "::" + this.getMethodDecl();
   }


   /**
    * Sets the qualifiedMethodDecl attribute of the UMLMethod object
    *
    * @param str  The new qualifiedMethodDecl value
    */
   public void setQualifiedMethodDecl (String str) { }


   /**
    * Get the methodDecl attribute of the UMLMethod object
    *
    * @return   The methodDecl value
    */
   private String getMethodDeclWithoutResultType()
   {
      int count = 0;

      String result = getName() + " (";
      Iterator iter = iteratorOfParam();
      UMLParam currentParam = null;

      while (iter.hasNext())
      {
         count++;
         if (count > 1)
         {
            result += ", ";
         }
         currentParam = (UMLParam) iter.next();
         if ( (currentParam != null) &&  (currentParam.getFParamType() != null))
         {
            result += currentParam.getName() + ": "
               + currentParam.getFParamType().getName();
         }
      }

      result += ")";

      return result;
   }


   /**
    * Get the methodDecl attribute of the UMLMethod object
    *
    * @return   The methodDecl value
    */
   public String getMethodDecl()
   {
      String result = getMethodDeclWithoutResultType();
      if (getResultType() != null)
      {
         result += ": " + getResultType().getName();
      }

      return result;
   }


   /**
    * Get the java attribute of the UMLMethod object
    *
    * @return   The java value
    */
   public String getJava()
   {
      String result = FDeclaration.VISIBILITY_STRING[getVisibility()] + " ";
      if (getResultType() != null)
      {
         result += getResultType().getName() + " ";
      }
      result += getName() + " (";
      UMLParam parameter = null;
      Iterator iter = iteratorOfParam();

      while (iter.hasNext())
      {
         parameter = (UMLParam) iter.next();
         result += parameter.getFParamType().getName() + " " + parameter.getName();
         if (iter.hasNext())
         {
            result += ", ";
         }
      }
      result += ");";
      return result;
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlStatic;


   /**
    * Get the static attribute of the UMLMethod object
    *
    * @return   The static value
    */
   public boolean isStatic()
   {
      return umlStatic;
   }


   /**
    * Get the umlStatic attribute of the UMLMethod object
    *
    * @return   The umlStatic value
    */
   public boolean isUmlStatic()
   {
      return  (this.isStatic());
   }


   /**
    * Sets the static attribute of the UMLMethod object
    *
    * @param umlStatic  The new static value
    */
   public void setStatic (boolean umlStatic)
   {
      boolean oldValue = this.umlStatic;
      this.umlStatic = umlStatic;
      firePropertyChange ("umlStatic", oldValue, umlStatic);

   }


   /**
    * Wrapper to meet the style guides. Toggles the static flag.
    *
    * @param umlStatic  new value
    * @see              #setStatic
    */
   public void setUmlStatic (boolean umlStatic)
   {
      this.setStatic (umlStatic);
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlSignal;


   /**
    * Sets the signal attribute of the UMLMethod object
    *
    * @param umlSignal  The new signal value
    */
   public void setSignal (boolean umlSignal)
   {
      boolean oldValue = this.umlSignal;
      UMLClass parent = getParent();
      if (parent != null)
      {
         //update signals in UMLClass
         setParent (null);
      }
      this.umlSignal = umlSignal;
      if (parent != null)
      {
         setParent (parent);
      }
      firePropertyChange ("signal", oldValue, umlSignal);
   }


   /**
    * Wrapper to meet the style guides. Toggles the signal flag.
    *
    * @param umlSignal  new value
    * @see              #setSignal
    */
   public void setUmlSignal (boolean umlSignal)
   {
      this.setSignal (umlSignal);
   }


   /**
    * Get the signal attribute of the UMLMethod object
    *
    * @return   The signal value
    */
   public boolean isSignal()
   {
      return umlSignal;
   }


   /**
    * Wrapper to meet the style guides. Requests the current value of the signal flag.
    *
    * @return   The umlSignal value
    * @see      #isSignal
    */
   public boolean isUmlSignal()
   {
      return  (this.isSignal());
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlFinal;


   /**
    * Sets the final attribute of the UMLMethod object
    *
    * @param umlFinal  The new final value
    */
   public void setFinal (boolean umlFinal)
   {
      boolean oldValue = this.umlFinal;
      this.umlFinal = umlFinal;

      firePropertyChange ("final", oldValue, umlFinal);
   }


   /**
    * Wrapper to meet the style guides. Toggles the final flag.
    *
    * @param umlFinal  new value
    * @see             #setFinal
    */
   public void setUmlFinal (boolean umlFinal)
   {
      this.setFinal (umlFinal);
   }


   /**
    * Get the final attribute of the UMLMethod object
    *
    * @return   The final value
    */
   public boolean isFinal()
   {
      return umlFinal;
   }


   /**
    * Wrapper to meet the style guides. Requests the current value of the final flag.
    *
    * @return   The umlFinal value
    * @see      #isFinal
    */
   public boolean isUmlFinal()
   {
      return  (this.isFinal());
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlAbstract;


   /**
    * Sets the abstract attribute of the UMLMethod object
    *
    * @param umlAbstract  The new abstract value
    */
   public void setAbstract (boolean umlAbstract)
   {
      setUmlAbstract (umlAbstract);
   }


   /**
    * Wrapper for to meet the style guides. Toggles the abstract flag.
    *
    * @param umlAbstract  The new umlAbstract value
    * @see                #setAbstract
    */
   public void setUmlAbstract (boolean umlAbstract)
   {
      boolean oldValue = this.umlAbstract;
      this.umlAbstract = umlAbstract;

      if (umlAbstract == true && getParent() != null)
      {
         getParent().setUmlAbstract (true);
      }

      firePropertyChange ("umlAbstract", oldValue, umlAbstract);
   }


   /**
    * Get the abstract attribute of the UMLMethod object
    *
    * @return   The abstract value
    */
   public boolean isAbstract()
   {
      return umlAbstract;
   }


   /**
    * Wrapper to meet the style guides. Requests the current value of the abstract flag.
    *
    * @return   The umlAbstract value
    * @see      #isAbstract
    */
   public boolean isUmlAbstract()
   {
      return  (this.isAbstract());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlNative;


   /**
    * Get the umlNative attribute of the UMLMethod object
    *
    * @return   The umlNative value
    */
   public boolean isUmlNative()
   {
      return this.umlNative;
   } // isUmlNative


   /**
    * Sets the umlNative attribute of the UMLMethod object
    *
    * @param newUmlNative  The new umlNative value
    */
   public void setUmlNative (boolean newUmlNative)
   {
      boolean oldValue = this.umlNative;
      this.umlNative = newUmlNative;
      firePropertyChange ("umlNative", oldValue, umlNative);
   } // setUmlNative


   /**
    * Sets the synchronized attribute of the UMLMethod object
    *
    * @param umlSynchronized  The new synchronized value
    */
   public void setSynchronized (boolean umlSynchronized)
   {
      boolean oldValue = this.umlSynchronized;
      this.umlSynchronized = umlSynchronized;

      firePropertyChange ("synchronized", oldValue, umlSynchronized);
   }


   /**
    * Get the synchronized attribute of the UMLMethod object
    *
    * @return   The synchronized value
    */
   public boolean isSynchronized()
   {
      return this.umlSynchronized;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean umlSynchronized;


   /**
    * Get the umlSynchronized attribute of the UMLMethod object
    *
    * @return   The umlSynchronized value
    */
   public boolean isUmlSynchronized()
   {
      return this.isSynchronized();
   } // getUmlSynchronized


   /**
    * Sets the umlSynchronized attribute of the UMLMethod object
    *
    * @param newUmlSynchronized  The new umlSynchronized value
    */
   public void setUmlSynchronized (boolean newUmlSynchronized)
   {
      boolean oldValue = this.umlSynchronized;
      this.umlSynchronized = newUmlSynchronized;
      firePropertyChange ("umlSynchronized", oldValue, umlSynchronized);
   } // setUmlSynchronized


   /**
    * <pre>
    *            1                    1 +---------------------+
    * UMLMethod ------------------------+ getFullMethodName() | UMLClass
    *            methods         parent +---------------------+
    * </pre>
    */
   private transient UMLClass parent;

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the parent attribute of the UMLMethod object
    *
    * @return   The parent value
    */
   public UMLClass getParent()
   {
      return this.parent;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FMethod#getFParent()
    */
   /**
    * Get the fParent attribute of the UMLMethod object
    *
    * @return   The fParent value
    */
   public FClass getFParent()
   {
      return getParent();
   }
   // TODO-END


   /**
    * Sets the parent attribute of the UMLMethod object
    *
    * @param parent  The new parent value
    */
   public void setParent (FClass parent)
   {
      if (this.parent != parent)
      {
         UMLClass oldParent = this.parent;
         if (this.parent != null)
         {
            this.parent = null;
            oldParent.removeFromMethods (this);
         }
         this.parent = (UMLClass) parent;
         if (parent != null)
         {
            parent.addToMethods (this);
         }

         // side effects
         firePropertyChange (PARENT_PROPERTY, oldParent, parent);
      }
   }


   /**
    * Get the visibilityType attribute of the UMLMethod class
    *
    * @param visibility  No description provided
    * @return            The visibilityType value
    */
   public static String getVisibilityType (int visibility)
   {
      String visibilityType = null;

      switch (visibility)
      {
         case FDeclaration.PRIVATE:
            visibilityType = IconsPreferences.PRIV_METH;
            break;
         case FDeclaration.PROTECTED:
            visibilityType = IconsPreferences.PROT_METH;
            break;
         case FDeclaration.PACKAGE:
            visibilityType = IconsPreferences.PACK_METH;
            break;
         case FDeclaration.PUBLIC:
            visibilityType = IconsPreferences.PUB_METH;
            break;
      }

      return visibilityType;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLType resultType;


   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the resultType attribute of the UMLMethod object
    *
    * @return   The resultType value
    */
   public UMLType getResultType()
   {
      return resultType;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FMethod#getFResultType()
    */
   /**
    * Get the fResultType attribute of the UMLMethod object
    *
    * @return   The fResultType value
    */
   public FType getFResultType()
   {
      return getResultType();
   }
   // TODO-END

   /**
    * Get the uMLType attribute of the UMLMethod object
    *
    * @return   The uMLType value
    */
   public UMLType getUMLType()
   {
      return getResultType();
   }


   /**
    * Sets the resultType attribute of the UMLMethod object
    *
    * @param value  The new resultType value
    */
   public void setResultType (FType value)
   {
      if (this.resultType != value)
      {
         UMLType oldResultType = this.resultType;
         if (this.resultType != null)
         {
            this.resultType = null;
            oldResultType.removeFromRevResultType (this);
         }
         this.resultType = (UMLType) value;
         if (value != null)
         {
             ((UMLType) value).addToRevResultType (this);
         }

         firePropertyChange (RESULT_TYPE_PROPERTY, oldResultType, value);
      }
   }


   /**
    * UMLAttribute : 'resultTypeIsPointer : Boolean '
    */
   private boolean resultTypeIsPointer = false;


   /**
    * Get the value of resultTypeIsPointer.
    *
    * @return   Value of resultTypeIsPointer.
    */
   public boolean isResultTypeIsPointer()
   {
      return this.resultTypeIsPointer;
   }


   /**
    * Set the value of resultTypeIsPointer.
    *
    * @param resultTypeIsPointer  Value to assign to resultTypeIsPointer.
    */
   public void setResultTypeIsPointer (boolean resultTypeIsPointer)
   {
      if (this.resultTypeIsPointer != resultTypeIsPointer)
      {
         this.resultTypeIsPointer = resultTypeIsPointer;
      }
   }


   /**
    * Returns true, if the method is a constructor. The result type may be null
    * or UMLBaseTypes.CONSTRUCTOR, which is an UML-Basetype and respresents null.
    * Furthermore the method name is the same as the class name.
    *
    * @return   True, if the method is a constructor
    */
   public boolean isConstructor()
   {
      return  ( (resultType == null) ||  (resultType.getName().equals (UMLBaseTypes.CONSTRUCTOR)))
         &&  (parent != null) &&  (getName().equals (parent.getName()));
   }


   /**
    * @return       true, if method is a constructor
    * @deprecated   use #isConstructor instead
    */
   public boolean isJavaConstructor()
   {
      return isConstructor();
   }


   /**
    * <pre>
    *          +-----------+ 1                          1
    * UMLClass | getName() +------------------------------ UMLMethod
    *          +-----------+ declares    declaredInMethod
    * </pre>
    */
   private transient FPropHashMap declares;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInDeclares (FClass obj)
   {
      return  ( (this.declares != null) &&
          (obj != null) &&  (obj.getName() != null) &&
          (this.declares.get (obj.getName()) == obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInDeclares (String key)
   {
      return  ( (this.declares != null) &&
          (key != null) &&
         this.declares.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfDeclares()
   {
      return  ( (this.declares == null)
         ? 0
         : this.declares.size());
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromDeclares attribute of the UMLMethod object
    *
    * @param key  No description provided
    * @return     The fromDeclares value
    */
   public UMLClass getFromDeclares (String key)
   {
      return  ( ( (this.declares == null) ||  (key == null))
         ? null
         : (UMLClass) this.declares.get (key));
   }


   /**
    * Get the fromFDeclares attribute of the UMLMethod object
    *
    * @param key  No description provided
    * @return     The fromFDeclares value
    * @see        de.uni_paderborn.fujaba.metamodel.FMethod#getFromFDeclares(java.lang.String)
    */
   public FClass getFromFDeclares (String key)
   {
      return getFromDeclares (key);
   }
   // TODO-END

   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToDeclares (FClass obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getName() != null))
      {
         if (this.declares == null)
         {
            this.declares = new FPropHashMap (this, DECLARES_PROPERTY);
         }
         UMLClass oldValue = (UMLClass) this.declares.put (obj.getName(), obj);
         if (oldValue != obj)
         {
            if (oldValue != null)
            {
               oldValue.setDeclaredInMethod (null);
            }
            obj.setDeclaredInMethod (this);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromDeclares (FClass obj)
   {
      boolean changed = false;

      if ( (this.declares != null) &&  (obj != null) &&  (obj.getName() != null))
      {
         UMLClass oldValue = (UMLClass) this.declares.get (obj.getName());
         if (oldValue == obj)
         {
            this.declares.remove (obj.getName());
            obj.setDeclaredInMethod (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromDeclares (String key)
   {
      boolean changed = false;

      if ( (this.declares != null) &&  (key != null))
      {
         UMLClass tmpObj = (UMLClass) this.declares.get (key);
         if (tmpObj != null)
         {
            this.declares.remove (key);
            tmpObj.setDeclaredInMethod (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromDeclares()
   {
      UMLClass tmpObj;
      Iterator iter = this.iteratorOfDeclares();

      while (iter.hasNext())
      {
         tmpObj = (UMLClass) iter.next();
         this.removeFromDeclares (tmpObj);
      }
   }


   /**
    * This method is needed only for loading FTreeMaps, do not use it in other cases.
    *
    * @param pair  The object added.
    */
   protected void addToDeclares (KeyValuePair pair)
   {
      if (pair != null)
      {
         UMLClass elem = (UMLClass) pair.getValue();
         String key = (String) pair.getKey();
         if ( (elem != null) &&  (key != null) && ! (key.equals (elem.getName())))
         {
            elem.setName (key);
         }
         addToDeclares (elem);
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FPropLinkedList param = null; // reverse UMLMethod revParam


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param newParam  No description provided
    * @return          No description provided
    */
   public boolean hasInParam (FParam newParam)
   {
      if (param == null)
      {
         return false;
      }
      return param.contains (newParam);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfParam()
   {
      if (param == null)
      {
         return 0;
      }
      return param.size();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfParam()
   {
      if (this.param == null)
      {
         return FEmptyIterator.get();
      }
      return this.param.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param newParam  The object added.
    */
   public void addToParam (FParam newParam)
   {
      if (newParam != null && !hasInParam (newParam))
      {
         if (param == null)
         {
            param = new FPropLinkedList (this, PARAM_PROPERTY);
         }
         param.add (newParam);
         newParam.setRevParam (this);
         //----- side-effects begin
         updateAssocOverFullMethodName();

         //----- side-effects end
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param index     The object added.
    * @param newParam  The object added.
    */
   public void addToParam (int index, FParam newParam)
   {
      if (newParam != null && !hasInParam (newParam))
      {
         if (param == null)
         {
            param = new FPropLinkedList (this, PARAM_PROPERTY);
         }
         param.add (index, newParam);
         newParam.setRevParam (this);
         //----- side-effects begin
         updateAssocOverFullMethodName();

         //----- side-effects end
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param newParam  No description provided
    */
   public void removeFromParam (FParam newParam)
   {
      if (newParam != null && hasInParam (newParam))
      {
         param.remove (newParam);
         newParam.removeYou();

         //----- side-effects begin
         updateAssocOverFullMethodName();

         //----- side-effects ende
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final void removeAllFromParam()
   {
      UMLParam item;
      Iterator iter = iteratorOfParam();

      while (iter.hasNext())
      {
         item = (UMLParam) iter.next();
         removeFromParam (item);
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLStartActivity revSpec; // reverse UMLMethod spec

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the revSpec attribute of the UMLMethod object
    *
    * @return   The revSpec value
    */
   public UMLStartActivity getRevSpec()
   {
      return revSpec;
   }


   /**
    * Get the fRevSpec attribute of the UMLMethod object
    *
    * @return   The fRevSpec value
    * @see      de.uni_paderborn.fujaba.metamodel.FMethod#getFRevSpec()
    */
   public FElement getFRevSpec()
   {
      return getRevSpec();
   }
   // TODO-END

   /**
    * Sets the revSpec attribute of the UMLMethod object
    *
    * @param revSpec  The new revSpec value
    */
   public void setRevSpec (FElement revSpec)
   {
      if (this.revSpec != revSpec)
      {
         // new partner
         UMLStartActivity oldRevSpec = this.revSpec;
         if (this.revSpec != null)
         { // inform old partner

            this.revSpec = null;
            oldRevSpec.setSpec (null);
         }
         this.revSpec = (UMLStartActivity) revSpec;
         if (revSpec != null)
         {
            // inform new partner
             ((UMLStartActivity) revSpec).setSpec (this);

            // side effect
            UMLActivityDiagram umlActivityDiagram = (UMLActivityDiagram)  ((UMLStartActivity) revSpec).getFirstFromDiagrams();
            if (umlActivityDiagram != null)
            {
               this.setStoryDiag (umlActivityDiagram);
            }

            // A new activity diagram is set, the method body should be removed.
            setMethodBody (null);
         }

         // inform storyDiag, too
         firePropertyChange ("revSpec", oldRevSpec, revSpec);
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLActivityDiagram storyDiag; // reverse UMLMethod spec

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the storyDiag attribute of the UMLMethod object
    *
    * @return   The revSpec value
    */
   public UMLActivityDiagram getStoryDiag()
   {
      return storyDiag;
   }


   /**
    * Get the fStoryDiagram attribute of the UMLMethod object
    *
    * @return   The fStoryDiagram value
    * @see      de.uni_paderborn.fujaba.metamodel.FMethod#getFStoryDiagram()
    */
   public FDiagram getFStoryDiagram()
   {
      return getStoryDiagram();
   }
   // TODO-END

   /**
    * Sets the revSpec attribute of the UMLMethod object
    *
    * @param storyDiag  The new storyDiag value
    */
   public void setStoryDiag (UMLActivityDiagram storyDiag)
   {
      if (this.storyDiag != storyDiag)
      { // new partner

         UMLActivityDiagram oldStoryDiag = this.storyDiag;
         if (this.storyDiag != null)
         { // inform old partner

            this.storyDiag = null;
            oldStoryDiag.setStoryMethod (null);
         }
         this.storyDiag = storyDiag;
         if (storyDiag != null)
         { // inform new partner

            storyDiag.setStoryMethod (this);
         }

         //todo: XProM need this condition: if (this.getDisplayLevel() == FDeclaration.DESIGN_DISPLAY_LEVEL)
         firePropertyChange ("storyDiag", oldStoryDiag, storyDiag);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void removeRevSpec()
   {
      if (this.revSpec != null)
      {
         UMLActivityDiagram diag = this.revSpec.getActivityDiagram();
         if (diag != null)
         {
            diag.removeYou();
         }
         setRevSpec (null);
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String methodBody;


   /**
    * Sets the methodBody of the UMLMethod
    *
    * @param methodBody  The new methodBody value
    */
   public void setMethodBody (String methodBody)
   {
      setMethodBody (methodBody, false);
   }


   /**
    * Sets the methodBody of the UMLMethod
    *
    * @param methodBody  The new methodBody value
    * @param generated   The new methodBody value
    */
   public void setMethodBody (String methodBody, boolean generated)
   {
      this.methodBody = methodBody;

      if (this.methodBody != null && !this.syncMethodBody)
      {
         if (!UMLProject.isLoading())
         {
            createActivityDiagram (generated);
         }
      }
   }


   /**
    * Get the methodBody of the UMLMethod
    *
    * @return   The methodBody
    */
   public String getMethodBody()
   {
      return this.methodBody;
   }


   /**
    * Needed to synchronize method body and activity diagram.
    */
   private transient boolean syncMethodBody = false;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param generated  No description provided
    */
   private void createActivityDiagram (boolean generated)
   {
      // synchronizing methodbody
      this.syncMethodBody = true;
      String tmpMethodBody = getMethodBody();

      String parentName = null;
      if (getParent() != null)
      {
         parentName = getParent().getName();
      }
      else
      {
         parentName = "no name";
      }

      UMLActivityDiagram activityDiagram = null;
      if (this.getRevSpec() != null)
      {
         // remove the old activity diagram
         UMLStartActivity startActivity = this.getRevSpec();
         activityDiagram = startActivity.getActivityDiagram();
         if (activityDiagram != null)
         {
            activityDiagram.removeYou();
         }
      }

      boolean coobraPersistent = isCoobraPersistent();

      // create a new activity diagram
      activityDiagram = new UMLActivityDiagram (coobraPersistent);
      activityDiagram.setName (parentName + "::" + getName());
      activityDiagram.setGenerated (generated);
      activityDiagram.setStoryMethod (this);
      UMLProject.get().addToDiags (activityDiagram);

      UMLStartActivity start = new UMLStartActivity (coobraPersistent);
      start.setGenerated (generated);
      activityDiagram.addToElements (start);
      start.setSpec (this);

      UMLStatement statement = new UMLStatement (coobraPersistent);
      statement.setStatement (tmpMethodBody);
      statement.setGenerated (generated);

      UMLStatementActivity statementActivity = new UMLStatementActivity (coobraPersistent, true);
      statementActivity.setState (statement);
      statementActivity.setGenerated (generated);
      activityDiagram.addToElements (statementActivity);

      UMLTransitionGuard newGuard = new UMLTransitionGuard (coobraPersistent);
      newGuard.setType (UMLTransitionGuard.NONE);
      newGuard.setGenerated (generated);

      UMLTransition newTransition = new UMLTransition (coobraPersistent);
      newTransition.setGuard (newGuard);
      newTransition.setSource (start);
      newTransition.setTarget (statementActivity);
      newTransition.setGenerated (generated);
      activityDiagram.addToElements (newTransition);

      UMLStopActivity stop = new UMLStopActivity (coobraPersistent, false);
      stop.setGenerated (generated);
      activityDiagram.addToElements (stop);

      newGuard = new UMLTransitionGuard (coobraPersistent);
      newGuard.setType (UMLTransitionGuard.NONE);
      newGuard.setGenerated (generated);

      newTransition = new UMLTransition (coobraPersistent);
      newTransition.setGuard (newGuard);
      newTransition.setSource (statementActivity);
      newTransition.setTarget (stop);
      newTransition.setGenerated (generated);
      activityDiagram.addToElements (newTransition);

      // at least set the old method body since it was removed by setRevSpec()
      setMethodBody (tmpMethodBody);
      this.syncMethodBody = false;
   }


   /**
    * Creates a new body of the method with the given text be sure the methods class is set
    * using setParent !
    *
    * @param bodyText   the body text
    * @param generated  true if the body is auto generated (e.g. from the analyzer engines)
    * @deprecated       use setMethodBody(String methodBody) instead
    */
   public void setPatternMethodBody (String bodyText, boolean generated)
   {
      setMethodBody (bodyText, generated);
   } // setPatternMethodBody


   /**
    * Sets the methodBodyText attribute of the UMLMethod object
    *
    * @param buf    The new methodBodyText value
    * @deprecated   use setMethodBody(String methodBody) instead
    */
   public void setMethodBodyText (StringBuffer buf)
   {
      setMethodBody (buf.toString(), false);
   }


   /**
    * Get the methodBodyText attribute of the UMLMethod object
    *
    * @return       The methodBodyText value
    * @deprecated   use getMethodBody() instead
    */
   public StringBuffer getMethodBodyText()
   {
      return new StringBuffer (getMethodBody());
   }


   /**
    * <pre>
    *                   astRootNode     0..1
    * UMLMethod -----------------------------> ASTRootNode
    *                            astRootNode
    * </pre>
    */
   private transient SoftReference astRootNodeReference;


   /**
    * Returns the abstract syntax tree (AST) root node of the method
    * If the method body isn't parsed yet, the method body will be parsed.
    *
    * @return   The AST root node
    */
   public ASTRootNode getASTRootNode()
   {
      ASTRootNode astRootNode = null;

      if ( (this.astRootNodeReference == null ||
         this.astRootNodeReference.get() == null) &&
         getMethodBody() != null)
      {
         Parser parser = ParserManager.get().getCurrentParser();
         if (parser != null)
         {
            astRootNode = parser.parseMethodBody (this);
            setASTRootNode (astRootNode);
         }
      }
      else if (this.astRootNodeReference != null)
      {
         astRootNode = (ASTRootNode) astRootNodeReference.get();
      }

      return astRootNode;
   }


   /**
    * Sets the abstract syntax tree (AST) root node of the method
    *
    * @param astRootNode  The new AST root node
    */
   public void setASTRootNode (ASTRootNode astRootNode)
   {
      if (astRootNode != null &&
          (astRootNodeReference == null || astRootNodeReference.get() != astRootNode))
      {
         this.astRootNodeReference = new SoftReference (astRootNode);
         astRootNode.setMethod (this);
      }
      else if (astRootNode == null)
      {
         this.astRootNodeReference = null;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeStoryDiagram()
   {
      UMLDiagram diag = this.getStoryDiagram();
      if (diag != null)
      {
         diag.removeYou();
      }
   } // removeStoryDiagram


   /**
    * Get the storyDiagram attribute of the UMLMethod object
    *
    * @return   The storyDiagram value
    */
   public UMLActivityDiagram getStoryDiagram()
   {
      UMLDiagram diag = null;
      UMLStartActivity activity = getRevSpec();

      if (activity != null &&
         activity.sizeOfDiagrams() > 0)
      {
         diag = (UMLDiagram) activity.iteratorOfDiagrams().next();
      }
      return (UMLActivityDiagram) diag;
   } // getStoryDiagram


   /**
    * <pre>
    *          n      throwsTypes  0..1
    * UMLType ------------------------- UMLMethod
    *          throws      throwMethod
    * </pre>
    */
   private FHashSet throwsTypes;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInThrowsTypes (UMLType value)
   {
      return  ( (this.throwsTypes != null) &&
          (value != null) &&
         this.throwsTypes.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfThrowsTypes()
   {
      return  ( (this.throwsTypes == null)
         ? FEmptyIterator.get()
         : this.throwsTypes.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfThrowsTypes()
   {
      return  ( (this.throwsTypes == null)
         ? 0
         : this.throwsTypes.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToThrowsTypes (UMLType value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.throwsTypes == null)
         {
            this.throwsTypes = new FPropHashSet (this, "throwsTypes");
         }
         changed = this.throwsTypes.add (value);
         if (changed)
         {
            //value.setThrowMethod (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromThrowsTypes (UMLType value)
   {
      boolean changed = false;
      if ( (this.throwsTypes != null) &&  (value != null))
      {
         changed = this.throwsTypes.remove (value);
         if (changed)
         {
            //value.setThrowMethod (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromThrowsTypes()
   {
      UMLType tmpValue;
      Iterator iter = this.iteratorOfThrowsTypes();
      while (iter.hasNext())
      {
         tmpValue = (UMLType) iter.next();
         this.removeFromThrowsTypes (tmpValue);
      }
   }


   /**
    * Hangs the current ASGElement into the ASG-tree. Needed for cut'n'paste.
    *
    * @param parent
    */
   public void setCutCopyPasteParent (FElement parent)
   {
      if (parent instanceof UMLClass)
      {
          ((UMLClass) parent).addToMethods (this);
      }
   }


   /**
    * <pre>
    *          0..1                            0..*
    * UMLAttr -------------------------------------- UMLMethod
    *          accessedAttribute      accessMethods
    * </pre>
    */
   private transient UMLAttr accessedAttribute;


   /**
    * Sets the accessedAttribute attribute of the UMLMethod object
    *
    * @param value  The new accessedAttribute value
    * @return       No description provided
    */
   public boolean setAccessedAttribute (FAttr value)
   {
      boolean changed = false;

      if (this.accessedAttribute != value)
      {
         if (this.accessedAttribute != null)
         {
            UMLAttr oldValue = this.accessedAttribute;
            this.accessedAttribute = null;
            oldValue.removeFromAccessMethods (this);
         }

         this.accessedAttribute = (UMLAttr) value;

         if (value != null)
         {
             ((UMLAttr) value).addToAccessMethods (this);
         }

         changed = true;
      }

      return changed;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the accessedAttribute attribute of the UMLMethod object
    *
    * @return   The accessedAttribute value
    */
   public UMLAttr getAccessedAttribute()
   {
      return this.accessedAttribute;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FMethod#getFAccessedAttribute()
    */
   /**
    * Get the fAccessedAttribute attribute of the UMLMethod object
    *
    * @return   The fAccessedAttribute value
    */
   public FAttr getFAccessedAttribute()
   {
      return getAccessedAttribute();
   }
   // TODO-END

   /**
    * <pre>
    *            0..1     method     0..1
    * UMLAction -------------------------- UMLMethod
    *            uMLAction      uMLMethod
    * </pre>
    */
   private UMLAction uMLAction;


   /**
    * Sets the uMLAction attribute of the UMLMethod object
    *
    * @param value  The new uMLAction value
    * @return       No description provided
    */
   public boolean setUMLAction (UMLAction value)
   {
      boolean changed = false;
      if (this.uMLAction != value)
      {
         UMLAction oldValue = this.uMLAction;
         if (this.uMLAction != null)
         {
            this.uMLAction = null;
            oldValue.setUMLMethod (null);
         }
         this.uMLAction = value;
         firePropertyChange ("uMLAction", oldValue, value);
         if (value != null)
         {
            value.setUMLMethod (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * Get the uMLAction attribute of the UMLMethod object
    *
    * @return   The uMLAction value
    */
   public UMLAction getUMLAction()
   {
      return this.uMLAction;
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      this.setParent (null);

      this.setResultType (null);
      this.removeAllFromDeclares();
      this.removeAllFromParam();
      this.removeRevSpec();

      if (FujabaDebug.SETATTRSTONULL)
      {
         this.cacheForFullMethodName = null;
         this.declares = null;
         this.param = null;
      }
      removeAllFromThrowsTypes();

      UMLAttr tmpAccessedAttribute = getAccessedAttribute();
      if (tmpAccessedAttribute != null)
      {
         setAccessedAttribute (null);
      } // if

      UMLAction tmpUMLAction = getUMLAction();
      if (tmpUMLAction != null)
      {
         setUMLAction (null);
      } // if

      super.removeYou();
   } // removeYou


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return getParent();
   }


   /**
    * Searches the ASG tree for a given id
    *
    * @param id  The id to search for.
    * @return    The Element with the given id, null if not found.
    */
   public ASGElement searchID (String id)
   {
      ASGElement elem = super.searchID (id);
      Iterator iter = iteratorOfParam();
      while ( (elem == null) &&  (iter.hasNext()))
      {
         elem =  ((UMLParam) iter.next()).searchID (id);
      }
      return elem;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public UMLActivityDiagram createStoryDiagram()
   {
      // create diagram
      UMLActivityDiagram diagram = new UMLActivityDiagram (this.getFullMethodName(), null);

      // add new diagram to project
      UMLProject.get().addToDiags (diagram);
      UMLProject.get().getGui().createNewTreeItems();

      // create start activity
      diagram.addToElements (new UMLStartActivity (this));

      return diagram;
   }


   /**
    * Find all methods in superclasses and -interfaces that have the same signature as this method.
    *
    * @return   iterator through overwritten/implemented methods
    */
   public Iterator iteratorOfOverriddenMethods()
   {
      UMLClass cls = getParent();
      if (cls != null)
      {
         String signature = getFullMethodName();
         Set methods = cls.findMethodsWithSignatureInSuperclasses (signature);
         return methods.iterator();
      }
      else
      {
         return FEmptyIterator.get();
      }
   }


   /**
    * Find all methods in subclasses and -interfaces that have the same signature as this method.
    *
    * @return   iterator through overwriting/implementing methods
    */
   public Iterator iteratorOfOverridingMethods()
   {
      UMLClass cls = getParent();
      if (cls != null)
      {
         String signature = getFullMethodName();
         Set methods = cls.findMethodsWithSignatureInSubclasses (signature);
         return methods.iterator();
      }
      else
      {
         return FEmptyIterator.get();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public int compareTo (Object obj)
   {
      if (obj instanceof FMethod)
      {
         final int cmp = getFullMethodName().compareTo ( ((FMethod) obj).getFullMethodName());
         if (cmp == 0 && !this.equals (obj))
         {
            return super.compareTo (obj);
         }
         else
         {
            return cmp;
         }
      }
      else
      {
         return super.compareTo (obj);
      }
   }
}

/*
 * $Log: UMLMethod.java,v $
 * Revision 1.380.2.5  2006/02/06 14:45:06  lowende
 * Fix for caching of the full method name (signature). Patch by Dietrich.
 *
 */
