/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.Iterator;

import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.unparse.UMLUnparseGetter;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FPropHashSet;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *                      +-----------+ 0..1        uMLStereotypes         0..1
 * UMLStereotypeManager | getText() |----------------------------------------- UMLStereotype
 *                      +-----------+ uMLStereotypeManager     uMLStereotypes
 *
 *              +-----------+ 0..n                   0..1
 * UMLIncrement | getText() |----------------------------- UMLStereotype
 *              +-----------+ increments     uMLStereotype
 * </pre>
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.85 $
 */
public class UMLStereotype extends UMLIncrement implements FStereotype
{
   /**
    * Constructor for class UMLStereotype
    */
   public UMLStereotype()
   {
      super();
   }


   /**
    * Constructor for class UMLStereotype
    *
    * @param text  text for the stereotype
    */
   public UMLStereotype (String text)
   {
      this();
      setText (text);
   }


   /**
    * The stereotype type
    */
   private String text = "";


   /**
    * Sets the text attribute of the UMLStereotype object
    *
    * @param text  The new text value
    * @return      the new value for the text attribute
    */
   public String setText (String text)
   {
      if ( (this.text == null && text != null) ||
          (this.text != null && !this.text.equals (text)))
      {
         String oldValue = this.text;
         this.text = text;
         firePropertyChange ("text", oldValue, text);
         setUMLStereotypeManager (null);
         if (text != null)
         {
            setUMLStereotypeManager (UMLStereotypeManager.get());
         }
      }

      return this.text;
   }


   /**
    * Get the text attribute of the UMLStereotype object
    *
    * @return   The text value
    */
   public String getText()
   {
      return text;
   }


   /**
    * <pre>
    *                      +-----------+ 0..1        uMLStereotypes         0..1
    * UMLStereotypeManager | getText() |----------------------------------------- UMLStereotype
    *                      +-----------+ uMLStereotypeManager     uMLStereotypes
    * </pre>
    */
   private UMLStereotypeManager uMLStereotypeManager;


   /**
    * Get the uMLStereotypeManager attribute of the UMLStereotype object
    *
    * @return   The uMLStereotypeManager value
    */
   public UMLStereotypeManager getUMLStereotypeManager()
   {
      return this.uMLStereotypeManager;
   }


   /**
    * Sets the uMLStereotypeManager attribute of the UMLStereotype object
    *
    * @param value  The new uMLStereotypeManager value
    * @return       true when manager was changed
    */
   protected boolean setUMLStereotypeManager (UMLStereotypeManager value)
   {
      boolean changed = false;
      if (this.uMLStereotypeManager != value)
      {
         if (this.uMLStereotypeManager != null)
         {
            UMLStereotypeManager oldValue = this.uMLStereotypeManager;
            this.uMLStereotypeManager = null;
            oldValue.removeFromStereotypes (this);
         }
         this.uMLStereotypeManager = value;
         if (value != null)
         {
            value.addToStereotypes (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * <pre>
    *              +-----------+ 0..n                   0..1
    * UMLIncrement | getText() |----------------------------- UMLStereotype
    *              +-----------+ increments     uMLStereotype
    * </pre>
    */
   private transient FPropHashSet increments;


   /**
    * Mark the value with this stereotype
    *
    * @param value  The object that is marked
    * @return       true if object was not already marked
    */
   public boolean addToIncrements (FIncrement value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.increments == null)
         {
            this.increments = new FPropHashSet (this, "increments");
         }
         changed = this.increments.add (value);
         if (changed)
         {
            value.addToStereotypes (this);
         }
      }
      return changed;
   }


   /**
    * Check if increment has this stereotype.
    *
    * @param value  increment of interest
    * @return       if value is marked with this stereotype
    */
   public boolean hasInIncrements (FIncrement value)
   {
      return  ( (this.increments != null) &&
          (value != null) &&
         this.increments.contains (value));
   }


   /**
    * List all increments that have this stereotype.
    *
    * @return   iterator through UMLIncrements
    */
   public Iterator iteratorOfIncrements()
   {
      return  ( (this.increments == null)
         ? FEmptyIterator.get()
         : this.increments.iterator());
   }


   /**
    * remove this stereotype from all increments.
    */
   public void removeAllFromIncrements()
   {
      UMLIncrement tmpValue;
      Iterator iter = this.iteratorOfIncrements();
      while (iter.hasNext())
      {
         tmpValue = (UMLIncrement) iter.next();
         this.removeFromIncrements (tmpValue);
      }
   }


   /**
    * Unmark the value.
    *
    * @param value  increment that will no longer be of this stereotype
    * @return       true if increment had this stereotype
    */
   public boolean removeFromIncrements (FIncrement value)
   {
      boolean changed = false;
      if ( (this.increments != null) &&  (value != null))
      {
         changed = this.increments.remove (value);
         if (changed)
         {
            value.removeFromStereotypes (this);
         }
      }
      return changed;
   }


   /**
    * number of increments of this stereotype.
    *
    * @return   count of increments
    */
   public int sizeOfIncrements()
   {
      return  ( (this.increments == null)
         ? 0
         : this.increments.size());
   }


   /**
    * Access method for an one to n association.
    * Just for loading old project files
    *
    * @param value  The object added.
    * @return       No description provided
    * @deprecated   use increments association instead
    */
   public boolean addToUMLClass (UMLClass value)
   {
      return addToIncrements (value);
   }


   /**
    * @return   the name of the unparse module for this stereotype
    */
   protected String createUnparseModuleName()
   {
      return UMLUnparseGetter.getUnparseModuleName (this);
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      return getText();
      //return "UMLStereotype[" + getText() + "," + getID() + "]";
   }


   /**
    * delete this stereotype
    */
   public void removeYou()
   {
      removeAllFromIncrements();
      setUMLStereotypeManager (null);
      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      //todo: this is a top level object - should it get a package?
      return UMLProject.get();
   }
}

/*
 * $Log: UMLStereotype.java,v $
 * Revision 1.85  2005/02/14 14:28:21  cschneid
 * introduced message view, getParentElement() for FElement, code generation and compiler messages in message view
 *
 */
