/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml.actions;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.util.*;

import javax.swing.*;
import javax.xml.parsers.*;

import org.apache.log4j.Logger;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.gui.SelectFromListDialog;
import de.uni_paderborn.fujaba.uml.*;


/**
 * Merge a Sequence Diagram into a Statechart Diagram: not implemented, yet!
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.19.2.1 $
 */
public class MergeSequenceDiagramIntoStateChartAction extends AbstractAction
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (MergeSequenceDiagramIntoStateChartAction.class);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    */
   public void actionPerformed (ActionEvent event)
   {
      FrameMain frameMain = FrameMain.get();

      //FIXME: was this meant:
      UMLStatechart statechart = (UMLStatechart) UMLProject.get().getCurrentDiagram();

      try
      {
         if (log.isInfoEnabled())
         {
            log.info ("FIXME: This is work in progress!");
         }

         JFileChooser fileChooser = frameMain.getFileChooser();
         fileChooser.setCurrentDirectory
             (new File (UMLProject.get().getRootDir(),
            "examples/xml-sequence-diagrams"));
         fileChooser.setFileSelectionMode (JFileChooser.FILES_ONLY);
         // FIXME: remove next line, I just hate that fu*#!$& small window
         // and don't know how to do it better
         fileChooser.setPreferredSize (new Dimension (600, 600));

         if (fileChooser.showOpenDialog (frameMain.getFrame()) == JFileChooser.APPROVE_OPTION)
         {
            frameMain.setStatusLabel ("Merging...");
            frameMain.setCursorWait();

            File sdFile = fileChooser.getSelectedFile();
            if (sdFile.exists() && sdFile.isFile())
            {
               Collection sequenceDiagramObjects = sequenceDiagramObjects (sdFile);

               SelectFromListDialog objectsDialog = new SelectFromListDialog (frameMain.getFrame());
               objectsDialog.setObjects (sequenceDiagramObjects);
               objectsDialog.setHeading ("Select an object of the sequence-diagram:    ");
               objectsDialog.show();

               if (objectsDialog.getSelectedObject() != null)
               {
                  String tracedObject = (String) objectsDialog.getSelectedObject();
                  Trace trace = createSequenceDiagramTrace (sdFile, tracedObject);
                  if (trace != null)
                  {
                     if (log.isInfoEnabled())
                     {
                        log.info (trace);
                     }
                     mergeTraceIntoStatechart (statechart, trace);
                     frameMain.setStatusLabel ("Merged.");
                  }
                  else
                  {
                     throw new Exception ("Error while reading " + sdFile + ".");
                  }
               }
               else
               {
                  frameMain.setStatusLabel ("Cancelled.");
               }
            }
            else
            {
               throw new Exception ("Can't open " + sdFile + ".");
            }
         }
         else
         {
            frameMain.setStatusLabel ("Cancelled.");
         }
      }
      catch (Exception e)
      {
         frameMain.setStatusLabel (e.getMessage());
         if (log.isInfoEnabled())
         {
            log.info (e);
         }
         e.printStackTrace();
      }
      finally
      {
         frameMain.setCursorDefault();
         UMLProject.get().refreshDisplay();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statechart  No description provided
    * @param trace       No description provided
    * @return            No description provided
    */
   boolean mergeTraceIntoStatechart (UMLStatechart statechart, Trace trace)
   {
      boolean result = false;

      if (!statechartCanExecuteTrace (statechart, trace) &&
         !trace.isEmpty())
      {
         Iterator iter = statechart.iteratorOfElements();
         while (iter.hasNext() && result == false)
         {
            ASGElement asgElement = (ASGElement) iter.next();

            if (asgElement instanceof UMLComplexState)
            {
               UMLComplexState state = (UMLComplexState) asgElement;
               Trace traceForState = new Trace (trace);
               if (traceForState.startsWith (state))
               {
                  if (traceForState.getSize() != trace.getSize())
                  {
                     if (mergeTraceIntoStatechart (statechart, traceForState,
                        state, true, "   "))
                     {
                        result = true;
                     }
                  }
               }
            }
            else if (asgElement instanceof UMLTransition)
            {
               UMLTransition transition = (UMLTransition) asgElement;
               Trace traceForTransition = new Trace (trace);
               if (traceForTransition.startsWith (transition))
               {
                  if (traceForTransition.getSize() != trace.getSize())
                  {
                     if (mergeTraceIntoStatechart (statechart, traceForTransition,
                        transition, "   "))
                     {
                        result = true;
                     }
                  }
               }
            }
         }

         // If merging starting from existing states/transitions
         // didn't succeed, create a new state and merge from there.
         if (result == false)
         {
            // This may (if trace.first() is an InEvent) add
            // information not contained in the sequence diagram!  (the
            // state's action should be marked as "unspecified" or
            // "changeable", see below)
            UMLComplexState newState = trace.createState (statechart);

            if (mergeTraceIntoStatechart (statechart, trace, newState, true, "   "))
            {
               result = true;
            }
            else
            {
               newState.removeYou();
            }
         }
      }
      return result;
   }


   /**
    * Merge trace into statechart. state already exists and is already "walked along", i.e.
    * it is irrelevant for the further merging of trace.
    *
    * @param statechart                 No description provided
    * @param trace                      No description provided
    * @param state                      No description provided
    * @param allowNewLeavingTransition  No description provided
    * @param prefix                     No description provided
    * @return                           No description provided
    */
   boolean mergeTraceIntoStatechart (UMLStatechart statechart, Trace trace,
                                     UMLComplexState state,
                                     boolean allowNewLeavingTransition,
                                     String prefix)
   {
      // If the trace is empty, it's just fine, trace is completey
      // merged.
      if (trace.isEmpty())
      {
         if (log.isInfoEnabled())
         {
            log.info (prefix + "merge(null, " + state + ")");
         }

         if (log.isInfoEnabled())
         {
            log.info (prefix + "trace is empty, merge is ok");
         }
         return true;
      }
      else
      {
         if (log.isInfoEnabled())
         {
            log.info (prefix + "merge(" + trace.first() + ", " + state + ")");
         }

         // If there is a leaving transition with the incoming event
         // inEvent (there should only be one), the algorithm must
         // walk along this transition.  Otherwise a new transition
         // with an event appropriate for executing trace is created.
         Iterator iter = state.iteratorOfExit();
         while (iter.hasNext())
         {
            UMLTransition transition = (UMLTransition) iter.next();
            if (trace.startsWith (transition))
            {
               if (log.isInfoEnabled())
               {
                  log.info (prefix + "have to follow transition.");
               }

               return mergeTraceIntoStatechart (statechart, trace, transition, prefix + "   ");
            }
         }

         // else
         if (allowNewLeavingTransition)
         {
            if (log.isInfoEnabled())
            {
               log.info (prefix + "creating new transition.");
            }

            UMLTransition newTransition = trace.createTransition (statechart, state);
            if (newTransition != null)
            {
               if (mergeTraceIntoStatechart (statechart, trace, newTransition, prefix + "   "))
               {
                  return true;
               }
               else
               {
                  if (log.isInfoEnabled())
                  {
                     log.info (prefix + "merge failed, backtrack: remove transition from " + newTransition.getSource() + " to " + newTransition.getTarget() + ".");
                  }
                  newTransition.removeYou();
                  return false;
               }
            }
            else
            {
               if (log.isInfoEnabled())
               {
                  log.info (prefix + "merge failed, couldn't create transition from state " + state + ".");
               }
               return false;
            }
         }
         else
         {
            if (log.isInfoEnabled())
            {
               log.info (prefix + "merge failed, not allowed to create new leaving transition from state " + state + ".");
            }
            return false;
         }
      }
   }


   /**
    * Merge trace into statechart. transition already exists, maybe without a target state,
    * and is already "walked along", i.e. it is irrelevant for the further merging of trace.
    *
    * @param statechart  No description provided
    * @param trace       No description provided
    * @param transition  No description provided
    * @param prefix      No description provided
    * @return            No description provided
    */
   boolean mergeTraceIntoStatechart (UMLStatechart statechart, Trace trace,
                                     UMLTransition transition,
                                     String prefix)
   {
      // Try to find a "good" already existing target state (or create
      // one if none is found) for transition, in case it doesn't have
      // a target, yet.  Otherwise check if that target matches the
      // trace.

      // If the trace is empty and transition already has a target,
      // it's just fine, trace is completey merged.  Otherwise a new
      // state is created, as a transition must have a target state.
      // Note that this introduces extra information which is not
      // contained in the sequence diagram: in this case the sequence
      // diagram ends with an InEvent and doesn't force the do-Action
      // of the newly created state to be empty.  The state should
      // rather have an "unspecified" do-Action which may be modified
      // by this algorithm when merging another sequence diagram.  This
      // is not implemented, yet.  For now, sequence diagrams shouldn't
      // end with an incoming event (which may be considered as a bad
      // thing anyway, as the diagram doesn't specify a reaction for
      // this event, then, but it may of course happen while designing
      // such a diagram and applying this algorithm before it is
      // complete).
      if (trace.isEmpty())
      {
         if (log.isInfoEnabled())
         {
            log.info (prefix + "merge(null, " + transition + ")");
         }

         System.out.print (prefix + "trace is empty, ");
         if (transition.getTarget() == null)
         {
            if (log.isInfoEnabled())
            {
               log.info ("have to create new state, ");
            }
            UMLComplexState newState
                = NewStateAction.createState (statechart, "");
            transition.setTarget (newState);
            statechart.addToElements (newState);
         }
         if (log.isInfoEnabled())
         {
            log.info ("merge is ok.");
         }
         return true;
      }

      else
      {
         if (log.isInfoEnabled())
         {
            log.info (prefix + "merge(" + trace.first() + ", " + transition + ")");
         }

         // transition already has a target state.  This target
         // must match trace.  Otherwise the trace can't be merged.
         if (transition.getTarget() != null)
         {
            UMLComplexState target
                = (UMLComplexState) transition.getTarget();

            if (trace.startsWith (target))
            {
               if (log.isInfoEnabled())
               {
                  log.info (prefix + "transition has target, have to follow to state " + target);
               }

               return mergeTraceIntoStatechart (statechart, trace, target, true, prefix + "   ");
            }
            else
            {
               if (log.isInfoEnabled())
               {
                  log.info (prefix + "merge failed, target " + target + " doesn't match trace.");
               }
               return false;
            }
         }

         // transition doesn't have a target, yet.  So either find
         // a "good" target or create a new one.  A "good" target
         // is a state which matches the trace.  Note that a state
         // without any actions can also be worth a try, as a
         // leaving transition may have outEvent as action.  The
         // checking of transition-actions is not implemented, yet,
         // but states without any do-Actions are already
         // considered.  All such states are tested as target.  If
         // all these possible exisiting targets fail to do the
         // merge, a new state is created.
         else
         {
            if (log.isInfoEnabled())
            {
               log.info (prefix + "transition has no target");
            }

            Iterator iter = statechart.iteratorOfElements();
            while (iter.hasNext())
            {
               ASGElement asgElement = (ASGElement) iter.next();
               if (asgElement instanceof UMLComplexState &&
                  asgElement != statechart.getInitialState())
               {
                  UMLComplexState state = (UMLComplexState) asgElement;

                  Trace traceForState = new Trace (trace);

                  if (traceForState.startsWith (state))
                  {
                     if (log.isInfoEnabled())
                     {
                        log.info (prefix + "trying target " + state);
                     }

                     transition.setTarget (state);

                     // Note: if this merge creates a new transition t
                     // from state to somewhere there are both a newly
                     // created incoming transition (namely this method's
                     // argument transition) and a newly created outgoing
                     // transition (namely t).  Probably this might
                     // generally be considered a bad thing, as it may
                     // "misuse" state.  (Introduce another parameter?
                     // Like "boolean allowNewTransition" and pass false
                     // here as value?)
                     if (mergeTraceIntoStatechart (statechart, traceForState,
                        state, true, prefix + "   "))
                     {
                        return true;
                     }
                     else
                     {
                        if (log.isInfoEnabled())
                        {
                           log.info (prefix + "target " + state + " failed.");
                        }
                        transition.setTarget (null);
                     }
                  }
               }
            }

            // else (every possible existing state checked) create
            // new state.  If even that last resort fails, backtrack.
            UMLComplexState newState = trace.createState (statechart);

            if (log.isInfoEnabled())
            {
               log.info (prefix + "all targets tried, creating new state " + newState);
            }

            transition.setTarget (newState);
            if (mergeTraceIntoStatechart (statechart, trace, newState, true, prefix + "   "))
            {
               return true;
            }
            else
            {
               if (log.isInfoEnabled())
               {
                  log.info (prefix + "merge failed, backtrack: remove state " + newState + ".");
               }
               transition.setTarget (null);
               newState.removeYou();
               return false;
            }
         }
      }
   }


   // FIXME: implement
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statechart  No description provided
    * @param trace       No description provided
    * @return            No description provided
    */
   boolean statechartCanExecuteTrace (UMLStatechart statechart, Trace trace)
   {
      if (trace.isEmpty())
      {
         return true;
      }
      else
      {
         return false;
      }
   }


   /**
    * Collect all participating objects in a sequence diagram and return them. When fujaba
    * sequence diagrams will finally be used, only objects of the correct class should be offered
    * for selection of course.
    *
    * @param sdFile  No description provided
    * @return        No description provided
    */
   private Collection sequenceDiagramObjects (File sdFile)
   {
      TreeSet diagramObjects = new TreeSet();

      Document document = openSequenceDiagram (sdFile);
      if (document != null)
      {
         NodeList lst = document.getElementsByTagName ("*");
         for (int i = 0; i < lst.getLength(); ++i)
         {
            Node node = lst.item (i);
            if (node.getNodeType() == Node.ELEMENT_NODE)
            {
               if (node.getNodeName().equals ("sender") ||
                  node.getNodeName().equals ("receiver"))
               {
                  String name = node.getFirstChild().getNodeValue().trim();
                  diagramObjects.add (name);
               }
            }
         }
      }

      return diagramObjects;
   }


   /**
    * Create a trace for an object in a sequence diagram. This trace contains all the information
    * needed for the specified tracedObject, namely which events arrive and which are sent
    * and so on.
    *
    * @param file          No description provided
    * @param tracedObject  No description provided
    * @return              No description provided
    */
   private Trace createSequenceDiagramTrace (File file, String tracedObject)
   {
      Trace trace = null;

      Document document = openSequenceDiagram (file);
      if (document != null)
      {
         NodeList lst = document.getElementsByTagName ("*");
         trace = new Trace();
         int length = lst.getLength();
         for (int i = 0; i < length; ++i)
         {
            Node node = lst.item (i);
            if (node.getNodeType() == Node.ELEMENT_NODE)
            {
               if (node.getNodeName().equals ("event"))
               {
                  Event event = createEvent (node, tracedObject);
                  if (event != null)
                  {
                     trace.appendItem (event);
                  }
               }
               else if (node.getNodeName().equals ("state"))
               {
                  State state = createState (node);
                  if (state != null)
                  {
                     trace.appendItem (state);
                  }
               }
            }
         }
      }

      return trace;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param file  No description provided
    * @return      No description provided
    */
   private Document openSequenceDiagram (File file)
   {
      Document document = null;
      DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      factory.setValidating (true);
      factory.setNamespaceAware (true);
      try
      {
         DocumentBuilder builder = factory.newDocumentBuilder();
         // FIXME: ErrorHandler should be set:
         // builder.setErrorHandler(own ErrorHandler);

         document = builder.parse (file);
         DocumentType doctype = document.getDoctype();
         boolean valid = false;
         if (doctype != null)
         {
            if (doctype.getName().equals ("sequence-diagram"))
            {
               valid = true;
            }
         }
         if (valid)
         {
            document.normalize();
         }
         else
         {
            throw new RuntimeException ("invalid xml-document");
         }
      }
      catch (SAXException sxe)
      {
         // Error generated during parsing)
         Exception x = sxe;
         if (sxe.getException() != null)
         {
            x = sxe.getException();
         }
         x.printStackTrace();
      }
      catch (ParserConfigurationException pce)
      {
         // Parser with specified options can't be built
         pce.printStackTrace();
      }
      catch (IOException ioe)
      {
         // I/O error
         ioe.printStackTrace();
      }

      return document;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param eventNode     No description provided
    * @param tracedObject  No description provided
    * @return              No description provided
    */
   private Event createEvent (Node eventNode, String tracedObject)
   {
      String name = "";
      String sender = "";
      String receiver = "";

      NodeList lst = eventNode.getChildNodes();
      int length = lst.getLength();
      for (int i = 0; i < length; ++i)
      {
         Node node = lst.item (i);
         if (node.getNodeType() == Node.ELEMENT_NODE)
         {
            if (node.getNodeName().equals ("name"))
            {
               name = node.getFirstChild().getNodeValue().trim();
            }
            if (node.getNodeName().equals ("sender"))
            {
               sender = node.getFirstChild().getNodeValue().trim();
            }
            if (node.getNodeName().equals ("receiver"))
            {
               receiver = node.getFirstChild().getNodeValue().trim();
            }
         }
      }

      Event result = null;

      // FIXME: what if sender==receiver?  OutEvent followed by InEvent?
      if (sender.equals (receiver))
      {
         throw new
            RuntimeException ("sender==receiver not yet supported for"
            + " events (" + name + " from/to " + sender + ")");
      }
      // outgoing event
      else if (sender.equals (tracedObject))
      {
         result = new OutEvent (name, receiver);
      }
      // incoming event (= transition label)
      else if (receiver.equals (tracedObject))
      {
         result = new InEvent (name, sender);
      }

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param stateNode  No description provided
    * @return           No description provided
    */
   private State createState (Node stateNode)
   {
      String name = "";

      NodeList lst = stateNode.getChildNodes();
      int length = lst.getLength();
      for (int i = 0; i < length; ++i)
      {
         Node node = lst.item (i);
         if (node.getNodeType() == Node.ELEMENT_NODE)
         {
            if (node.getNodeName().equals ("name"))
            {
               name = node.getFirstChild().getNodeValue().trim();
            }
         }
      }

      return new State (name);
   }
}


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.19.2.1 $
 */
class TraceItem
{


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param action  No description provided
    * @return        No description provided
    */
   public boolean matchesAction (String action)
   {
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    * @return       No description provided
    */
   public boolean matchesEvent (String event)
   {
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statename  No description provided
    * @return           No description provided
    */
   public boolean matchesStatename (String statename)
   {
      return false;
   }
}


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.19.2.1 $
 */
class State extends TraceItem
{


   /**
    * Constructor for class State
    *
    * @param name  No description provided
    */
   public State (String name)
   {
      setName (name);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      String result = "State(" + name + ")";
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name;


   /**
    * Get the name attribute of the State object
    *
    * @return   The name value
    */
   public String getName()
   {
      return name;
   }


   /**
    * Sets the name attribute of the State object
    *
    * @param n  The new name value
    */
   public void setName (String n)
   {
      name = n;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statename  No description provided
    * @return           No description provided
    */
   public boolean matchesStatename (String statename)
   {
      return getName().equals (statename);
   }
}


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.19.2.1 $
 */
class Event extends TraceItem
{


   /**
    * Constructor for class Event
    *
    * @param name  No description provided
    */
   public Event (String name)
   {
      setName (name);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      String result = "Event(" + name + ")";
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name;


   /**
    * Get the name attribute of the Event object
    *
    * @return   The name value
    */
   public String getName()
   {
      return name;
   }


   /**
    * Sets the name attribute of the Event object
    *
    * @param n  The new name value
    */
   public void setName (String n)
   {
      name = n;
   }
}


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.19.2.1 $
 */
class InEvent extends Event
{


   /**
    * Constructor for class InEvent
    *
    * @param name    No description provided
    * @param sender  No description provided
    */
   public InEvent (String name, String sender)
   {
      super (name);
      setSender (sender);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      String result = "InEvent(" + getName() + " from " + sender + ")";
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String sender;


   /**
    * Get the sender attribute of the InEvent object
    *
    * @return   The sender value
    */
   public String getSender()
   {
      return sender;
   }


   /**
    * Sets the sender attribute of the InEvent object
    *
    * @param s  The new sender value
    */
   public void setSender (String s)
   {
      sender = s;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    * @return       No description provided
    */
   public boolean matchesEvent (String event)
   {
      return getName().equals (event);
   }
}



/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.19.2.1 $
 */
class OutEvent extends Event
{


   /**
    * Constructor for class OutEvent
    *
    * @param name      No description provided
    * @param receiver  No description provided
    */
   public OutEvent (String name, String receiver)
   {
      super (name);
      setReceiver (receiver);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      String result = "OutEvent(" + getName() + " to " + receiver + ")";
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String receiver;


   /**
    * Get the receiver attribute of the OutEvent object
    *
    * @return   The receiver value
    */
   public String getReceiver()
   {
      return receiver;
   }


   /**
    * Sets the receiver attribute of the OutEvent object
    *
    * @param r  The new receiver value
    */
   public void setReceiver (String r)
   {
      receiver = r;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param action  No description provided
    * @return        No description provided
    */
   public boolean matchesAction (String action)
   {
      return getName().equals (action);
   }
}



/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.19.2.1 $
 */
class Trace
{


   /**
    * Constructor for class Trace
    */
   public Trace() { }


   /**
    * Constructor for class Trace
    *
    * @param t  No description provided
    */
   public Trace (Trace t)
   {
      items = new LinkedList (t.items);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private LinkedList items = new LinkedList();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      String result = "[";
      Iterator iter = items.iterator();
      while (iter.hasNext())
      {
         result += iter.next();
         if (iter.hasNext())
         {
            result += ",\n ";
         }
      }
      result += "]";
      return result;
   }


   /**
    * Get the empty attribute of the Trace object
    *
    * @return   The empty value
    */
   public boolean isEmpty()
   {
      return  (items.size() == 0);
   }


   /**
    * Get the size attribute of the Trace object
    *
    * @return   The size value
    */
   public int getSize()
   {
      return items.size();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfItems()
   {
      return items.iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public TraceItem first()
   {
      return (TraceItem) items.get (0);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeFirst()
   {
      items.remove (0);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param item  No description provided
    */
   public void appendItem (TraceItem item)
   {
      items.add (item);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param state  No description provided
    * @return       No description provided
    */
   public boolean matches (UMLComplexState state)
   {
      return matchremove (state, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param state  No description provided
    * @return       No description provided
    */
   public boolean remove (UMLComplexState state)
   {
      return matchremove (state, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param state   No description provided
    * @param remove  No description provided
    * @return        No description provided
    */
   private boolean matchremove (UMLComplexState state, boolean remove)
   {
      boolean result = false;

      if (state.hasDoAction())
      {
         if (first() instanceof OutEvent)
         {
            if ( ((OutEvent) first()).getName().equals (state.getDoAction()))
            {
               if (remove)
               {
                  removeFirst();
               }
               result = true;
            }
         }
      }
      else
      {
         result = true;
      }

      return result;
   }


   /**
    * Check if state can be entered when executing trace. SIDE-EFFECT: trace is shortened as
    * if state was entered.
    *
    * @param state  No description provided
    * @return       No description provided
    */
   public boolean startsWith (UMLComplexState state)
   {
      boolean result = false;

      String doAction = state.getDoAction();
      if (doAction == null)
      {
         result = true;
      }
      else if (doAction.equals (""))
      {
         result = true;
      }
      else
      {
         TraceItem traceitem = this.first();
         if (traceitem.matchesAction (doAction))
         {
            result = true;
            this.removeFirst();
         }
      }

      return result;
   }


   /**
    * Check if transition can be followed along when executing trace. SIDE-EFFECT: trace is
    * shortened as if transition was taken.
    *
    * @param transition  No description provided
    * @return            No description provided
    */
   public boolean startsWith (UMLTransition transition)
   {
      boolean result = false;

      String event = transition.getEvent();
      if (event == null)
      {
         result = true;
      }
      else if (event.equals (""))
      {
         result = true;
      }
      else
      {
         TraceItem traceitem = this.first();
         if (traceitem.matchesEvent (event))
         {
            result = true;
            this.removeFirst();
         }
      }

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statechart  No description provided
    * @param source      No description provided
    * @return            No description provided
    */
   public UMLTransition createTransition (UMLStatechart statechart,
                                          UMLComplexState source)
   {
      UMLTransition newTransition = null;
      InEvent inEvent = null;

      if (this.first() instanceof InEvent)
      {
         // Only if there is no empty transition, another one can be created.
         if (getLeavingTransitionHavingNoEvent (source) == null)
         {
            inEvent = (InEvent) this.first();
            this.removeFirst();
            newTransition = new UMLTransition();
            newTransition.setSource (source);
            newTransition.setEvent (inEvent.getName());
            statechart.addToElements (newTransition);
         }
      }
      else
      {
         // Only if there are no transitions at all, an empty one can be
         // created.
         if (!hasLeavingTransitions (source))
         {
            newTransition = new UMLTransition();
            newTransition.setSource (source);
            newTransition.setEvent ("");
            statechart.addToElements (newTransition);
         }
      }

      return newTransition;
   }


   /**
    * Get the leavingTransitionHavingNoEvent attribute of the Trace object
    *
    * @param state  No description provided
    * @return       The leavingTransitionHavingNoEvent value
    */
   public UMLTransition getLeavingTransitionHavingNoEvent (UMLComplexState state)
   {
      UMLTransition result = null;

      Iterator iter = state.iteratorOfExit();
      while (iter.hasNext() && result == null)
      {
         UMLTransition transition = (UMLTransition) iter.next();
         if (!transition.hasEvent())
         {
            result = transition;
         }
      }

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param state  No description provided
    * @return       No description provided
    */
   public boolean hasLeavingTransitions (UMLComplexState state)
   {
      return state.sizeOfExit() > 0;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statechart  No description provided
    * @return            No description provided
    */
   public UMLComplexState createState (UMLStatechart statechart)
   {
      UMLComplexState newState = NewStateAction.createState (statechart, "");

      if (this.first() instanceof OutEvent)
      {
         OutEvent outEvent = (OutEvent) this.first();
         this.removeFirst();
         newState.setDoAction (outEvent.getName());
      }

      statechart.addToElements (newState);
      return newState;
   }

}

/*
 * $Log: MergeSequenceDiagramIntoStateChartAction.java,v $
 * Revision 1.19.2.1  2005/09/30 18:56:52  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
