/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml.actions;

import java.awt.event.ActionEvent;
import java.util.HashSet;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.*;

import de.uni_paderborn.fujaba.app.action.UpdateProjectTreeAction;
import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.codegen.*;
import de.uni_paderborn.fujaba.metamodel.FDeclaration;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.uni_paderborn.fujaba.uml.*;


/**
 * @author    <a href="mailto:creckord@upb.de">Carsten Reckord</a>
 * @version   $Revision: 1.7 $
 */
public class RemoveGeneratedMethodsAction extends AbstractAction
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      Iterator iter = findClasses();
      while (iter.hasNext())
      {
         UMLClass umlClass = (UMLClass) iter.next();
         removeGeneratedMethods (umlClass);
      }
      iter = findStatecharts();
      while (iter.hasNext())
      {
         UMLActivityDiagram statechart = (UMLActivityDiagram) iter.next();
         removeStatechartMethods (statechart);
      }
      new UpdateProjectTreeAction().actionPerformed (null);
   }


   /**
    * @param umlClass
    */
   private void removeGeneratedMethods (UMLClass umlClass)
   {
      removeRemoveYouMethod (umlClass);
      removeRoleAccessors (umlClass);
      removeAttrAccessors (umlClass);
      removeOldStatechartMethods (umlClass);
   }


   /**
    * @param umlClass
    */
   private void removeAttrAccessors (UMLClass umlClass)
   {
      UMLAttr attr;
      AccessorOOHandler handler = null;
      Iterator attrIter = umlClass.iteratorOfAttrs();
      if (attrIter.hasNext())
      {
         attr = (UMLAttr) attrIter.next();
         handler = findAccessorOOHandler (attr);
         removeAttrAccessors (attr, handler);

         while (attrIter.hasNext())
         {
            attr = (UMLAttr) attrIter.next();
            removeAttrAccessors (attr, handler);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param attr     No description provided
    * @param handler  No description provided
    */
   private void removeAttrAccessors (UMLAttr attr, AccessorOOHandler handler)
   {
      if (handler != null)
      {
         handler.removeOldAccessors (attr);
      }
      Iterator accessors = attr.iteratorOfAccessMethods();
      while (accessors.hasNext())
      {
         UMLMethod method = (UMLMethod) accessors.next();
         removeMethod (method);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlMethod  No description provided
    */
   private void removeMethod (UMLMethod umlMethod)
   {
      UMLActivityDiagram activityDiag = umlMethod.getStoryDiagram();
      umlMethod.removeYou();
      if (activityDiag != null)
      {
         activityDiag.removeYou();
      }
   }


   /**
    * @param umlClass
    */
   private void removeRoleAccessors (UMLClass umlClass)
   {
      UMLRole role;
      AccessorOOHandler handler = null;
      Iterator roleIter = umlClass.iteratorOfRoles();

      if (roleIter.hasNext())
      {
         role = (UMLRole) roleIter.next();
         handler = findAccessorOOHandler (role);
         removeRoleAccessors (role, handler);

         while (roleIter.hasNext())
         {
            role = (UMLRole) roleIter.next();
            removeRoleAccessors (role, handler);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param role     No description provided
    * @param handler  No description provided
    */
   private void removeRoleAccessors (UMLRole role, AccessorOOHandler handler)
   {
      if (handler != null)
      {
         handler.removeOldAccessors (role);
      }
      UMLAttr attr = role.getAssociatedAttribute();
      if (attr != null)
      {
         Iterator accessors = attr.iteratorOfAccessMethods();
         while (accessors.hasNext())
         {
            UMLMethod method = (UMLMethod) accessors.next();
            removeMethod (method);
         }
         attr.removeYou();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param element  No description provided
    * @return         No description provided
    */
   private AccessorOOHandler findAccessorOOHandler (ASGElement element)
   {
      Iterator iter = CodeGenFactory.get().getCurrentStrategy().iteratorOfHandlerChain();
      boolean done = false;

      while (!done && iter.hasNext())
      {
         CodeGenStrategyHandler nextHandler = (CodeGenStrategyHandler) iter.next();

         if (nextHandler instanceof AccessorOOHandler && nextHandler.isResponsible ((FElement) element))
         {
            return (AccessorOOHandler) nextHandler;
         }
      }
      return null;
   }


   /**
    * @param umlClass
    */
   private void removeRemoveYouMethod (UMLClass umlClass)
   {
      UMLMethod removeMethod = umlClass.getFromMethodsByShortName ("removeYou");
      if (removeMethod != null)
      {
         removeMethod (removeMethod);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statechart  No description provided
    */
   private void removeStatechartMethods (UMLActivityDiagram statechart)
   {
      UMLStatechartOOHandler handler = findStatechartOOHandler (statechart);
      if (handler != null)
      {
         handler.removeGeneratedElements (statechart);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param umlClass  No description provided
    */
   private void removeOldStatechartMethods (UMLClass umlClass)
   {
      Pattern namePattern = Pattern.compile ("^(?:entryAction\\d*.+|exitAction\\\\d*.+|initState[cC]hart|" +
         "action\\d+For.+From.+To.+|after\\d+|alwaysTrue|doAction\\d+.+|" +
         "waitForResultForEvent|notifyMe|when\\d+|guard\\d+For.+From.+To.+)$");
      Matcher matcher = namePattern.matcher ("");

      Iterator methods = umlClass.iteratorOfMethods();
      while (methods.hasNext())
      {
         UMLMethod umlMethod = (UMLMethod) methods.next();

         if (umlMethod.isGenerated() || umlMethod.getDisplayLevel() <= FDeclaration.CODE_DISPLAY_LEVEL)
         {
            String name = umlMethod.getName();
            matcher.reset (name);
            if (matcher.matches())
            {
               removeMethod (umlMethod);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param statechart  No description provided
    * @return            No description provided
    */
   private UMLStatechartOOHandler findStatechartOOHandler (UMLActivityDiagram statechart)
   {
      Iterator iter = CodeGenFactory.get().getCurrentStrategy().iteratorOfHandlerChain();
      boolean done = false;

      while (!done && iter.hasNext())
      {
         CodeGenStrategyHandler nextHandler = (CodeGenStrategyHandler) iter.next();

         if (nextHandler instanceof UMLStatechartOOHandler && nextHandler.isResponsible ((FElement) statechart))
         {
            return (UMLStatechartOOHandler) nextHandler;
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private Iterator findClasses()
   {
      HashSet classes = new HashSet();

      Iterator diagIter = UMLProject.get().iteratorOfDiags();
      while (diagIter.hasNext())
      {
         ASGDiagram diag = (ASGDiagram) diagIter.next();
         if (diag instanceof UMLClassDiagram)
         {
            UMLClassDiagram classDiag = (UMLClassDiagram) diag;

            Iterator elems = classDiag.iteratorOfElements();
            while (elems.hasNext())
            {
               ASGElement elem = (ASGElement) elems.next();
               if (elem instanceof UMLClass)
               {
                  classes.add (elem);
               }
            }
         }
      }
      return classes.iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private Iterator findStatecharts()
   {
      HashSet statecharts = new HashSet();

      Iterator diagIter = UMLProject.get().iteratorOfDiags();
      while (diagIter.hasNext())
      {
         ASGDiagram diag = (ASGDiagram) diagIter.next();
         //if (diag instanceof UMLStatechart)
         if (diag instanceof UMLActivityDiagram &&  ((UMLActivityDiagram) diag).isStatechart())
         {
            statecharts.add (diag);
         }
      }
      return statecharts.iterator();
   }
}

/*
 * $Log: RemoveGeneratedMethodsAction.java,v $
 * Revision 1.7  2004/10/20 19:51:23  schneider
 * fixed doubled constants - UMLDeclarationInterface deprecated
 *
 */
