/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml.utility;

import java.util.*;

import de.uni_paderborn.fujaba.asg.*;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FTreeSet;


/**
 * Utility class for reusable UMLClassDiagram methodes.
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.20 $
 */
public class UMLClassDiagramUtility extends ASGUtility
{

   /**
    * Constructor for abstract class ASGUtility
    *
    * @param diagram  No description provided
    */
   public UMLClassDiagramUtility (ASGDiagram diagram)
   {
      super (diagram);
   }


   /**
    * Only for tests, will be removed.
    *
    * @param class1  The new ofAssociations value
    * @param class2  The new ofAssociations value
    * @return        No description provided
    */
   public static Set setOfAssociations (UMLClass class1, UMLClass class2)
   {
      HashSet set = new HashSet();

      if (class1 != null && class2 != null)
      {
         Iterator iter = class1.iteratorOfRoles();
         while (iter.hasNext())
         {
            UMLRole leftRole = (UMLRole) iter.next();
            UMLAssoc assoc = leftRole.getAssoc();
            UMLRole rightRole = assoc.getPartnerRole (leftRole);

            if (rightRole.getTarget() == class2)
            {
               set.add (assoc);
            }
         }

         UMLGeneralization generalization;
         UMLClass superClass;

         iter = class1.iteratorOfRevSubclass();
         while (iter.hasNext())
         {
            generalization = (UMLGeneralization) iter.next();
            superClass = generalization.getSuperclass();
            set.addAll (setOfAssociations (superClass, class2));
         }

         iter = class2.iteratorOfRevSubclass();
         while (iter.hasNext())
         {
            generalization = (UMLGeneralization) iter.next();
            superClass = generalization.getSuperclass();
            set.addAll (setOfAssociations (class1, superClass));
         }
      }

      return set;
   }


   /**
    * Calculates all associations between two classes and returns the list of associations.
    * Inherited assocs will be also considered.
    *
    * @param class1                     the first class.
    * @param class2                     the second class.
    * @return                           An FTreeSet containing all possible associations between the specified classes.
    * @throws IllegalArgumentException  If at least one of the parameters is null.
    */
   public static FTreeSet calculatePossibleAssocs (UMLClass class1, UMLClass class2) throws IllegalArgumentException
   {
      if ( (class1 == null) ||  (class2 == null))
      {
         throw new IllegalArgumentException ("At least one argument is null!");
      }

      Iterator iter;
      FTreeSet intersection = new FTreeSet();
      UMLAssoc assoc = null;

      // Calculate the intersection of assocsFromClass1 and
      // assocsFromClass2.  The intersection is the set of
      // associations which exist between both class1 and class2.
      FTreeSet assocsFromClass1 = class1.getAllAssocs();

      iter = assocsFromClass1.iterator();
      while (iter.hasNext())
      {
         assoc = (UMLAssoc) iter.next();

         if (class1.isChildOf (assoc.getLeftRole().getTarget())
            && class2.isChildOf (assoc.getRightRole().getTarget()))
         {
            intersection.add (assoc);
            continue;
         }
         if (class1.isChildOf (assoc.getRightRole().getTarget())
            && class2.isChildOf (assoc.getLeftRole().getTarget()))
         {
            intersection.add (assoc);
            continue;
         }
      }

      return intersection;
   } // calculatePossibleAssocs


   /**
    * Calculates all associations between two classes via ASGElementRef and returns the list
    * of associations.
    *
    * @param refClass   No description provided
    * @param elemClass  No description provided
    * @return           No description provided
    */
   public static FTreeSet calculatePossibleAssocsViaASGElementRef (UMLClass refClass, UMLClass elemClass)
   {
      if ( (refClass == null) ||  (elemClass == null) ||  (refClass == elemClass)
         ||  (isDerivedfrom (refClass, "ASGElementRef")))
      {
         return null;
      }

      Iterator assocIter;
      Iterator refClassIter;

      Object obj = null;

      FTreeSet assocs = new FTreeSet();
      FTreeSet refClasses = new FTreeSet();

      UMLClass reachableRefClass;

      // ---- all associations from the supposed reference class
      FTreeSet assocsFromClass = refClass.getAllAssocs();
      assocIter = assocsFromClass.iterator();
      while (assocIter.hasNext())
      {
         obj = assocIter.next();
         UMLAssoc assoc = (UMLAssoc) obj;

         refClasses.add (assoc.getLeftRole().getTarget());
         refClasses.add (assoc.getRightRole().getTarget());

         // ---- iterator of all classes reachable via
         // ---- an associations from the supposed reference class
         refClassIter = refClasses.iterator();
         while (refClassIter.hasNext())
         {
            reachableRefClass = (UMLClass) refClassIter.next();

            if (isDerivedfrom (reachableRefClass, "ASGElementRef"))
            {
               // ---- iterator of all generalizations from the supposed element class
               if (isDerivedfrom (elemClass, "ASGElement"))
               {
                  assocs.add (obj);
               } // end if
            } // end if
         } // end while
      } // end while

      if (assocs.isEmpty())
      {
         return null;
      }
      else
      {
         return assocs;
      }

   } // end calculatePossibleAssocsViaASGElementRef


   /**
    * @param cls           No description provided
    * @param superClsName  No description provided
    * @return              The derivedfrom value
    */
   public static boolean isDerivedfrom (UMLClass cls, String superClsName)
   {
      boolean result = false;
      From superClsIter = (From) cls.iteratorOfSuperClasses();

      while (superClsIter.hasNext())
      {
         UMLClass superCls = (UMLClass) superClsIter.next();
         if (superCls.getName().equals (superClsName))
         {
            return true;
         }
         else
         {
            result = isDerivedfrom (superCls, superClsName);
         }
      }

      return result;
   }



   /**
    * returns the association for a given association name.
    *
    * @param assocName  No description provided
    * @return           The associations value
    */
   public static UMLAssoc getAssoc (String assocName)
   {
      UMLClassDiagram clazzDiagram = null;
      UMLAssoc foundAssoc = null;

      Iterator diagIter = UMLProject.get().iteratorOfDiags();

      while (diagIter.hasNext() &&  (null == clazzDiagram))
      {
         ASGDiagram diagram = (ASGDiagram) diagIter.next();
         if (diagram instanceof UMLClassDiagram)
         {
            clazzDiagram = (UMLClassDiagram) diagram;

            Iterator iter = clazzDiagram.iteratorOfElements();
            while ( (iter.hasNext()) &&  (null == foundAssoc))
            {
               ASGElement asgElement = (ASGElement) iter.next();
               if (asgElement instanceof UMLAssoc)
               {
                  UMLAssoc assoc = (UMLAssoc) asgElement;
                  if ( (assoc.getName()).equals (assocName))
                  {
                     foundAssoc = assoc;
                  }
               }
            }

            if (null == foundAssoc)
            {
               clazzDiagram = null;
            } // if

         } // if
      } // while

      return foundAssoc;
   } // getAssoc


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param class1     No description provided
    * @param class2     No description provided
    * @param assocName  No description provided
    * @return           No description provided
    */
   public static UMLAssoc searchForAssoc (UMLClass class1, UMLClass class2, String assocName)
   {
      UMLAssoc foundAssoc = null;
      boolean found = false;

      Iterator iter = calculatePossibleAssocs (class1, class2).iterator();
      while (iter.hasNext() && !found)
      {
         UMLAssoc currentAssoc = (UMLAssoc) iter.next();
         if (currentAssoc.getName().equals (assocName))
         {
            foundAssoc = currentAssoc;
            found = true;
         }
      }

      return foundAssoc;
   } // searchForAssoc (UMLClass, UMLClass, String)


   /**
    * removes tmpMethod with has the same name (not signature!) as UMLMethod method
    *
    * @param cls     No description provided
    * @param method  No description provided
    */
   public static void removeMethodBySameName (UMLClass cls, UMLMethod method)
   {
      UMLMethod tmpMethod = null;
      Iterator iter = cls.iteratorOfMethods();
      boolean found = false;

      if (method != null)
      {
         while (!found && iter.hasNext())
         {
            tmpMethod = (UMLMethod) iter.next();
            found =  (tmpMethod.getName().equals (method.getName()));
         }
      }

      if ( (method != null) && found)
      {
         tmpMethod.setParent (null);
         tmpMethod.removeYou();
      }

   } // end removeMethodBySameName

}

/*
 * $Log: UMLClassDiagramUtility.java,v $
 * Revision 1.20  2005/02/01 15:21:29  fklar
 * Fixed a bug in calculatePossibleAssocs that occured if class1 was a child of
 * leftRoleTarget AND rightRoleTarget and class2 was only a child of leftRoleTarget.
 * E.g. if class1 is subclass of ASGElementRef (which is a subclass of ASGElement) and class2
 * is a subclass of ASGElement.
 * In this case the assoc between both classes wasn't recognized as an intersection.
 *
 */
