/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views;

import java.util.*;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.BasicIncrement;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashSet;


/**
 * UMLClass: 'AbstractViewFilter'
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.14 $
 */
public abstract class AbstractFilter extends BasicIncrement implements Filter
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag     No description provided
    * @param item     No description provided
    * @param context  No description provided
    * @return         No description provided
    */
   public abstract Set grow1Context (UMLDiagram diag, UMLDiagramItem item, ItemSet context);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag  No description provided
    * @param item  No description provided
    * @return      No description provided
    */
   public Set get1Context (UMLDiagram diag, UMLDiagramItem item)
   {
      return getNContext (diag, item, 1);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag   No description provided
    * @param items  No description provided
    * @return       No description provided
    */
   public Set get1Context (UMLDiagram diag, Iterator items)
   {
      return getNContext (diag, items, 1);
   }


   /**
    * Get the nContext attribute of the AbstractFilter object
    *
    * @param diag   No description provided
    * @param item   No description provided
    * @param depth  No description provided
    * @return       The nContext value
    */
   public Set getNContext (UMLDiagram diag, UMLDiagramItem item, int depth)
   {
      Set items = new HashSet();
      items.add (item);

      return getNContext (diag, items.iterator(), depth);
   }


   /**
    * Get the nContext attribute of the AbstractFilter object
    *
    * @param diag   No description provided
    * @param items  No description provided
    * @param depth  No description provided
    * @return       The nContext value
    */
   public Set getNContext (UMLDiagram diag, Iterator items, int depth)
   {
      ItemSet result = new ItemSet();
      Set context = null;
      Set tmpSet = null;
      while (items.hasNext())
      {

         Object tmpObj = items.next();
         if (! (tmpObj instanceof UMLDiagramItem) || !diag.hasInElements ((UMLDiagramItem) tmpObj))
         {
            continue;
         }

         UMLDiagramItem item = (UMLDiagramItem) tmpObj;

         boolean grows = true;

         context = new ItemSet();
         context.add (item);
         result.add (item);

         for (int i = depth; i != 0 && grows; i--)
         {
            Iterator contextIterator = context.iterator();
            context = new ItemSet();
            while (contextIterator.hasNext())
            {
               tmpSet = grow1Context (diag, (UMLDiagramItem) contextIterator.next(), result);

               if (tmpSet != null)
               {
                  context.addAll (tmpSet);
               }
            }
            grows =  (context != null && !context.isEmpty());
         }
      }
      return result;
   }


   /**
    * Get the name attribute of the AbstractFilter object
    *
    * @return   The name value
    */
   public String getName()
   {
      String result = this.getClass().getName();
      int nameStart = result.lastIndexOf (".") + 1;

      return result.substring (nameStart);
   }


   /**
    * @return   same as getName()
    */
   public String toString()
   {
      return getName();
   }


   /**
    * Get all UMLConnections in Diagram diag which are of class type and which have a start-
    * and end-item which are in the items-Set
    *
    * @param diag   the diagram
    * @param items  the items whose connections are seeked for
    * @param type   the type of allowed connections
    * @return       the connections as Set
    */
   public static Set getConnections (UMLDiagram diag, Set items, Class type)
   {
      if (type == null)
      {
         type = UMLConnection.class;
      }
      else if (!UMLConnection.class.isAssignableFrom (type))
      {
         throw new IllegalArgumentException ("Field type is not a subtype of UMLConnection");
      }

      FHashSet connections = new FHashSet();

      Iterator diagIter = diag.iteratorOfElements();
      while (diagIter.hasNext())
      {
         ASGElement asgElement = (ASGElement) diagIter.next();

         if (type.isAssignableFrom (asgElement.getClass()))
         {
            UMLConnection con = (UMLConnection) asgElement;

            if (items.contains (con.getSourceConnector()) && items.contains (con.getTargetConnector()))
            {
               connections.add (asgElement);
            }
         }
      }
      return connections;
   }


   /**
    * @param diag        the diagram
    * @param connection  the connection
    * @return            true if the diagram contains both ends of the connection
    */
   public static boolean containsConnectors (UMLDiagram diag, UMLConnection connection)
   {
      return  (diag != null && connection != null && diag.hasInElements (connection.getSourceConnector())
         && diag.hasInElements (connection.getTargetConnector()));
   }


   /**
    * @param diag        the diagram
    * @param connection  the connection
    * @return            true if the diagram contains both ends of the connection and the connection
    *      itself
    */
   public static boolean containsConnection (UMLDiagram diag, UMLConnection connection)
   {
      return  (diag != null && connection != null && diag.hasInElements (connection) &&
         diag.hasInElements (connection.getSourceConnector()) && diag.hasInElements (connection.getTargetConnector()));
   }


   /**
    * @param context     the context
    * @param connection  the connection
    * @return            true if the context contains both ends of the connection
    */
   public static boolean containsConnectors (Set context, UMLConnection connection)
   {
      return  (context != null && connection != null && context.contains (connection.getSourceConnector())
         && context.contains (connection.getTargetConnector()));
   }


   /**
    * @param context     the context
    * @param connection  the connection
    * @return            true if the context contains both ends of the connection and the connection
    *      itself
    */
   public static boolean containsConnection (Set context, UMLConnection connection)
   {
      return  (context != null && connection != null && context.contains (connection) &&
         context.contains (connection.getSourceConnector()) &&
         context.contains (connection.getTargetConnector()));
   }


   /**
    * Add the connection to the context if both ends of the connection are already in the context
    *
    * @param context     the context
    * @param connection  the connection
    * @return            true if the connection was added to the context
    */
   public static boolean tryAddConnection (Set context, UMLConnection connection)
   {
      if (containsConnectors (context, connection))
      {
         return context.add (connection);
      }
      return false;
   }


   /**
    * Add the connection and the item to the context if the end of the connection that is not
    * the item is already in the context
    *
    * @param context     the context
    * @param connection  the connection
    * @param item        the item
    * @return            true if the item was added to the context
    */
   public static boolean tryAddConnection (Set context, UMLConnection connection, UMLDiagramItem item)
   {
      if (context != null && context.contains (connection.getPartner (item)))
      {
         context.add (connection);
         return context.add (item);
      }
      return false;
   }


   /**
    * Add the connection to the context if the diagram contains the connection and the context
    * contains at least one of the connections ends. If the other end is not yet in the context
    * it is added too.
    *
    * @param diag        the diagram
    * @param context     the context
    * @param connection  the connection
    * @return            the item that was added or null if none was added
    */
   public static UMLDiagramItem tryAddConnection (UMLDiagram diag, Set context, UMLConnection connection)
   {
      if (containsConnection (diag, connection))
      {
         if (context.contains (connection.getSourceConnector()))
         {
            context.add (connection);
            if (context.add (connection.getTargetConnector()))
            {
               return connection.getTargetConnector();
            }
            else
            {
               return null;
            }
         }
         else if (context.contains (connection.getTargetConnector()))
         {
            context.add (connection);
            if (context.add (connection.getSourceConnector()))
            {
               return connection.getSourceConnector();
            }
            else
            {
               return null;
            }
         }
      }
      return null;
   }


   /**
    * Add the connection to the context if the diagram contains the connection and the context
    * contains at least the given item. If the other end is not yet in the context it is added
    * too.
    *
    * @param diag        the diagram
    * @param context     the context
    * @param connection  the connection
    * @param item        the item
    * @return            true if the connection was added
    */
   public static boolean tryAddConnection (UMLDiagram diag, Set context, UMLConnection connection, UMLDiagramItem item)
   {
      if (containsConnection (diag, connection))
      {
         return tryAddConnection (context, connection, item);
      }

      return false;
   }


   /**
    * <pre>
    *          0..1                            N
    * Filter ----------------------------- ViewDefinition
    *          filter     viewDefinitions
    * </pre>
    */
   private FHashSet viewDefinitions;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToViewDefinitions (ViewDefinition value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.viewDefinitions == null)
         {
            this.viewDefinitions = new FHashSet();
         }
         changed = this.viewDefinitions.add (value);
         if (changed)
         {
            value.setFilter (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInViewDefinitions (ViewDefinition value)
   {
      return  ( (this.viewDefinitions != null) &&
          (value != null) &&
         this.viewDefinitions.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfViewDefinitions()
   {
      return  ( (this.viewDefinitions == null)
         ? FEmptyIterator.get()
         : this.viewDefinitions.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfViewDefinitions()
   {
      return  ( (this.viewDefinitions == null)
         ? 0
         : this.viewDefinitions.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromViewDefinitions (ViewDefinition value)
   {
      boolean changed = false;

      if ( (this.viewDefinitions != null) &&  (value != null))
      {
         changed = this.viewDefinitions.remove (value);
         if (changed)
         {
            value.setFilter (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromViewDefinitions()
   {
      ViewDefinition tmpValue;
      Iterator iter = this.iteratorOfViewDefinitions();

      while (iter.hasNext())
      {
         tmpValue = (ViewDefinition) iter.next();
         this.removeFromViewDefinitions (tmpValue);
      }
   }
}

/*
 * $Log: AbstractFilter.java,v $
 * Revision 1.14  2004/10/20 17:50:30  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
