/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views;

import java.util.Iterator;
import java.util.Set;

import de.uni_paderborn.fujaba.uml.UMLDiagram;
import de.uni_paderborn.fujaba.uml.UMLDiagramItem;


/**
 * A Filter defines a rule for building n-Contexts on UMLDiagrams <h2> Associations </h2> <pre>
 *          0..1                            N
 * Filter ----------------------------- ViewDefinition
 *          filter     viewDefinitions
 * </pre>
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.10 $
 * @see       de.uni_paderborn.fujaba.views.ViewDefinition
 * @see       de.uni_paderborn.fujaba.views.ViewDiagram
 * @see       de.uni_paderborn.fujaba.uml.UMLDiagram
 * @see       de.uni_paderborn.fujaba.uml.UMLDiagramItem
 */
public interface Filter
{
   /**
    * Defines if this Filter is suited for a diagram or not.
    *
    * @param diag  the diagram
    * @return      true if this Filter can be used for diag, false otherwise
    */
   public boolean isForDiagram (UMLDiagram diag);


   /**
    * The Name of the Filter. Used for identification in the FilterManager and dialogs. Default
    * is the class name
    *
    * @return   the name of the Filter
    * @see      de.uni_paderborn.fujaba.views.FilterManager
    */
   public String getName();


   /**
    * In this Method the Rule for the 1-Context is defined. Each call to this Method increases
    * the Context by one.
    *
    * @param diag     The diagram in which the items have to be
    * @param item     The item on which the 1-Context is to be built
    * @param context  The existing context
    * @return         The new items which can be start points for more growing
    * @see            #get1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem)
    * @see            #get1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,java.util.Iterator)
    * @see            #getNContext(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem,int)
    * @see            #getNContext(de.uni_paderborn.fujaba.uml.UMLDiagram,java.util.Iterator,int)
    */
   public Set grow1Context (UMLDiagram diag, UMLDiagramItem item, ItemSet context);


   /**
    * Get the 1-Context for the given item
    *
    * @param diag  The diagram in which the items have to be
    * @param item  The item on which the 1-Context is to be built
    * @return      The 1-Context as Set
    * @see         #grow1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem,ItemSet)
    * @see         #get1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,java.util.Iterator)
    * @see         #getNContext(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem,int)
    * @see         #getNContext(de.uni_paderborn.fujaba.uml.UMLDiagram,java.util.Iterator,int)
    */
   public Set get1Context (UMLDiagram diag, UMLDiagramItem item);


   /**
    * Get the 1-Context for the items in the Iterator
    *
    * @param diag   The diagram in which the items have to be
    * @param items  Iterator listing the items on which the 1-Context is to be built
    * @return       The 1-Context as Set
    * @see          #grow1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem,ItemSet)
    * @see          #get1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem)
    * @see          #getNContext(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem,int)
    * @see          #getNContext(de.uni_paderborn.fujaba.uml.UMLDiagram,java.util.Iterator,int)
    */
   public Set get1Context (UMLDiagram diag, Iterator items);


   /**
    * Get the n-Context for the items in the Iterator
    *
    * @param diag   The diagram in which the items have to be
    * @param depth  the context to build
    * @param item   No description provided
    * @return       The context as Set
    * @see          #grow1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem,ItemSet)
    * @see          #get1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem)
    * @see          #get1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,java.util.Iterator)
    * @see          #getNContext(de.uni_paderborn.fujaba.uml.UMLDiagram,java.util.Iterator,int)
    */
   public Set getNContext (UMLDiagram diag, UMLDiagramItem item, int depth);


   /**
    * Get the n-Context for the items in the Iterator
    *
    * @param diag   The diagram in which the items have to be
    * @param items  Iterator listing the items on which the 1-Context is to be built
    * @param depth  the Context to build
    * @return       The context as Set
    * @see          #grow1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem,ItemSet)
    * @see          #get1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem)
    * @see          #get1Context(de.uni_paderborn.fujaba.uml.UMLDiagram,java.util.Iterator)
    * @see          #getNContext(de.uni_paderborn.fujaba.uml.UMLDiagram,de.uni_paderborn.fujaba.uml.UMLDiagramItem,int)
    */
   public Set getNContext (UMLDiagram diag, Iterator items, int depth);


   /**
    * gives a short description of the Filter
    *
    * @return   description text for the filter
    */
   public String getDescription();


   /**
    * <pre>
    *          0..1                            N
    * Filter ----------------------------- ViewDefinition
    *          filter     viewDefinitions
    * </pre> access Method for the above associaction
    *
    * @param value  The object added.
    * @return       No description provided
    * @see          de.uni_paderborn.fujaba.views.ViewDefinition
    */
   public boolean addToViewDefinitions (ViewDefinition value);


   /**
    * <pre>
    *          0..1                            N
    * Filter ----------------------------- ViewDefinition
    *          filter     viewDefinitions
    * </pre> access Method for the above associaction
    *
    * @param value  No description provided
    * @return       No description provided
    * @see          de.uni_paderborn.fujaba.views.ViewDefinition
    */
   public boolean hasInViewDefinitions (ViewDefinition value);


   /**
    * <pre>
    *          0..1                            N
    * Filter ----------------------------- ViewDefinition
    *          filter     viewDefinitions
    * </pre> access Method for the above associaction
    *
    * @return   No description provided
    * @see      de.uni_paderborn.fujaba.views.ViewDefinition
    */
   public Iterator iteratorOfViewDefinitions();


   /**
    * <pre>
    *          0..1                            N
    * Filter ----------------------------- ViewDefinition
    *          filter     viewDefinitions
    * </pre> access Method for the above associaction
    *
    * @return   No description provided
    * @see      de.uni_paderborn.fujaba.views.ViewDefinition
    */
   public int sizeOfViewDefinitions();


   /**
    * <pre>
    *          0..1                            N
    * Filter ----------------------------- ViewDefinition
    *          filter     viewDefinitions
    * </pre> access Method for the above associaction
    *
    * @param value  No description provided
    * @return       No description provided
    * @see          de.uni_paderborn.fujaba.views.ViewDefinition
    */
   public boolean removeFromViewDefinitions (ViewDefinition value);


   /**
    * <pre>
    *          0..1                            N
    * Filter ----------------------------- ViewDefinition
    *          filter     viewDefinitions
    * </pre> access Method for the above associaction
    *
    * @see   de.uni_paderborn.fujaba.views.ViewDefinition
    */
   public void removeAllFromViewDefinitions();
}

/*
 * $Log: Filter.java,v $
 * Revision 1.10  2004/10/20 17:50:30  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
