/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views;

import java.awt.*;
import java.beans.*;
import java.io.*;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;
import java.util.zip.*;

import javax.swing.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.basic.FujabaPropertyChangeSupport;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.uml.UMLDiagram;
import de.upb.tools.fca.*;
import de.upb.tools.pcs.CollectionChangeEvent;
import de.upb.tools.pcs.PropertyChangeClient;


/**
 * The FilterManager is sort of a Factory for filters It holds all known Filters which can
 * be accessed by their name. Additionally it has some comfort functions for loading/saving
 * and models for the gui and a function to search for new filters.
 *
 * @author    $Author: trinet $
 * @version   $Revision: 1.20 $
 */
public class FilterManager implements PropertyChangeClient
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (FilterManager.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static transient String SETTINGS_FILE_EXT = "ffs.gz";


   /**
    * Constructor for class FilterManager
    */
   private FilterManager()
   {
      addToFilters (new InheritanceFilter());
      addToFilters (new AssocFilter());
      addToFilters (new TransitionFilter());
      addToFilters (new LinkFilter());
      addToFilters (new CompositeFilter());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient FilterManager filterManager = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public static FilterManager get()
   {
      if (filterManager == null)
      {
         filterManager = new FilterManager();
      }

      return filterManager;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FujabaPropertyChangeSupport propertyChangeSupport = null;


   /**
    * Get the propertyChangeSupport attribute of the FilterManager object
    *
    * @return   The propertyChangeSupport value
    */
   public PropertyChangeSupport getPropertyChangeSupport()
   {
      return propertyChangeSupport;
   }


   /**
    * Access method for an one to n association.
    *
    * @param listener  The object added.
    */
   public void addPropertyChangeListener (PropertyChangeListener listener)
   {
      if (propertyChangeSupport == null)
      {
         propertyChangeSupport = new FujabaPropertyChangeSupport (this);
      }

      propertyChangeSupport.addPropertyChangeListener (listener);
   }


   /**
    * Access method for an one to n association.
    *
    * @param propertyName  The object added.
    * @param listener      The object added.
    */
   public void addPropertyChangeListener (String propertyName, PropertyChangeListener listener)
   {
      if (propertyChangeSupport == null)
      {
         propertyChangeSupport = new FujabaPropertyChangeSupport (this);
      }

      propertyChangeSupport.addPropertyChangeListener (propertyName, listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param listener  No description provided
    */
   public void removePropertyChangeListener (PropertyChangeListener listener)
   {
      if (propertyChangeSupport != null)
      {
         propertyChangeSupport.removePropertyChangeListener (listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param propertyName  No description provided
    * @param listener      No description provided
    */
   public void removePropertyChangeListener (String propertyName, PropertyChangeListener listener)
   {
      if (propertyChangeSupport != null)
      {
         propertyChangeSupport.removePropertyChangeListener (propertyName, listener);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   protected void firePropertyChange (PropertyChangeEvent e)
   {
      if (propertyChangeSupport != null)
      {
         propertyChangeSupport.firePropertyChange (e);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, Object oldValue, Object newValue)
   {
      if (oldValue == newValue || propertyChangeSupport == null)
      {
         return;
      }

      propertyChangeSupport.firePropertyChange (name, oldValue, newValue);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, boolean oldValue, boolean newValue)
   {
      if (oldValue == newValue || propertyChangeSupport == null)
      {
         return;
      }

      firePropertyChange (name, Boolean.valueOf (oldValue), Boolean.valueOf (newValue));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, int oldValue, int newValue)
   {
      if (oldValue == newValue)
      {
         return;
      }
      firePropertyChange (name, new Integer (oldValue), new Integer (newValue));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name      No description provided
    * @param oldValue  No description provided
    * @param newValue  No description provided
    */
   protected void firePropertyChange (String name, double oldValue, double newValue)
   {
      if (oldValue == newValue || propertyChangeSupport == null)
      {
         return;
      }
      firePropertyChange (name, new Double (oldValue), new Double (newValue));
   }


   /**
    * Instantiate the given filter class
    *
    * @param filterClass  No description provided
    * @return             an instance of the class
    */
   public static Filter createInstance (Class filterClass)
   {
      try
      {
         return (Filter) filterClass.newInstance();
      }
      catch (Exception e)
      {
      }

      Filter instance = null;
      Method[] methods = filterClass.getMethods();
      for (int i = 0; i < methods.length && instance == null; i++)
      {
         if (Modifier.isStatic (methods[i].getModifiers()) &&
            methods[i].getParameterTypes().length == 0 &&
            Filter.class.isAssignableFrom (methods[i].getReturnType()))
         {
            try
            {
               instance = (Filter) methods[i].invoke (null, new Object[0]);
            }
            catch (Exception e)
            {
            }
         }
      }
      return instance;
   }


   /**
    * Instantiate the class defined in the given file
    *
    * @param classFile  No description provided
    * @return           an instance of the class
    */
   public static Filter createInstance (File classFile)
   {
      return createInstance (FilterClassLoader.get().defineClass (classFile));
   }


   /**
    * Instantiate the class defined in the given file
    *
    * @param classFile  No description provided
    * @param entry      No description provided
    * @return           an instance of the class
    */
   public static Filter createInstance (ZipFile classFile, ZipEntry entry)
   {
      return createInstance (FilterClassLoader.get().defineClass (classFile, entry));
   }


   /**
    * Save the Filter to the Output Stream
    *
    * @param filter  No description provided
    * @param stream  No description provided
    */
   public static void saveFilter (ConfigurableFilter filter, OutputStream stream)
   {
      PrintWriter out = new PrintWriter (stream);
      // write the header at the beginning of the file
      out.print ("# Fujaba-FilterSettings-File "
         + "(do not alter this file!!!)\n");
      out.print ("# Date    : " + new Date (System.currentTimeMillis()).toString() + "\n");
      out.print ("-;FilterClass;" + filter.getClass().getName() + "\n");
      out.flush();
      filter.writeSettings (out);
      out.flush();
   }


   /**
    * Save the Filter to the file
    *
    * @param filter        No description provided
    * @param file          No description provided
    * @throws IOException  Exception description not provided
    */
   public static void saveFilter (ConfigurableFilter filter, File file) throws IOException
   {
      OutputStream stream = new GZIPOutputStream (new FileOutputStream (file));
      saveFilter (filter, stream);
      stream.flush();
      stream.close();

      if (get().hasInFilters (filter))
      {
         get().addToSettingsFiles (filter, file);
      }
   }


   /**
    * read the settings from the stream and create a new Filter with them
    *
    * @param stream  No description provided
    * @return        No description provided
    */
   public static ConfigurableFilter loadFilter (InputStream stream)
   {
      ConfigurableFilter filter = null;
      BufferedReader fr = null;
      String line;
      String className = null;

      StringTokenizer lineTok;

      try
      {
         fr = new BufferedReader (new InputStreamReader (stream), 32768);

         // check, if it is the correct FileFormat
         line = fr.readLine();
         if (!line.startsWith ("# Fujaba-FilterSettings-File"))
         {
            throw new Exception ("This is not a Fujaba filtersettings (.ffs) file");
         }

         // Now parse the rest of the file
         while (fr.ready() && className == null)
         {
            line = fr.readLine();

            lineTok = new StringTokenizer (line, ";");
            lineTok.nextToken();

            switch (line.charAt (0))
            {
               case '#': // documentation

                  break;
               case '-': // file specific variables

                  String variable = lineTok.nextToken();
                  if (variable.compareTo ("FilterClass") == 0)
                  {
                     className = lineTok.nextToken();
                  }
                  break;
               default:
                  if (line.trim().length() > 0)
                  {
                     throw new Exception ("Unexpected line while searching for class name of filter");
                  }
                  break;
            }
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
         throw new RuntimeException ("An error occured while reading the settings file." +
            "\nMessage: " + e);
      }

      if (className != null)
      {
         try
         {
            Class clazz = FilterClassLoader.get().loadClass (className);
            if (!ConfigurableFilter.class.isAssignableFrom (clazz))
            {
               throw new IllegalArgumentException ("The InputStream does not denote a " +
                  "ConfigurableFilter");
            }

            filter = (ConfigurableFilter) createInstance (clazz);
            filter.readSettings (fr);
         }
         catch (ClassNotFoundException e)
         {
            log.error ("Class not found: " + className);
         }
      }
      return filter;
   }


   /**
    * read the settings from the file and create a new Filter with them
    *
    * @param file          No description provided
    * @return              No description provided
    * @throws IOException  Exception description not provided
    */
   public static ConfigurableFilter loadFilter (File file) throws IOException
   {
      InputStream stream = new GZIPInputStream (new FileInputStream (file));
      ConfigurableFilter filter = loadFilter (stream);
      return filter;
   }


   /**
    * rescan the filter dir defined in the Environment Options
    *
    * @see   #rescanDir(java.io.File)
    */
   public void rescanDir()
   {
      File dir = new File (GeneralPreferences.get().getViewFilterFolder());
      rescanDir (dir);
   }


   /**
    * search the directory for new class files or new settings files and create the filters
    * defined by them
    *
    * @param dir  No description provided
    */
   private void rescanDir (File dir)
   {
      if (!dir.exists())
      {
         return;
      }

      File[] files = dir.listFiles (new AFileFilter());
      for (int i = 0; i < files.length; i++)
      {
         String path = null;
         try
         {
            path = files[i].getCanonicalPath();
         }
         catch (IOException ioe)
         {
            path = files[i].getAbsolutePath();
         }

         if (!files[i].isDirectory() &&
             (path.toLowerCase().endsWith (".zip") || path.toLowerCase().endsWith (".jar")))
         {
            path += "[" + files[i].lastModified() + "]";
         }

         if (!hasInScannedFiles (path))
         {
            addToScannedFiles (path);
            if (files[i].isDirectory())
            {
               rescanDir (files[i]);
            }
            if (path.toLowerCase().endsWith (".zip") || path.toLowerCase().endsWith (".jar"))
            {
               rescanZipFile (files[i]);
            }
            else if (path.toLowerCase().endsWith ("." + SETTINGS_FILE_EXT))
            {
               try
               {
                  ConfigurableFilter filter = loadFilter (files[i]);
                  addToFilters (filter);
                  addToSettingsFiles (filter, files[i]);
               }
               catch (IOException e)
               {
                  log.error ("Found " + path + ", error while loading: " + e);
               }
            }
            else if (path.toLowerCase().endsWith (".class"))
            {
               Class clazz = null;
               try
               {
                  clazz = FilterClassLoader.get().defineClass (files[i]);
               }
               catch (Exception e)
               {
                  continue;
               }
               if (Filter.class.isAssignableFrom (clazz))
               {
                  addToFilters (createInstance (clazz));
               }
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param file  No description provided
    */
   private void rescanZipFile (File file)
   {
      ZipFile zipFile = null;
      try
      {
         zipFile = new ZipFile (file);
      }
      catch (Exception e)
      {
         return;
      }

      String zipName = null;
      try
      {
         zipName = file.getCanonicalPath();
      }
      catch (IOException ioe)
      {
         zipName = file.getAbsolutePath();
      }
      Enumeration enumeration = zipFile.entries();
      while (enumeration.hasMoreElements())
      {
         ZipEntry entry = (ZipEntry) enumeration.nextElement();
         if (entry.isDirectory())
         {
            continue;
         }

         String name = entry.getName();
         String fileName = null;
         try
         {
            fileName =  (new File (name)).getCanonicalPath();
         }
         catch (IOException ioe)
         {
            fileName =  (new File (name)).getAbsolutePath();
         }

         if (!hasInScannedFiles (zipName + ":" + name) &&
            !hasInScannedFiles (fileName))
         {
            addToScannedFiles (zipName + ":" + name);
            InputStream stream = null;
            try
            {
               stream = zipFile.getInputStream (entry);
            }
            catch (IOException ioe)
            {
               continue;
            }

            if (name.toLowerCase().endsWith ("." + SETTINGS_FILE_EXT))
            {
               ConfigurableFilter filter = loadFilter (stream);
               addToFilters (filter);
               addToSettingsFiles (filter, new File (name));
            }
            else if (name.toLowerCase().endsWith (".class"))
            {
               Class clazz = null;
               try
               {
                  clazz = FilterClassLoader.get().defineClass (stream, entry.getSize());
               }
               catch (Exception e)
               {
                  continue;
               }
               if (Filter.class.isAssignableFrom (clazz))
               {
                  addToFilters (createInstance (clazz));
               }
            }
            try
            {
               stream.close();
            }
            catch (IOException ioe)
            {
            }
         }
      }
   }


   /**
    * <pre>
    *                                       N
    * FilterManager --------------------------> String
    *                            scannedFiles
    * </pre>
    */
   private transient FHashSet scannedFiles;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   protected boolean addToScannedFiles (String value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.scannedFiles == null)
         {
            this.scannedFiles = new FHashSet();
         }
         changed = this.scannedFiles.add (value);
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInScannedFiles (String value)
   {
      return  ( (this.scannedFiles != null) &&
          (value != null) &&
         this.scannedFiles.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfScannedFiles()
   {
      return  ( (this.scannedFiles == null)
         ? FEmptyIterator.get()
         : this.scannedFiles.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfScannedFiles()
   {
      return  ( (this.scannedFiles == null)
         ? 0
         : this.scannedFiles.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   protected boolean removeFromScannedFiles (String value)
   {
      boolean changed = false;

      if ( (this.scannedFiles != null) &&  (value != null))
      {
         changed = this.scannedFiles.remove (value);
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void removeAllFromScannedFiles()
   {
      String tmpValue;
      Iterator iter = this.iteratorOfScannedFiles();

      while (iter.hasNext())
      {
         tmpValue = (String) iter.next();
         this.removeFromScannedFiles (tmpValue);
      }
   }


   /**
    * <pre>
    *               +-----------+                    1
    * FilterManager | getName() +----------------------> Filter
    *               +-----------+              filters
    * </pre>
    */
   private transient FHashMap filters;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInFilters (Filter value)
   {
      if ( (this.filters == null) ||  (value == null))
      {
         return false;
      }

      String name = value.getName();
      return  (name != null && this.filters.get (name) == value);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInFilters (String key, Filter value)
   {
      return  ( (this.filters != null) &&
          (value != null) &&  (key != null) &&
          (this.filters.get (key) == value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInFilters (String key)
   {
      return  ( (this.filters != null) &&
          (key != null) &&
         this.filters.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfFilters()
   {
      return  ( (this.filters == null)
         ? FEmptyIterator.get()
         : this.filters.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfFilters()
   {
      return  ( (this.filters == null)
         ? FEmptyIterator.get()
         : this.filters.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfFilters()
   {
      return  ( (this.filters == null)
         ? FEmptyIterator.get()
         : this.filters.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfFilters()
   {
      return  ( (this.filters == null)
         ? 0
         : this.filters.size());
   }


   /**
    * Get the fromFilters attribute of the FilterManager object
    *
    * @param key  No description provided
    * @return     The fromFilters value
    */
   public Filter getFromFilters (String key)
   {
      Filter result =  ( ( (this.filters == null) ||  (key == null))
         ? null
         : (Filter) this.filters.get (key));

      if (result == null)
      {
         rescanDir();
         result =  ( ( (this.filters == null) ||  (key == null))
            ? null
            : (Filter) this.filters.get (key));
      }
      return result;
   }


   /**
    * Get the keyFromFilters attribute of the FilterManager object
    *
    * @param value  No description provided
    * @return       The keyFromFilters value
    */
   public String getKeyFromFilters (Filter value)
   {
      if (value == null)
      {
         return null;
      }

      String key = value.getName();
      if (key != null && ! (getFromFilters (key) == value))
      {
         key = null;
      }
      return key;
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToFilters (Filter value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.filters == null)
         {
            this.filters = new FPropHashMap (this, "filters");
         }
         Filter oldValue = (Filter) this.filters.put (value.getName(), value);
         if (oldValue != value)
         {
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromFilters (Filter value)
   {
      boolean changed = false;
      if ( (this.filters != null) &&  (value != null) &&  (value.getName() != null))
      {
         Filter oldValue = (Filter) this.filters.get (value.getName());
         if (oldValue == value)
         {
            this.filters.remove (value.getName());
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromFilters (String key)
   {
      boolean changed = false;
      if ( (this.filters != null) &&  (key != null))
      {
         Filter tmpValue = (Filter) this.filters.get (key);
         if (tmpValue != null)
         {
            this.filters.remove (key);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromFilters()
   {
      Iterator iter = iteratorOfFilters();
      Filter entry;
      while (iter.hasNext())
      {
         entry = (Filter) iter.next();
         removeFromFilters (entry);
      }
   }


   /**
    * Get the filter attribute of the FilterManager object
    *
    * @param name  No description provided
    * @return      The filter value
    */
   public Filter getFilter (String name)
   {
      Filter filter = getFromFilters (name);
      if (filter != null)
      {
         Filter clone = null;

         try
         {
            if (filter instanceof ConfigurableFilter)
            {
               clone = (Filter)  ((ConfigurableFilter) filter).clone();
            }
            else
            {
               clone = createInstance (filter.getClass());
            }
         }
         catch (Exception e)
         {
         }
         return clone;
      }
      return null;
   }

   // --------------------------------------------------------------------------

   /**
    * <pre>
    *               +-----+                    1
    * FilterManager | key +----------------------> File
    *               +-----+              settingsFiles
    * </pre>
    */
   private transient FHashMap settingsFiles;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInSettingsFiles (File value)
   {
      return  ( (this.settingsFiles != null) &&
          (value != null) &&
         this.settingsFiles.containsValue (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInSettingsFiles (ConfigurableFilter key, File value)
   {
      return  ( (this.settingsFiles != null) &&
          (value != null) &&  (key != null) &&
          (this.settingsFiles.get (key) == value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInSettingsFiles (ConfigurableFilter key)
   {
      return  ( (this.settingsFiles != null) &&
          (key != null) &&
         this.settingsFiles.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfSettingsFiles()
   {
      return  ( (this.settingsFiles == null)
         ? FEmptyIterator.get()
         : this.settingsFiles.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfSettingsFiles()
   {
      return  ( (this.settingsFiles == null)
         ? FEmptyIterator.get()
         : this.settingsFiles.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfSettingsFiles()
   {
      return  ( (this.settingsFiles == null)
         ? FEmptyIterator.get()
         : this.settingsFiles.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfSettingsFiles()
   {
      return  ( (this.settingsFiles == null)
         ? 0
         : this.settingsFiles.size());
   }


   /**
    * Get the fromSettingsFiles attribute of the FilterManager object
    *
    * @param key  No description provided
    * @return     The fromSettingsFiles value
    */
   public File getFromSettingsFiles (ConfigurableFilter key)
   {
      return  ( ( (this.settingsFiles == null) ||  (key == null))
         ? null
         : (File) this.settingsFiles.get (key));
   }


   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToSettingsFiles (ConfigurableFilter key, File value)
   {
      boolean changed = false;
      if ( (value != null) &&  (key != null))
      {
         if (this.settingsFiles == null)
         {
            this.settingsFiles = new FPropHashMap (this, "settingsFiles"); // or FTreeMap ()

         }
         File oldValue = (File) this.settingsFiles.put (key, value);
         if (oldValue != value)
         {
            changed = true;
         }
      }
      return changed;
   }


   /**
    * Access method for an one to n association.
    *
    * @param entry  The object added.
    * @return       No description provided
    */
   public boolean addToSettingsFiles (Map.Entry entry)
   {
      return addToSettingsFiles ((ConfigurableFilter) entry.getKey(), (File) entry.getValue());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromSettingsFiles (File value)
   {
      boolean changed = false;
      if ( (this.settingsFiles != null) &&  (value != null))
      {
         Iterator iter = this.entriesOfSettingsFiles();
         Map.Entry entry;
         while (iter.hasNext())
         {
            entry = (Map.Entry) iter.next();
            if (entry.getValue() == value)
            {
               changed = changed | this.removeFromSettingsFiles ((ConfigurableFilter) entry.getKey(), value);
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key    No description provided
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromSettingsFiles (ConfigurableFilter key, File value)
   {
      boolean changed = false;
      if ( (this.settingsFiles != null) &&  (value != null) &&  (key != null))
      {
         File oldValue = (File) this.settingsFiles.get (key);
         if (oldValue == value)
         {
            this.settingsFiles.remove (key);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromSettingsFiles (ConfigurableFilter key)
   {
      boolean changed = false;
      if ( (this.settingsFiles != null) &&  (key != null))
      {
         File tmpValue = (File) this.settingsFiles.get (key);
         if (tmpValue != null)
         {
            this.settingsFiles.remove (key);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromSettingsFiles()
   {
      Iterator iter = entriesOfSettingsFiles();
      Map.Entry entry;
      while (iter.hasNext())
      {
         entry = (Map.Entry) iter.next();
         removeFromSettingsFiles ((ConfigurableFilter) entry.getKey(), (File) entry.getValue());
      }
   }

   //----------------------------------------------------------------------------------------------

   /**
    * Get the listModel attribute of the FilterManager object
    *
    * @param diagram  No description provided
    * @return         The listModel value
    */
   public ComboBoxModel getListModel (UMLDiagram diagram)
   {
      while (diagram != null && diagram instanceof ViewDiagram)
      {
         diagram =  ((ViewDiagram) diagram).getDiagram();
      }
      FMListModel model = new FMListModel (diagram);
      fillModel (model);
      return model;
   }


   /**
    * @param diagram  No description provided
    * @return         a ComboBoxModel that contains all the known filters that are valid for
    *      the diagram
    * @see            de.uni_paderborn.fujaba.views.Filter#isForDiagram
    */
   public ComboBoxModel getPrototypeListModel (UMLDiagram diagram)
   {
      while (diagram != null && diagram instanceof ViewDiagram)
      {
         diagram =  ((ViewDiagram) diagram).getDiagram();
      }
      FMListModel model = new FMListModel (diagram);
      fillPrototypeModel (model);

      return model;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param model  No description provided
    */
   private void fillModel (FMListModel model)
   {
      UMLDiagram diagram = model.getDiagram();
      while (diagram != null && diagram instanceof ViewDiagram)
      {
         diagram =  ((ViewDiagram) diagram).getDiagram();
      }
      Iterator filterIter = iteratorOfFilters();
      while (filterIter.hasNext())
      {
         Filter filter = (Filter) filterIter.next();
         if (diagram == null || filter.isForDiagram (diagram))
         {
            Filter clone = null;

            try
            {
               if (filter instanceof ConfigurableFilter)
               {
                  clone = (Filter)  ((ConfigurableFilter) filter).clone();
               }
               else
               {
                  clone = createInstance (filter.getClass());
               }
            }
            catch (Exception e)
            {
            }

            if (clone != null)
            {
               model.addElement (clone);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param model  No description provided
    */
   private void fillPrototypeModel (FMListModel model)
   {
      UMLDiagram diagram = model.getDiagram();
      while (diagram != null && diagram instanceof ViewDiagram)
      {
         diagram =  ((ViewDiagram) diagram).getDiagram();
      }
      Iterator filterIter = iteratorOfFilters();
      while (filterIter.hasNext())
      {
         Filter filter = (Filter) filterIter.next();
         if (diagram == null || filter.isForDiagram (diagram))
         {
            model.addElement (filter);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient ListCellRenderer renderer = null;


   /**
    * A Renderer for Filters
    *
    * @return   The listCellRenderer value
    */
   public static ListCellRenderer getListCellRenderer()
   {
      if (renderer == null)
      {
         renderer = new FMListCellRenderer();
      }

      return renderer;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: trinet $
    * @version   $Revision: 1.20 $
    */
   private final static class FMListCellRenderer extends DefaultListCellRenderer
   {
      /**
       * Get the listCellRendererComponent attribute of the FMListCellRenderer object
       *
       * @param list          No description provided
       * @param value         No description provided
       * @param index         No description provided
       * @param isSelected    No description provided
       * @param cellHasFocus  No description provided
       * @return              The listCellRendererComponent value
       */
      public Component getListCellRendererComponent (JList list,
                                                     Object value,
                                                     int index,
                                                     boolean isSelected,
                                                     boolean cellHasFocus)
      {
         if (value == null)
         {
            return super.getListCellRendererComponent (list, value,
               index, isSelected, cellHasFocus);
         }

         String name =  ((Filter) value).getName();
         if (name == null)
         {
            name = value.toString();
         }

         return super.getListCellRendererComponent (list, name,
            index, isSelected, cellHasFocus);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: trinet $
    * @version   $Revision: 1.20 $
    */
   public final static class FMListModel extends DefaultComboBoxModel implements PropertyChangeListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private boolean clone = false;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private UMLDiagram diagram = null;


      /**
       * Get the diagram attribute of the FMListModel object
       *
       * @return   The diagram value
       */
      public UMLDiagram getDiagram()
      {
         return diagram;
      }


      /**
       * Constructor for class FMListModel
       *
       * @param diagram  No description provided
       * @param clone    No description provided
       */
      public FMListModel (UMLDiagram diagram, boolean clone)
      {
         this (diagram);
         this.clone = clone;
      }


      /**
       * Constructor for class FMListModel
       *
       * @param diagram  No description provided
       */
      public FMListModel (UMLDiagram diagram)
      {
         super();
         while (diagram != null && diagram instanceof ViewDiagram)
         {
            diagram =  ((ViewDiagram) diagram).getDiagram();
         }
         this.diagram = diagram;
         FilterManager.get().addPropertyChangeListener ("filters", this);
      }


      /**
       * Access method for an one to n association.
       *
       * @param object  The object added.
       */
      public void addElement (Object object)
      {
         if (! (object instanceof Filter))
         {
            throw new IllegalArgumentException ("Object is not a Filter");
         }

         Filter filter = (Filter) object;
         if (diagram != null && !filter.isForDiagram (diagram))
         {
            throw new IllegalArgumentException ("Object is not suited for UMLDiagram " +
               diagram);
         }

         String name =  ((Filter) object).getName();
         int size = this.getSize();
         for (int i = 0; i < size; i++)
         {
            String compare =  ((Filter) this.getElementAt (i)).getName();
            if (compare.compareTo (name) >= 0)
            {
               this.insertElementAt (object, i);
               return;
            }
         }
         super.addElement (object);
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param o  No description provided
       * @param i  No description provided
       */
      public void insertElementAt (Object o, int i)
      {
         if (! (o instanceof Filter))
         {
            throw new IllegalArgumentException ("Object is not a Filter");
         }

         Filter filter = (Filter) o;
         if (diagram != null && !filter.isForDiagram (diagram))
         {
            throw new IllegalArgumentException ("Object is not suited for UMLDiagram " +
               diagram);
         }

         super.insertElementAt (o, i);
      }


      /**
       * Sets the selectedItem attribute of the FMListModel object
       *
       * @param o  The new selectedItem value
       */
      public void setSelectedItem (Object o)
      {
         if (! (o instanceof Filter))
         {
            throw new IllegalArgumentException ("Object is not a Filter");
         }

         Filter filter = (Filter) o;
         if (diagram != null && !filter.isForDiagram (diagram))
         {
            throw new IllegalArgumentException ("Object is not suited for UMLDiagram " +
               diagram);
         }

         super.setSelectedItem (o);
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void propertyChange (PropertyChangeEvent e)
      {
         if (e instanceof CollectionChangeEvent)
         {
            CollectionChangeEvent event = (CollectionChangeEvent) e;

            if (event.getType() == CollectionChangeEvent.ADDED)
            {
               Filter filter = (Filter) event.getNewValue();
               if (diagram == null || filter.isForDiagram (diagram))
               {
                  Filter add = null;

                  if (clone)
                  {
                     try
                     {
                        if (filter instanceof ConfigurableFilter)
                        {
                           add = (Filter)  ((ConfigurableFilter) filter).clone();
                        }
                        else
                        {
                           add = createInstance (filter.getClass());
                        }
                     }
                     catch (Exception ex)
                     {
                     }
                  }
                  else
                  {
                     add = filter;
                  }

                  if (add != null)
                  {
                     addElement (add);
                  }
               }
            }
            else if (event.getType() == CollectionChangeEvent.REMOVED)
            {
               Object item = event.getOldValue();
               boolean selected =  (getSelectedItem() == item);

               removeElement (item);
               if (selected && getSize() > 0)
               {
                  setSelectedItem (getElementAt (0));
               }
            }
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public void removeYou()
      {
         removeAllElements();
         FilterManager.get().removePropertyChangeListener (this);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: trinet $
    * @version   $Revision: 1.20 $
    */
   private final static class AFileFilter implements FileFilter
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param file  No description provided
       * @return      No description provided
       */
      public boolean accept (File file)
      {
         if (file.isDirectory())
         {
            return true;
         }

         String name = file.getName().toLowerCase();
         if (name.endsWith (".class") || name.endsWith (".zip") || name.endsWith (".jar") || name.endsWith ("." + SETTINGS_FILE_EXT.toLowerCase()))
         {
            return true;
         }

         return false;
      }
   }
}

/*
 * $Log: FilterManager.java,v $
 * Revision 1.20  2004/11/08 19:16:21  trinet
 * changed saving of FSAProperties, added String support for Maps in BasicIncrement, some cleanup
 *
 */
