/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views;

import java.util.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.basic.BasicIncrement;
import de.uni_paderborn.fujaba.uml.UMLDiagram;
import de.uni_paderborn.fujaba.uml.UMLDiagramItem;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashSet;


/**
 * A ViewDefinition consists of a Filter and all the information needed to use that filter
 * like source diagram, context, start items.<p>
 *
 * A number of ViewDefinitions defines a ViewDiagram.<p>
 *
 * <h2>Associations</h2> <pre>
 *                 N                      0..1
 * ViewDefinition ----------------------------- Filter
 *                 viewDefinitions      filter
 *
 *                                    N
 * ViewDefinition ----------------------> UMLDiagramItem
 *                           startItems
 *
 *                 N                      0..1
 * ViewDefinition ----------------------------- ViewDiagram
 *                 ViewDefinitions     diagram
 * </pre>
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.17 $
 */
public class ViewDefinition extends BasicIncrement
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (ViewDefinition.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean update = false;


   /**
    * Specifies whether the ViewDefinition should be automatically updated when the source
    * diagram changes
    *
    * @param update  do update or not
    * @return        true if the value was changed
    * @see           #isUpdate
    */
   public boolean setUpdate (boolean update)
   {
      boolean changed = false;
      if (update != this.update)
      {
         this.update = update;
         if (update)
         {
            update();
         }

         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name = null;


   /**
    * Sets the name attribute of the ViewDefinition object
    *
    * @param name  The new name value
    * @return      No description provided
    */
   public boolean setName (String name)
   {
      boolean changed = false;
      if (this.name != name)
      {
         this.name = name;
         changed = true;
      }
      return changed;
   }


   /**
    * Get the name attribute of the ViewDefinition object
    *
    * @return   The name value
    */
   public String getName()
   {
      return name;
   }


   /**
    * @return   true if the update property is set
    * @see      #setUpdate
    */
   public boolean isUpdate()
   {
      return update;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int context = 0;


   /**
    * define the context for the view
    *
    * @param context  the context
    * @see            #getContext
    */
   public void setContext (int context)
   {
      if (this.context != context)
      {
         this.context = context;
         if (getDiagram() != null && isUpdate())
         {
            update();
         }
      }
   }


   /**
    * get the context for the view
    *
    * @return   the context
    * @see      #getContext
    */
   public int getContext()
   {
      return context;
   }


   /**
    * <pre>
    *          N                            N
    * ViewDefinition ----------------------------- UMLDiagramItem
    *          viewDefinitions     startItems
    * </pre>
    */
   private FHashSet startItems;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToStartItems (UMLDiagramItem value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.startItems == null)
         {
            this.startItems = new FHashSet();
         }
         changed = this.startItems.add (value);
         if (changed)
         {
            value.addToViewDefinitions (this);
            if (getDiagram() != null && isUpdate())
            {
               update();
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInStartItems (UMLDiagramItem value)
   {
      return  ( (this.startItems != null) &&
          (value != null) &&
         this.startItems.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfStartItems()
   {
      return  ( (this.startItems == null)
         ? FEmptyIterator.get()
         : this.startItems.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfStartItems()
   {
      return  ( (this.startItems == null)
         ? 0
         : this.startItems.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromStartItems (UMLDiagramItem value)
   {
      boolean changed = false;

      if ( (this.startItems != null) &&  (value != null))
      {
         changed = this.startItems.remove (value);
         if (changed)
         {
            value.removeFromViewDefinitions (this);
            if (getDiagram() != null && isUpdate())
            {
               update();
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromStartItems()
   {
      UMLDiagramItem tmpValue;
      Iterator iter = this.iteratorOfStartItems();
      boolean update = isUpdate();
      setUpdate (false);

      while (iter.hasNext())
      {
         tmpValue = (UMLDiagramItem) iter.next();
         this.removeFromStartItems (tmpValue);
      }
      setUpdate (update);
      if (getDiagram() != null && update)
      {
         update();
      }
   }


   /**
    * <pre>
    *          N                            N
    * ViewDefinition ----------------------------- UMLDiagramItem
    *          revItems     items
    * </pre>
    */
   private transient FHashSet items;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToItems (UMLDiagramItem value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.items == null)
         {
            this.items = new FHashSet();
         }
         changed = this.items.add (value);
         if (changed)
         {
            value.addToRevItems (this);
            if (!evaluating && getDiagram() != null)
            {
               getDiagram().addToElements (value);
            }
         }
      }
      return changed;
   }


   /**
    * This is needed for parsing XML-files
    *
    * @param value  The object added.
    * @param diag   The object added.
    * @return       No description provided
    */
   public boolean addToItems (UMLDiagramItem value, UMLDiagram diag)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.items == null)
         {
            this.items = new FHashSet();
         }
         changed = this.items.add (value);
         if (changed)
         {
            value.addToRevItems (this);
            if (!evaluating && getDiagram() != null)
            {
               getDiagram().addToItems (value, diag);
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInItems (UMLDiagramItem value)
   {
      return  ( (this.items != null) &&
          (value != null) &&
         this.items.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfItems()
   {
      return  ( (this.items == null)
         ? FEmptyIterator.get()
         : this.items.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfItems()
   {
      return  ( (this.items == null)
         ? 0
         : this.items.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromItems (UMLDiagramItem value)
   {
      boolean changed = false;

      if ( (this.items != null) &&  (value != null))
      {
         changed = this.items.remove (value);
         if (changed)
         {
            value.removeFromRevItems (this);
            if (!evaluating && getDiagram() != null)
            {
               getDiagram().checkAndRemoveFromItems (value, this);
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromItems()
   {
      UMLDiagramItem tmpValue;
      Iterator iter = this.iteratorOfItems();
      HashSet oldItems = new HashSet();
      boolean wasEvaluating = evaluating;
      evaluating = true;

      while (iter.hasNext())
      {
         tmpValue = (UMLDiagramItem) iter.next();
         this.removeFromItems (tmpValue);
         oldItems.add (tmpValue);
      }
      if (!wasEvaluating && getDiagram() != null)
      {
         getDiagram().checkAndRemoveFromItems (oldItems, this);
      }
      evaluating = wasEvaluating;
   }


   /**
    * Get the items attribute of the ViewDefinition object
    *
    * @return   The items value
    */
   protected Set getItems()
   {
      return items;
   }


   /**
    * <pre>
    *                 N                      0..1
    * ViewDefinition ----------------------------- ViewDiagram
    *                 ViewDefinitions     diagram
    * </pre>
    */
   private ViewDiagram diagram;


   /**
    * Sets the diagram attribute of the ViewDefinition object
    *
    * @param value  The new diagram value
    * @return       No description provided
    */
   public boolean setDiagram (ViewDiagram value)
   {
      if (this.diagram != value)
      {
         if (this.diagram != null)
         {
            ViewDiagram oldValue = this.diagram;
            this.diagram = null;
            oldValue.removeFromViewDefinitions (this);
            oldValue.checkAndRemoveFromItems (this.items, this);
         }
         removeAllFromItems();
         this.diagram = value;
         if (value != null)
         {
            this.diagram.addToViewDefinitions (this);
            update();
         }
         return true;
      }

      return false;
   }


   /**
    * Get the diagram attribute of the ViewDefinition object
    *
    * @return   The diagram value
    */
   public ViewDiagram getDiagram()
   {
      return this.diagram;
   }


   /**
    * <pre>
    *                 N                      0..1
    * ViewDefinition ----------------------------- Filter
    *                 viewDefinitions      filter
    * </pre>
    */
   private Filter filter;


   /**
    * Sets the filter attribute of the ViewDefinition object
    *
    * @param value  The new filter value
    * @return       No description provided
    */
   public boolean setFilter (Filter value)
   {
      if (this.filter != value)
      {
         if (this.filter != null)
         {
            Filter oldValue = this.filter;
            this.filter = null;
            oldValue.removeFromViewDefinitions (this);
         }
         this.filter = value;
         if (value != null)
         {
            this.filter.addToViewDefinitions (this);
            if (isUpdate())
            {
               update();
            }
         }

         return true;
      }

      return false;
   }


   /**
    * Get the filter attribute of the ViewDefinition object
    *
    * @return   The filter value
    */
   public Filter getFilter()
   {
      return this.filter;
   }


   /**
    * @return   the DiagramItems for the current context
    */
   private Set getContextItems()
   {
      if (getFilter() == null)
      {
         return new FHashSet();
      }

      ViewDiagram diagram = getDiagram();

      if (diagram == null || diagram.getDiagram() == null)
      {
         return new FHashSet();
      }

      Set items = getFilter().getNContext (diagram.getOriginalDiagram(), iteratorOfStartItems(), getContext());
      if (items == null)
      {
         items = new FHashSet();
      }

      return items;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient boolean evaluating = false;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected Set evaluate()
   {
      boolean alreadyEvaluating = this.evaluating;

      if (!update)
      {
         log.error ("Call to evaluate() with update=false in ViewDefinition " + name + "[" + Integer.toHexString (hashCode()) + "]");
      }
      if (alreadyEvaluating)
      {
         log.error ("recurring call to evaluate() in ViewDefinition " + name + "[" + Integer.toHexString (hashCode()) + "]");
      }

      this.evaluating = true;
      update();

      if (!alreadyEvaluating)
      {
         this.evaluating = false;
      }

      return this.items;
   }


   /**
    * update the ViewDefinition.<p>
    *
    * Recalculates the context if the source diagram changed
    */
   public void update()
   {
      Set newItems = getContextItems();

      if (newItems == null)
      {
         removeAllFromItems();
      }
      else
      {
         Iterator itemIter = iteratorOfItems();
         while (itemIter.hasNext())
         {
            UMLDiagramItem item = (UMLDiagramItem) itemIter.next();
            if (newItems.contains (item))
            {
               newItems.remove (item);
            }
            else
            {
               removeFromItems (item);
            }
         }

         itemIter = newItems.iterator();
         while (itemIter.hasNext())
         {
            UMLDiagramItem item = (UMLDiagramItem) itemIter.next();
            addToItems (item);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      removeAllFromItems();
      setDiagram (null);
      setFilter (null);
      removeAllFromStartItems();
      super.removeYou();
   }
}

/*
 * $Log: ViewDefinition.java,v $
 * Revision 1.17  2004/10/20 17:50:30  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
